/*******************************************************************************
 * Copyright (c) 2017 Zeligsoft (2009) Limited and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Young-Soo Roh - Initial API and implementation
 *******************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.ChoicePointTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.ICodeSnippetTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.OperationTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.StateEntryTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.StateExitTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.TransitionEffectTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.TransitionGuardTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.TriggerGuardTab;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.Page;
import org.eclipse.ui.part.PageBook;
import org.eclipse.uml2.uml.Trigger;

/**
 * <p>
 * Code snippet page containing available tabs {@link ICodeSnippetTab}. This
 * page controls life cycle of visible tabs.
 * </p>
 * 
 * @author ysroh
 *
 */
public class CodeSnippetPage extends Page implements ICodeSnippetPage {

	/**
	 * Main page composite.
	 */
	private Composite pageControl;

	/**
	 * Selected EObject.
	 */
	private EObject currentSelectedEObject;

	/**
	 * Default empty tab item.
	 */
	private Control defaultPage;

	/**
	 * Available tabs.
	 */
	private List<ICodeSnippetTab> tabs = new ArrayList<>();

	/**
	 * Tab folder composite by element type.
	 */
	private Map<IElementType, Control> elementTypeToPage = new HashMap<IElementType, Control>();

	/**
	 * Listeners.
	 */
	private List<ISelectionChangedListener> listeners = new ArrayList<ISelectionChangedListener>();

	/**
	 * Page book element type.
	 */
	private PageBook pageBook;

	/**
	 * 
	 * Constructor.
	 *
	 */
	public CodeSnippetPage() {

	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			if (selection.isEmpty()) {
				currentSelectedEObject = null;
			} else {
				EObject eo = EMFHelper.getEObject(((IStructuredSelection) selection).getFirstElement());

				if (eo != null && !eo.equals(currentSelectedEObject)) {
					currentSelectedEObject = eo;
				}
			}

			CTabFolder tabFolder = null;

			for (ICodeSnippetTab tab : tabs) {
				if (tab.shouldShowThisTab(currentSelectedEObject)) {
					tabFolder = getPageFolder(currentSelectedEObject);
					if (!tab.controlCreated()) {
						// create tab control if not already created.
						tab.createControl(tabFolder);
					}
					// notify tab of new selection
					tab.setInput(currentSelectedEObject);
				}
			}

			if (tabFolder != null) {
				// selection is valid
				if (tabFolder.getSelection() == null) {
					// select first tab if not selection has been made
					tabFolder.setSelection(0);
				}

				pageBook.showPage(tabFolder);
			} else {
				// show default page
				pageBook.showPage(defaultPage);
			}
		}
	}

	/**
	 * Create tab folder for give element type.
	 * 
	 * @param context
	 *            element
	 * @return tab folder composite
	 */
	private CTabFolder getPageFolder(EObject context) {
		IElementType type = ElementTypeRegistry.getInstance().getElementType(context,
				org.eclipse.papyrus.infra.services.edit.utils.ElementTypeUtils.getEditContext());
		Control folder = elementTypeToPage.get(type);
		if (folder != null) {
			return (CTabFolder) folder;
		}

		// create tab folder
		CTabFolder tabFolder = new CTabFolder(pageBook, SWT.BOTTOM);
		tabFolder.setLayoutData(new GridData(GridData.FILL_BOTH));
		elementTypeToPage.put(type, tabFolder);
		return tabFolder;
	}

	@Override
	public void createControl(Composite parent) {

		// ToDo: we can provide extension mechanism if needed
		// but for now it is fairly limited tabs we need.
		tabs.add(new TransitionEffectTab());
		tabs.add(new TransitionGuardTab());
		tabs.add(new StateEntryTab());
		tabs.add(new StateExitTab());
		tabs.add(new TriggerGuardTab());
		tabs.add(new OperationTab());
		tabs.add(new ChoicePointTab());

		// compose page control
		pageControl = new Composite(parent, SWT.NONE);
		pageControl.setLayout(new FillLayout());

		pageBook = new PageBook(pageControl, SWT.NONE);

		createDefaultPage();

		getSite().setSelectionProvider(this);
	}

	/**
	 * Create default empty tab.
	 */
	private void createDefaultPage() {
		Composite page = new Composite(pageBook, SWT.NONE);
		page.setLayoutData(new GridData(GridData.FILL_BOTH));
		page.setLayout(new GridLayout());
		CLabel label = new CLabel(page, SWT.NONE);
		label.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, false, false));
		label.setText("Code snippet view is not available for the current selection.");
		defaultPage = page;
	}

	@Override
	public Control getControl() {
		return pageControl;
	}

	@Override
	public void setFocus() {
		pageControl.setFocus();
	}

	@Override
	public void dispose() {
		getSite().setSelectionProvider(null);
		tabs.stream().forEach(t -> t.dispose());
		super.dispose();
	}

	@Override
	public ISelection getSelection() {
		StructuredSelection selection = new StructuredSelection();
		if (currentSelectedEObject != null) {
			if (currentSelectedEObject instanceof Trigger) {
				// show its transition in the properties view
				selection = new StructuredSelection(currentSelectedEObject.eContainer());
			} else {
				selection = new StructuredSelection(currentSelectedEObject);
			}
		}
		return selection;
	}

	@Override
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		listeners.add(listener);
	}

	@Override
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		listeners.remove(listener);
	}

	@Override
	public void setSelection(ISelection selection) {
		// not required for code snippet view.
	}
}
