/*****************************************************************************
 * Copyright (c) 2016 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.utils;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Package;

/**
 * Utility class for Packages in UML-RT models
 */
public class PackageUtils {

	private PackageUtils() {
		// no constructor.
	}

	/**
	 * Returns all protocols contained in a given package. It returns all {@link Collaboration}s contained in protocol containers directly nested in the specified Package
	 * 
	 * @param containerPackage
	 *            the package from which all protocols have to be found.
	 * @return the set of protocols contained in protocol containers directly contained in the specified Package or an empty set.
	 */
	public static Set<Collaboration> getProtocols(Package containerPackage) {
		Set<Collaboration> result = new HashSet<Collaboration>();
		for (Package protocolContainer : getProtocolContainers(containerPackage)) {
			Collaboration protocol = ProtocolContainerUtils.getProtocol(protocolContainer);
			if (protocol != null) {
				result.add(protocol);
			}
		}
		return result;
	}

	/**
	 * Returns all protocol containers directly contained in the specified package.
	 * 
	 * @param containerPackage
	 *            the package from which all protocol containers have to be found.
	 * @return the set of protocol containers directly contained in the specified Package or an empty set.
	 */
	public static Set<Package> getProtocolContainers(Package containerPackage) {
		if (containerPackage == null) {
			return Collections.emptySet();
		}
		Set<Package> result = new HashSet<Package>();
		for (Package nestedPackage : containerPackage.getNestedPackages()) {
			if (ProtocolContainerUtils.isProtocolContainer(nestedPackage)) {
				result.add(nestedPackage);
			}
		}
		return result;

	}

	/**
	 * Returns the protocol by name, in the given Package. If none is found, <code>null</code> will be returned
	 * 
	 * @param containerPackage
	 *            the package in which protocol should be found
	 * @param name
	 *            name of the protocol to find
	 * @return the protocol with the specified name or <code>null</code> if none was found
	 */
	public static Collaboration getProtocol(Package containerPackage, String name) {
		Set<Collaboration> protocols = getProtocols(containerPackage);
		for (Collaboration protocol : protocols) {
			String protocolName = protocol.getName();
			if (protocolName != null && protocolName.equals(name)) {
				return protocol;
			}
		}
		return null;
	}
}
