/*******************************************************************************
 * Copyright (c) 2017 Zeligsoft (2009) Limited  and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/

package org.eclipse.papyrusrt.codegen.cpp.validation

import org.eclipse.core.runtime.IStatus
import org.eclipse.core.runtime.MultiStatus
import org.eclipse.core.runtime.Status
import org.eclipse.emf.ecore.EObject
import org.eclipse.papyrusrt.codegen.CodeGenPlugin
import org.eclipse.papyrusrt.xtumlrt.common.Capsule
import org.eclipse.papyrusrt.xtumlrt.common.CommonElement
import org.eclipse.papyrusrt.xtumlrt.common.Port
import org.eclipse.papyrusrt.xtumlrt.statemach.CompositeState
import org.eclipse.papyrusrt.xtumlrt.statemach.EntryPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.ExitPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.JunctionPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.Transition
import org.eclipse.papyrusrt.xtumlrt.trans.TransformValidator
import org.eclipse.papyrusrt.xtumlrt.trans.from.uml.UML2xtumlrtTranslator
import org.eclipse.papyrusrt.xtumlrt.util.DetailedException
import org.eclipse.uml2.uml.Pseudostate
import static extension org.eclipse.papyrusrt.xtumlrt.util.XTUMLRTUtil.*
import org.eclipse.papyrusrt.xtumlrt.common.CapsulePart
import org.eclipse.papyrusrt.xtumlrt.common.Connector

/**
 * Post UML2xtumlrt SM validation
 * @author ysroh
 * 
 */
class PostUML2xtumlrtValidator implements TransformValidator<EObject> {

	private UML2xtumlrtTranslator translator

	new(UML2xtumlrtTranslator translator) {
		this.translator = translator
	}

	override MultiStatus validate(EObject element) {
		val status = new MultiStatus(CodeGenPlugin.ID, IStatus.INFO, "UML-RT Code Generator Invoked", null)
		element.eAllContents.forEach[validateGeneratedElement(status)]
		status
	}

	protected def void validateGeneratedElement(EObject o, MultiStatus result) {
		if(o instanceof CommonElement) {
			val source = translator.getSource(o)
			if(source !== null) {
				o.validateElement(source, result)
			}
		}
	}

	protected dispatch def void validateElement(EObject o, EObject source, MultiStatus result) {
	}

	protected dispatch def void validateElement(JunctionPoint junction, EObject source, MultiStatus result) {
		if (junction.outgoingTransitions.size != 1) {
			val exception = new DetailedException(
				"Junction point " + (translator.getSource(junction) as Pseudostate).qualifiedName +
					" must have exactly one outgoing transition")
			val status = new Status(IStatus.ERROR, CodeGenPlugin.ID, exception.message, exception)
			result.add(status)
		}
	}

	/**
	 * Validate if the transition belongs to correct state
	 * let n1 be the source node of t and n2 its target node.
	 * 1) sibling transition: the owner of n1 is S, the owner of n2 is also S and the owner of t must be S as well.
	 * 2) entering transition: the owner of n2 is S; then the owner of t must be S as well (and n1 is S or one of its entry points)
	 * 3) exiting transition: the owner of n1 is S; then the owner of t must be S as well (and n2 is S or one of its exit points)
	 * 4) through transition: then n1 = n2 is a composite state S and the owner of t is S.
	 */
	protected dispatch def void validateElement(Transition t, EObject source, MultiStatus result) {
		if (translator.getSource(t) === null) {
			// This might be purely generate so ignore it
			return
		}
		val transitionOwner = t.eContainer
		val qn = (translator.getSource(t) as org.eclipse.uml2.uml.Transition).qualifiedName

		var EObject sourceNode = t.sourceVertex
		if (sourceNode instanceof EntryPoint || sourceNode instanceof ExitPoint) {
			sourceNode = sourceNode.eContainer
		}
		var EObject targetNode = t.targetVertex
		if (targetNode instanceof EntryPoint || targetNode instanceof ExitPoint) {
			targetNode = sourceNode.eContainer
		}

		var Exception exception = null

		// check to see if this is through transition
		if (sourceNode instanceof CompositeState && sourceNode == targetNode) {
			// check through transition
			if(t.sourceVertex instanceof EntryPoint || t.targetVertex instanceof ExitPoint) {
				if(targetNode !== transitionOwner) {
					exception = new DetailedException("Through transition " + qn +
						" must have same owner state as source and target vertex")
				}
			} else {
				// We cannot differentiate through transition with loop transition
				// so just check loosely
				if(targetNode !== transitionOwner && targetNode.eContainer !== transitionOwner) {
					exception = new DetailedException("Transition " + qn + " does not belong to correct state")
				}
			}

		} // check to see if this is entering transition
		else if(t.sourceVertex instanceof EntryPoint || sourceNode === targetNode.eContainer) {
			if(sourceNode !== transitionOwner) {
				exception = new DetailedException(
					"Entering transition " + qn + " must have same owner state as target vertex")
			}
		} // check to see if this is exiting transition
		else if(t.targetVertex instanceof ExitPoint || targetNode === sourceNode.eContainer) {
			if(targetNode !== transitionOwner) {
				exception = new DetailedException(
					"Exiting transition " + qn + " must have same owner state as source vertex")
			}
		} // check to see if this is sibling transition 
		else if(sourceNode.eContainer == targetNode.eContainer) {
			if(targetNode.eContainer !== transitionOwner) {
				exception = new DetailedException(
					"Sibling transition " + qn +
						" must have same owner state with its vertices if the source and target has same state"
				)
			}
		}
		if(exception !== null) {
			val status = new Status(IStatus.ERROR, CodeGenPlugin.ID, exception.message, exception)
			result.add(status)
		}
	}

	protected dispatch def void validateElement(Port port, EObject source, MultiStatus result) {
		val capsule = port.owner as Capsule
		val capsuleParent = capsule.redefines
		if (capsuleParent instanceof Capsule) {
			val allParentPorts = capsuleParent.getAllRTPorts
			if (allParentPorts.exists[ name == port.name && port.redefines !== it ]) {
				val qn = (translator.getSource(port) as org.eclipse.uml2.uml.Port).qualifiedName
				val exception = new DetailedException(
					"Port " + qn + " has the same name as a port in the parent capsule but it does not redefine it"
				)
				val status = new Status(IStatus.ERROR, CodeGenPlugin.ID, exception.message, exception)
				result.add(status)
			}
		}
	}

    protected dispatch def void validateElement(CapsulePart part, EObject source, MultiStatus result) {
        val capsule = part.owner as Capsule
        val capsuleParent = capsule.redefines
        if (capsuleParent instanceof Capsule) {
            val allParentParts = capsuleParent.getAllCapsuleParts
            if (allParentParts.exists[ name == part.name && part.redefines !== it ]) {
                val qn = (translator.getSource(part) as org.eclipse.uml2.uml.Property).qualifiedName
                val exception = new DetailedException(
                    "Part " + qn + " has the same name as a part in the parent capsule but it does not redefine it"
                )
                val status = new Status(IStatus.ERROR, CodeGenPlugin.ID, exception.message, exception)
                result.add(status)
            }
        }
    }

    protected dispatch def void validateElement(Connector conn, EObject source, MultiStatus result) {
        val capsule = conn.owner as Capsule
        val capsuleParent = capsule.redefines
        if (capsuleParent instanceof Capsule) {
            val allParentParts = capsuleParent.getAllConnectors
            if (allParentParts.exists[ name == conn.name && conn.redefines !== it ]) {
                val qn = (translator.getSource(conn) as org.eclipse.uml2.uml.Property).qualifiedName
                val exception = new DetailedException(
                    "Connector " + qn + " has the same name as a connector in the parent capsule but it does not redefine it"
                )
                val status = new Status(IStatus.ERROR, CodeGenPlugin.ID, exception.message, exception)
                result.add(status)
            }
        }
    }


}
