package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.IObserving;
import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.ValueChangeEvent;
import org.eclipse.core.databinding.observable.value.ValueDiff;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue;
import org.eclipse.papyrusrt.umlrt.tooling.ui.util.Constants;
import org.eclipse.papyrusrt.umlrt.tooling.ui.util.RTPortHelper;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Stereotype;

public class PortRTKindObservableValue extends PapyrusObservableValue implements IObserving, IValueChangeListener {

	private Port port;
	private Stereotype stereo;
	private EObject stereotypeApplication;

	private boolean service;
	private boolean wired;
	private boolean behavior;
	private boolean publish;
	private boolean conjugated;
	private boolean notification;

	/**
	 * 
	 * Constructor.
	 *
	 * @param sourceElement
	 */
	public PortRTKindObservableValue(final Element sourceElement, final TransactionalEditingDomain domain) {
		super(sourceElement, sourceElement.eContainingFeature(), domain);
		if (sourceElement instanceof Port) {
			port = (Port) sourceElement;
			stereo = port.getAppliedStereotype(Constants.UML_RT_STEREOTYPE_QN);
			stereotypeApplication = port.getStereotypeApplication(stereo);
			stereotypeApplication.eAdapters().add(getListener());
			port.eAdapters().add(getListener());
			setUMLPropertiesValue();
			setUMLRTPropertiesValue();
		}
	}

	private AdapterImpl getListener() {
		return new AdapterImpl() {

			@Override
			public void notifyChanged(Notification notification) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();
				EList<EStructuralFeature> realTimeFeatureList = stereotypeApplication.eClass().getEStructuralFeatures();
				EList<EStructuralFeature> umlFeatureList = port.eClass().getEStructuralFeatures();

				if ((notifier == stereotypeApplication) && (type == Notification.SET)) {
					if (realTimeFeatureList.contains(feature)) {
						fireDiff(notification);
					}
				} else if ((notifier == port) && (type == Notification.SET)) {
					if (umlFeatureList.contains(feature)) {
						fireDiff(notification);
					}
				}
			}

			private void fireDiff(Notification notification) {
				final ValueDiff diff = Diffs.createValueDiff(notification.getOldValue(), notification.getNewValue());
				getRealm().exec(new Runnable() {
					@Override
					public void run() {
						fireValueChange(diff);
					}
				});
			}

		};
	}


	/**
	 * 
	 */
	private void setUMLRTPropertiesValue() {
		wired = RTPortHelper.getInstance().isWired(port);
		notification = RTPortHelper.getInstance().isNotification(port);
		publish = RTPortHelper.getInstance().isPublish(port);
	}

	/**
	 * 
	 */
	private void setUMLPropertiesValue() {
		service = RTPortHelper.getInstance().isService(port);
		behavior = RTPortHelper.getInstance().isBehavior(port);
		conjugated = RTPortHelper.getInstance().isConjugated(port);
	}

	@Override
	public Object getValueType() {
		return PortRTKindEnum.class;
	}

	@Override
	public Object getObserved() {
		return port;
	}



	@Override
	protected Object doGetValue() {
		PortRTKindEnum kind = PortRTKindEnum.RELAY;

		setUMLPropertiesValue();
		setUMLRTPropertiesValue();

		if (service && wired && behavior && !publish) {
			kind = PortRTKindEnum.EXTERNAL;
		} else if (service && behavior && publish && !wired) {
			kind = PortRTKindEnum.SPP;
		} else if (wired && behavior && !service && !publish) {
			kind = PortRTKindEnum.INTERNAL;
		} else if (service && wired && !behavior && !publish) {
			kind = PortRTKindEnum.RELAY;
		} else if (behavior && !wired && !publish && !service) {
			kind = PortRTKindEnum.SAP;
		}

		return kind;
	}

	@Override
	protected void doSetValue(Object value) {
		super.doSetValue(value);

	}



	@Override
	public Command getCommand(Object value) {
		Command command = null;
		if (value instanceof PortRTKindEnum) {
			// For Each Kind of Port, set the property accordingly
			switch ((PortRTKindEnum) value) {
			case EXTERNAL:
				command = getCommandForRTPort(true, false, true, true);
				break;
			case INTERNAL:
				command = getCommandForRTPort(true, false, false, true);
				break;
			case RELAY:
				command = getCommandForRTPort(false, false, true, true);
				break;
			case SAP:
				command = getCommandForRTPort(true, false, false, false);
				break;
			case SPP:
				command = getCommandForRTPort(true, true, true, false);
				break;
			default:
				// Relay Port by Default
				command = getCommandForRTPort(false, false, true, true);
				break;
			}
		}
		return command;
	}

	private Command getCommandForRTPort(boolean behavior, boolean publish, boolean service, boolean wired) {
		CompoundCommand command = new CompoundCommand();
		command.append(getSetBehaviorCommand(behavior));
		command.append(getSetPublishCommand(publish));
		command.append(getSetServiceCommand(service));
		command.append(getSetWiredCommand(wired));
		return command;
	}

	private Command getSetPublishCommand(final boolean publish) {
		RecordingCommand publishCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				port.setValue(stereo, Constants.PUBLISH, publish);
			}

			@Override
			public boolean canExecute() {
				return true;
			}

		};

		return publishCommand;
	}



	private Command getSetWiredCommand(final boolean wired) {
		RecordingCommand wiredCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				port.setValue(stereo, Constants.WIRED, wired);
			}

			@Override
			public boolean canExecute() {
				return true;
			}
		};

		return wiredCommand;
	}

	private Command getSetServiceCommand(final boolean service) {
		RecordingCommand serviceCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				port.setIsService(service);
			}

			@Override
			public boolean canExecute() {
				return true;
			}
		};

		return serviceCommand;
	}

	private Command getSetBehaviorCommand(final boolean behavior) {
		RecordingCommand behaviorCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				port.setIsBehavior(behavior);
			}

			@Override
			public boolean canExecute() {
				return true;
			}
		};

		return behaviorCommand;
	}

	@Override
	public void handleValueChange(ValueChangeEvent event) {
		fireChange();

	}





}
