/*******************************************************************************
* Copyright (c) 2015 Zeligsoft (2009) Limited  and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*******************************************************************************/

package org.eclipse.papyrusrt.codegen.xtumlrt.trans

import static extension org.eclipse.papyrusrt.codegen.utils.UMLRealTimeStateMachProfileUtil.*
import static extension org.eclipse.papyrusrt.codegen.utils.GeneralUtil.*
import static extension org.eclipse.papyrusrt.codegen.utils.QualifiedNames.*
import org.eclipse.papyrusrt.codegen.utils.UML2CppUtil
import org.eclipse.papyrusrt.xtumlrt.common.CommonElement
import org.eclipse.papyrusrt.xtumlrt.common.CommonFactory
import org.eclipse.papyrusrt.xtumlrt.common.CompositeState
import org.eclipse.papyrusrt.xtumlrt.common.ChoicePoint
import org.eclipse.papyrusrt.xtumlrt.common.DeepHistory
import org.eclipse.papyrusrt.xtumlrt.common.EntryPoint
import org.eclipse.papyrusrt.xtumlrt.common.ExitPoint
import org.eclipse.papyrusrt.xtumlrt.common.InitialPoint
import org.eclipse.papyrusrt.xtumlrt.common.JunctionPoint
import org.eclipse.papyrusrt.xtumlrt.common.Pseudostate
import org.eclipse.papyrusrt.xtumlrt.common.Signal
import org.eclipse.papyrusrt.xtumlrt.common.SimpleState
import org.eclipse.papyrusrt.xtumlrt.common.State
import org.eclipse.papyrusrt.xtumlrt.common.StateMachine
import org.eclipse.papyrusrt.xtumlrt.common.Transition
import org.eclipse.papyrusrt.xtumlrt.common.Trigger
import org.eclipse.papyrusrt.xtumlrt.umlrt.RTTrigger
import org.eclipse.papyrusrt.xtumlrt.umlrt.RTPort
import org.eclipse.papyrusrt.xtumlrt.umlrt.UmlrtFactory
import org.eclipse.papyrusrt.codegen.statemachines.flat.model.smflatmodel.SmflatmodelFactory
import org.eclipse.emf.ecore.EObject
import org.eclipse.papyrusrt.codegen.statemachines.flat.model.smflatmodel.EntryAction
import org.eclipse.papyrusrt.codegen.statemachines.flat.model.smflatmodel.ExitAction
import org.eclipse.papyrusrt.xtumlrt.common.ActionCode
import org.eclipse.papyrusrt.xtumlrt.common.Vertex
import org.eclipse.papyrusrt.xtumlrt.common.Guard

class UML2xtumlrtSMTranslator extends UML2xtumlrtTranslator
{

    // UML2xtumlrtTranslator provides methods to translate ports and signals, used in triggers
    UML2xtumlrtModelTranslator uml2xtumlrtTranslator

    new (UML2xtumlrtModelTranslator uml2xtumlrtTranslator)
    {
        this.uml2xtumlrtTranslator = uml2xtumlrtTranslator
    }

    dispatch def CommonElement translate( org.eclipse.uml2.uml.Element element )
    {
    }

    /**
     * This is the main method. It creates a dummy composite state to be used
     * as the State Machine's top composite state.
     *
     * @param originalStateMachine  - A {@link org.eclipse.uml2.uml.StateMachine}
     * @return A {@link StateMachine}
     */
    dispatch def StateMachine
    create CommonFactory.eINSTANCE.createStateMachine
    translate( org.eclipse.uml2.uml.StateMachine originalStateMachine )
    {
        name = originalStateMachine.name
        top = translateElement( originalStateMachine.ownedRegion ) as CompositeState
        if (top !== null) top.name = "top"
    }

    /**
     * Translates a region into a composite state. This is because in UML-RT, (composite) states
     * have exactly one region, so there is a one-to-one correspondance between them.
     */
    dispatch def CompositeState
    create
        if (region == null) null
        else CommonFactory.eINSTANCE.createCompositeState
    translate( org.eclipse.uml2.uml.Region region )
    {
        if (region !== null)
        {
            initial = translateOptionalElement( region.initialPoint ) as InitialPoint
            deepHistory = translateOptionalElement( region.deepHistoryPoint ) as DeepHistory
            for (element : region.choicePoints)
            {
                choicePoints.addIfNotNull( translateElement(element) as ChoicePoint )
            }
            for (element : region.junctionPoints)
            {
                junctionPoints.addIfNotNull( translateElement(element) as JunctionPoint )
            }
            for (element : region.substates)
            {
                substates.addIfNotNull( translateElement(element) as State )
            }
            for (element : region.transitions)
            {
                transitions.addIfNotNull( translateElement(element) as Transition )
            }
        }
    }

    /**
     * Translates a state from the source model into a state in the internal
     * representation.
     *
     * Essentially it just decides which translation to apply depending on
     * whether the state is simple or composite (in UML-RT there are no
     * "sub-state-machine" states.)
     *
     * @param originalState - the UML2 state element in the original model
     */
    dispatch def State
    create
        if (originalState.isSimpleState)
            translateSimpleState( originalState )
        else
            translateCompositeState( originalState )
    translate( org.eclipse.uml2.uml.State originalState )
    {
        if (it !== null)
        {
            name = originalState.name
            entryAction = translateFeature( originalState, "entry", org.eclipse.uml2.uml.Behavior, EntryAction, true ) as EntryAction
            exitAction = translateFeature( originalState, "exit", org.eclipse.uml2.uml.Behavior, ExitAction, true ) as ExitAction
            for (entryPoint : originalState.entryPoints)
            {
                entryPoints.addIfNotNull( translateElement(entryPoint) as EntryPoint )
            }
            for (exitPoint : originalState.exitPoints)
            {
                exitPoints.addIfNotNull( translateElement(exitPoint) as ExitPoint )
            }
        }
    }

    def isSimpleState( org.eclipse.uml2.uml.State originalState )
    {
        originalState.simple
        || originalState.composite
            && (originalState.regions === null
                || originalState.regions.empty
                || (originalState.regions.size == 1
                    && (originalState.regions.get(0) as org.eclipse.uml2.uml.Region).isEmpty))
    }

    def boolean isEmpty( org.eclipse.uml2.uml.Region region )
    {
        if (region.extendedRegion === null)
            region.ownedElements === null || region.ownedElements.empty
        else
            (region.ownedElements === null && region.ownedElements.empty) && region.extendedRegion.isEmpty
    }

    /**
     * Translates a simple state.
     */
    protected def SimpleState translateSimpleState( org.eclipse.uml2.uml.State originalState )
    {
        CommonFactory.eINSTANCE.createSimpleState
    }

    /**
     * Translates a composite state by translating its sub-elements (vertices
     * and transitions).
     *
     * <p>When traversing the sub-states, the translation is applied recursively.
     */
    protected def CompositeState
    translateCompositeState( org.eclipse.uml2.uml.State originalState )
    {
        translate( originalState.ownedRegion ) as CompositeState
    }

    dispatch def Pseudostate
    create
        if (originalPseudostate === null) null
        else
            switch (originalPseudostate.kind)
            {
                case org.eclipse.uml2.uml.PseudostateKind.INITIAL_LITERAL:
                    CommonFactory.eINSTANCE.createInitialPoint
                case org.eclipse.uml2.uml.PseudostateKind.DEEP_HISTORY_LITERAL:
                    CommonFactory.eINSTANCE.createDeepHistory
                case org.eclipse.uml2.uml.PseudostateKind.ENTRY_POINT_LITERAL:
                    CommonFactory.eINSTANCE.createEntryPoint
                case org.eclipse.uml2.uml.PseudostateKind.EXIT_POINT_LITERAL:
                    CommonFactory.eINSTANCE.createExitPoint
                case org.eclipse.uml2.uml.PseudostateKind.CHOICE_LITERAL:
                    CommonFactory.eINSTANCE.createChoicePoint
                case org.eclipse.uml2.uml.PseudostateKind.JUNCTION_LITERAL:
                    CommonFactory.eINSTANCE.createJunctionPoint
                default:
                    throw new RuntimeException( "Pseudostate '" + originalPseudostate.qualifiedName + "' has unsupported kind: '" + originalPseudostate.kind.toString + "'")
            }
    translate( org.eclipse.uml2.uml.Pseudostate originalPseudostate )
    {
        if (it !== null) name = originalPseudostate.name
    }

    dispatch def ActionCode
    create
        if (behaviour === null) null
        else if (behaviour.owner !== null)
        {
            if (behaviour.owner instanceof org.eclipse.uml2.uml.State)
            {
                val state = behaviour.owner as org.eclipse.uml2.uml.State
                if (behaviour === state.entry)
                    SmflatmodelFactory.eINSTANCE.createEntryAction
                else if (behaviour === state.exit)
                    SmflatmodelFactory.eINSTANCE.createExitAction
            }
            else if (behaviour.owner instanceof org.eclipse.uml2.uml.Transition)
            {
                val transition = behaviour.owner as org.eclipse.uml2.uml.Transition
                if (behaviour == transition.effect)
                    SmflatmodelFactory.eINSTANCE.createTransitionAction
            }
        }
        else
            throw new RuntimeException( "Behavior '" + behaviour.qualifiedName + "' has no owner.")
    translate( org.eclipse.uml2.uml.Behavior behaviour )
    {
        if (it !== null && behaviour !== null)
        {
            name = behaviour.name
            source = UML2CppUtil.getCppCode( behaviour )
        }
    }

    dispatch def Transition
    create CommonFactory.eINSTANCE.createTransition
    translate( org.eclipse.uml2.uml.Transition originalTransition )
    {
        name = originalTransition.name
        sourceVertex = translateFeature( originalTransition, "source", org.eclipse.uml2.uml.Vertex, Vertex ) as Vertex
        targetVertex = translateFeature( originalTransition, "target", org.eclipse.uml2.uml.Vertex, Vertex ) as Vertex
        for (trigger : originalTransition.triggers)
        {
            triggers.addIfNotNull( translateElement(trigger) as Trigger )
        }
        guard        = translateFeature( originalTransition, "guard", org.eclipse.uml2.uml.Constraint, Guard, true ) as Guard
        actionChain  = CommonFactory.eINSTANCE.createActionChain
        actionChain.actions.addIfNotNull( translateFeature( originalTransition, "effect", org.eclipse.uml2.uml.Behavior, ActionCode, true ) as ActionCode )
    }

    /**
     * @param trigger  - A {@link org.eclipse.uml2.uml.Trigger}
     * @return A {@link Trigger}
     */
    dispatch def Trigger
    create
        // if (trigger.isRTTrigger) // This test should be here, but the current profile doesn't have the RTTrigger stereotype
            translateRTTrigger( trigger )
    translate( org.eclipse.uml2.uml.Trigger trigger )
    {
        name = trigger.name
    }

    protected def RTTrigger translateRTTrigger( org.eclipse.uml2.uml.Trigger trigger )
    {
        val it = UmlrtFactory.eINSTANCE.createRTTrigger
        if (trigger.ports !== null)
        {
            for (port : trigger.ports)
            {
                ports.addIfNotNull( uml2xtumlrtTranslator.translateElement(port) as RTPort)
            }
        }
        val operation = (trigger.event as org.eclipse.uml2.uml.CallEvent).operation
        signal = uml2xtumlrtTranslator.translateElement( operation ) as Signal
        it
    }

    dispatch def create
        if (originalGuard === null) null
        else CommonFactory.eINSTANCE.createGuard
    translate( org.eclipse.uml2.uml.Constraint originalGuard )
    {
        if (it !== null)
        {
            name = originalGuard?.name
            body = CommonFactory.eINSTANCE.createActionCode
            body.source = UML2CppUtil.getCppCode( originalGuard )
        }
    }

    override translateEnum(Enum<?> kind)
    {
        throw new UnsupportedOperationException("TODO: auto-generated method stub")
    }

    override resetTranslateCache(EObject element)
    {
        val key = newArrayList( element )
        _createCache_translate.remove( key )
    }

}