/*****************************************************************************
 * Copyright (c) 2014 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.types.advice;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gmf.runtime.common.core.command.CompositeCommand;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.commands.SetValueCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateRelationshipRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.papyrus.commands.wrappers.EMFtoGMFCommandWrapper;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.core.utils.ProtocolUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTPort;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.VisibilityKind;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Edit Helper Advice for {@link RTPort}
 */
public class RTPortEditHelperAdvice extends AbstractEditHelperAdvice {

	@Override
	public boolean approveRequest(IEditCommandRequest request) {
		// check that the type of the RTPort is a protocol, and nothing else
		if (request instanceof SetRequest) {
			SetRequest setRequest = (SetRequest) request;
			EStructuralFeature feature = setRequest.getFeature();
			if (UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature)) {
				// new value should be a protocol or null
				Object newValue = ((SetRequest) request).getValue();
				if (newValue instanceof EObject) {
					if (ProtocolUtils.isProtocol((EObject) newValue)) {
						return true;
					} else {
						return false;
					}
				} else {
					return false;
				}
			}
			return super.approveRequest(setRequest);
		} else if (request instanceof CreateRelationshipRequest) {
			IElementType type = ((CreateRelationshipRequest) request).getElementType();
			if (type != null && ElementTypeUtils.isTypeCompatible(type, ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.RT_CONNECTOR_ID))) {
				return checkSourceAndTarget(((CreateRelationshipRequest) request));
			}
			return super.approveRequest(request);
		}
		return super.approveRequest(request);
	}

	protected boolean checkSourceAndTarget(CreateRelationshipRequest createRelationshipRequest) {
		EObject source = createRelationshipRequest.getSource();
		EObject target = createRelationshipRequest.getTarget();

		if (source != null) {
			if (ElementTypeUtils.matches(source, IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID)) {
				return false; // cannot connect a connector to a SAP
			} else if (ElementTypeUtils.matches(source, IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID)) {
				return false;
			}
		}

		if (target != null) {
			if (ElementTypeUtils.matches(target, IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID)) {
				return false; // cannot connect a connector to a SAP
			} else if (ElementTypeUtils.matches(target, IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID)) {
				return false;
			}
		}

		return true;
	}

	@Override
	protected ICommand getBeforeCreateRelationshipCommand(CreateRelationshipRequest request) {
		return super.getBeforeCreateRelationshipCommand(request);
	}

	@Override
	protected ICommand getAfterSetCommand(SetRequest request) {
		CompositeCommand cc = new CompositeCommand("Set RTPort");
		
		
		EStructuralFeature feature = request.getFeature();
		if (UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature)) {
			// if element is unnamed, set a name according to the new Type name
			Object newValue = ((SetRequest) request).getValue();
			EObject elementToEdit = request.getElementToEdit();
			if (newValue instanceof Type && elementToEdit instanceof Port && ((Port) elementToEdit).getName() == null) {
				String name = ((Type) newValue).getName();
				if (name != null && !name.isEmpty()) {
					String newName = Character.toLowerCase(name.charAt(0)) + name.substring(1);
					cc.add( new SetValueCommand(new SetRequest(request.getElementToEdit(), UMLPackage.eINSTANCE.getNamedElement_Name(), newName)));
				}
			}
		}
		
		ICommand setKindCommand = getSetKindCommand(request);
		if(null!=setKindCommand){
			cc.add(setKindCommand);
		}
		
		return cc.isEmpty()?super.getAfterSetCommand(request):cc;
	}
	
	
	protected ICommand getSetKindCommand(SetRequest request) {
		ICommand cmd = null;
		final EObject object = request.getElementToEdit();
		final Object kindParameter = request.getParameter(RTPortUtils.RTPORT_KIND_REQUEST_PARAMETER);
		if (object instanceof Port && (null!=kindParameter) && kindParameter instanceof RTPortKindEnum) {
			final Port rtPort = (Port) object;
			RecordingCommand command = new RecordingCommand(TransactionUtil.getEditingDomain(object)) {
				@Override
				protected void doExecute() {
					
					if(RTPortKindEnum.EXTERNAL.equals(kindParameter)){
						setExternalPort(rtPort);
					}
					if(RTPortKindEnum.INTERNAL.equals(kindParameter)){
						setInternalPort(rtPort);
					}
					if(RTPortKindEnum.RELAY.equals(kindParameter)){
						setRelayPort(rtPort);
					}
					if(RTPortKindEnum.SAP.equals(kindParameter)){
						setSAPPort(rtPort);
					}
					if(RTPortKindEnum.SPP.equals(kindParameter)){
						setSPPPort(rtPort);
					}
					
				}
			};
			cmd = EMFtoGMFCommandWrapper.wrap(command);
		}

		return cmd;
	}

	/**
	 * Sets the RT port as a external Behavior Port.
	 *
	 * @param externalBehaviorPort
	 *            the new RT port as a external Behavior Port
	 */
	protected void setExternalPort(final Port externalBehaviorPort) {
		// isService: true
		externalBehaviorPort.setIsService(true);

		// isBehavior: true
		externalBehaviorPort.setIsBehavior(true);

		// isWired: true
		RTPort stereotype = UMLUtil.getStereotypeApplication(externalBehaviorPort, RTPort.class);
		stereotype.setIsWired(true);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: public
		externalBehaviorPort.setVisibility(VisibilityKind.PUBLIC_LITERAL);
	}
	
	/**
	 * Sets the RT port as internal Behavior Port.
	 *
	 * @param internalBehaviorPort
	 *            the new RT port as internal Behavior Port
	 */
	protected void setInternalPort(final Port internalBehaviorPort) {
		// isService: false
		internalBehaviorPort.setIsService(false);

		// isBehavior: true
		internalBehaviorPort.setIsBehavior(true);

		// isWired: true
		RTPort stereotype = UMLUtil.getStereotypeApplication(internalBehaviorPort, RTPort.class);
		stereotype.setIsWired(true);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: protected
		internalBehaviorPort.setVisibility(VisibilityKind.PROTECTED_LITERAL);
	}
	
	/**
	 * Sets the RT port as a relay port.
	 *
	 * @param relayPort
	 *            the new RT port as a relay port
	 */
	protected void setRelayPort(final Port relayPort) {
		// isService: true
		relayPort.setIsService(true);

		// isBehavior: false
		relayPort.setIsBehavior(false);

		// isWired: true
		RTPort stereotype = UMLUtil.getStereotypeApplication(relayPort, RTPort.class);
		stereotype.setIsWired(true);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: public
		relayPort.setVisibility(VisibilityKind.PUBLIC_LITERAL);
	}
	
	/**
	 * Sets the RT port properties as an service Access Point.
	 *
	 * @param serviceAccessPoint
	 *            the new RT port as an service Access Point
	 */
	protected void setSAPPort(final Port serviceAccessPoint) {
		// isService: false
		serviceAccessPoint.setIsService(false);

		// isBehavior: true
		serviceAccessPoint.setIsBehavior(true);

		// isWired: false
		RTPort stereotype = UMLUtil.getStereotypeApplication(serviceAccessPoint, RTPort.class);
		stereotype.setIsWired(false);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: protected
		serviceAccessPoint.setVisibility(VisibilityKind.PROTECTED_LITERAL);
	}
	
	/**
	 * Sets the RT port as service provision point.
	 *
	 * @param serviceProvisionPoint
	 *            the new RT port as service provision point
	 */
	protected void setSPPPort(final Port serviceProvisionPoint) {
		// isService: true
		serviceProvisionPoint.setIsService(true);

		// isBehavior: true
		serviceProvisionPoint.setIsBehavior(true);

		// isWired: false
		RTPort stereotype = UMLUtil.getStereotypeApplication(serviceProvisionPoint, RTPort.class);
		stereotype.setIsWired(false);

		// isPublish: true
		stereotype.setIsPublish(true);

		// Visibility: public
		serviceProvisionPoint.setVisibility(VisibilityKind.PUBLIC_LITERAL);
	}

}
