/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Mickael ADAM (ALL4TEC) mickael.adam@all4tec.net - Initial API and Implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.drop;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gmf.runtime.diagram.ui.commands.ICommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest;
import org.eclipse.papyrus.infra.gmfdiag.dnd.strategy.TransactionalDropStrategy;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsuleUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.ProtocolUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortKindEnum;
import org.eclipse.swt.graphics.Image;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * Drop strategy to create a RTPort when droping a Protocol on a capsule.
 */
public abstract class AbstractProtocolToRTPortDropStrategy extends TransactionalDropStrategy {


	private static final String EMPTY_STRING = "";// $NON-NLS-0$

	/**
	 * Instantiates a new abstract protocol to rt port drop strategy.
	 */
	public AbstractProtocolToRTPortDropStrategy() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.papyrus.infra.gmfdiag.dnd.strategy.DropStrategy#getImage()
	 */
	@Override
	public Image getImage() {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.papyrus.infra.gmfdiag.dnd.strategy.DropStrategy#getPriority()
	 */
	@Override
	public int getPriority() {
		return 0;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.papyrus.infra.gmfdiag.dnd.strategy.TransactionalDropStrategy#doGetCommand(org.eclipse.gef.Request, org.eclipse.gef.EditPart)
	 */
	@Override
	protected Command doGetCommand(Request request, EditPart targetEditPart) {

		Command command = null;
		if ((request instanceof DropObjectsRequest)) {
			DropObjectsRequest dropReq = getDropObjectsRequest(request);
			if (dropReq != null) {
				List<Classifier> handledDroppedObjects = getDroppedProtocol(dropReq);
				EObject targetElement = getTargetSemanticElement(targetEditPart);
				if (canHandleRequest(handledDroppedObjects, targetElement) && canHandleDropPosition(dropReq.getLocation())) {
					// dropping a protocol on a Capsule => creating a rt port
					Point location = dropReq.getLocation();
					CompoundCommand compoundCommand = new CompoundCommand();
					for (EObject droppedObject : handledDroppedObjects) {

						compoundCommand.add(getCreateAndDropObjectCommand(droppedObject, (Classifier) targetElement, location, targetEditPart));
						location.performTranslate(20, 20);
						command = compoundCommand;
					}
				}
			}
		}

		return command;

	}

	/**
	 * Gets the creates the and drop object command.
	 *
	 * @param droppedObject
	 *            the dropped object
	 * @param targetClassifier
	 *            the target classifier
	 * @param location
	 *            the location
	 * @param targetEditPart
	 *            the target edit part
	 * @return the creates the and drop object command
	 */
	protected Command getCreateAndDropObjectCommand(EObject droppedObject, Classifier targetClassifier, Point location, EditPart targetEditPart) {
		// create RTport part
		CreateRTPortAndDisplayCommand command = new CreateRTPortAndDisplayCommand(targetClassifier, IUMLRTElementTypes.RT_PORT_ID, UMLPackage.eINSTANCE.getNamespace_OwnedMember(), droppedObject, location, targetEditPart);
		// set Name
		command.setRTPortName(getName(droppedObject));
		// Set kind
		command.setRTPortKind(getRTPortKind());
		return new ICommandProxy(command);
	}


	/**
	 * Gets the name of the dropped object.
	 *
	 * @param droppedObject
	 *            the dropped object
	 * @return the name
	 */
	protected String getName(EObject droppedObject) {
		String name = EMPTY_STRING;
		if (droppedObject instanceof NamedElement) {
			String elementName = ((NamedElement) droppedObject).getName();
			if (elementName.length() > 0) {
				name = Character.toLowerCase(elementName.charAt(0)) + elementName.substring(1);
			}
		}
		return name;

	}


	/**
	 * Can handle drop position.
	 *
	 * @param point
	 *            the point
	 * @return true, if successful
	 */
	protected boolean canHandleDropPosition(Point point) {
		return true;
	}

	/**
	 * Gets the dropped protocol.
	 *
	 * @param req
	 *            the req
	 * @return the dropped protocol
	 */
	protected List<Classifier> getDroppedProtocol(Request req) {
		List<EObject> droppedObjects = getSourceEObjects(req);
		List<Classifier> result = new ArrayList<Classifier>();
		if (droppedObjects != null) {
			for (EObject droppedObject : droppedObjects) {
				if (droppedObject instanceof Classifier) {
					if (ProtocolUtils.isProtocol(droppedObject)) {
						result.add((Classifier) droppedObject);
					}
				}
			}
		}
		return result;
	}

	/**
	 * Can handle request.
	 *
	 * @param droppedObjects
	 *            the dropped objects
	 * @param targetElement
	 *            the target element
	 * @return true, if successful
	 */
	protected boolean canHandleRequest(List<Classifier> droppedObjects, EObject targetElement) {
		boolean result = false;
		if (!droppedObjects.isEmpty()) {
			result = (targetElement instanceof Classifier && CapsuleUtils.isCapsule((Classifier) targetElement) && droppedObjects.size() > 0);
		}
		return result;
	}

	/**
	 * Gets the kind of the RTPort. The Kind is a {@link RTPortKindEnum}.
	 *
	 * @return the kind of the RTPort.
	 */
	abstract protected RTPortKindEnum getRTPortKind();

}
