/*****************************************************************************
 * Copyright (c) 2015 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Ansgar Radermacher (CEA LIST) ansgar.radermacher@cea.fr - Initial API and implementation
 *
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import java.util.ArrayList;

import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.papyrus.infra.widgets.providers.IStaticContentProvider;
import org.eclipse.papyrus.uml.properties.modelelement.UMLModelElement;
import org.eclipse.papyrus.views.properties.modelelement.ModelElement;
import org.eclipse.papyrus.views.properties.widgets.ReferenceDialog;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTMessageKind;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTMessageSet;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.uml2.uml.CallEvent;
import org.eclipse.uml2.uml.Event;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Specific event selection dialog for UML/RT. It filters events in function of
 * selected port.
 */
public class EventSelectionDialog extends ReferenceDialog {

	/**
	 * Constructor.
	 *
	 * @param parent
	 * @param style
	 */
	public EventSelectionDialog(Composite parent, int style) {
		super(parent, style);
	}

	protected Port port;

	@Override
	protected void doBinding() {
		editor.setLabelProvider(input.getLabelProvider(propertyPath));
		editor.setDirectCreation(input.getDirectCreation(propertyPath));
		editor.setMandatory(input.isMandatory(propertyPath));
		
		updatePort();

		Package protocolContainer = null;
		if ((port != null) && (port.getType() != null)) {
			protocolContainer = port.getType().getNearestPackage();
			// use specific content provider for the events of a protocol
		}
		editor.setContentProvider(new EventContentProvider(protocolContainer));

		if (valueEditor != null) {
			IObservableValue inputObservableValue = getInputObservableValue();

			if (inputObservableValue != null) {
				valueEditor.setStrategies();

				IValidator modelVal = getValidator();
				if (modelVal != null) {
					valueEditor.setModelValidator(modelVal);
				}
				valueEditor.setModelObservable(inputObservableValue);
			}
		}
	}

	/**
	 * Update the information about the selected source port
	 */
	protected void updatePort() {
		ModelElement portElement = input.getModelElement(propertyPath);
		EObject triggerEObj = ((UMLModelElement) portElement).getSource();
		port = null; // ((UMLModelElement) portElement).getFeature("port");
		if (triggerEObj instanceof Trigger) {
			Trigger trigger = (Trigger) triggerEObj;
			if (trigger.getPorts().size() > 0) {
				port = trigger.getPorts().get(0);
			}
		}
	}

	protected class EventContentProvider implements IStaticContentProvider {
		public EventContentProvider(Package protocolContainer) {
			this.protocolContainer = protocolContainer;
		}

		Package protocolContainer;

		@Override
		public Object[] getElements(Object inputElement) {
			return getElements();
		}

		@Override
		public void dispose() {
		}

		@Override
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			updatePort();
			if ((port != null) && (port.getType() != null)) {
				protocolContainer = port.getType().getNearestPackage();
			} else {
				protocolContainer = null;
			}
		}

		@Override
		public Object[] getElements() {
			ArrayList<Event> events = new ArrayList<Event>();
			if (protocolContainer != null) {
				for (PackageableElement pe : protocolContainer.getPackagedElements()) {
					if (pe instanceof CallEvent) {
						CallEvent event = (CallEvent) pe;
						Interface intf = event.getOperation().getInterface();
						RTMessageSet rtMessageSet = UMLUtil.getStereotypeApplication(intf, RTMessageSet.class);
						RTMessageKind kind = rtMessageSet.getRtMsgKind();
						if (kind == RTMessageKind.IN_OUT || (kind == RTMessageKind.IN && !port.isConjugated())
								|| (kind == RTMessageKind.OUT && port.isConjugated())) {
							events.add(event);
						}
					}
				}
			}
			return events.toArray();
		}
	}
}
