/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Onder Gurcan <onder.gurcan@cea.fr>
 *
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.core.types.advice;

import java.util.Collection;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.commands.ConfigureElementCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.ConfigureRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateRelationshipRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.papyrus.commands.Activator;
import org.eclipse.papyrus.infra.core.resource.ModelSet;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.infra.viewpoints.configuration.PapyrusView;
import org.eclipse.papyrus.infra.viewpoints.policy.PolicyChecker;
import org.eclipse.papyrus.infra.viewpoints.policy.ViewPrototype;
import org.eclipse.papyrus.uml.diagram.composite.CreateCompositeDiagramCommand;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsuleUtils;
import org.eclipse.uml2.uml.Class;


/**
 * The helperadvice class used for UMLRealTime::Protocol.
 *
 * @author Onder Gurcan <onder.gurcan@cea.fr>
 *
 */
public class CapsuleEditHelperAdvice extends AbstractEditHelperAdvice {

	@Override
	public boolean approveRequest(IEditCommandRequest request) {
		if (request instanceof CreateRelationshipRequest) {
			return approveCreateRelationshipRequest(request);
		} else {
			return super.approveRequest(request);
		}
	}

	private boolean approveCreateRelationshipRequest(IEditCommandRequest request) {
		IElementType type = ((CreateRelationshipRequest) request).getElementType();
		if (type != null && ElementTypeUtils.isTypeCompatible(type, ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.RT_CONNECTOR_ID))) {
			return checkSourceAndTargetForRTConnector(((CreateRelationshipRequest) request));
		}
		return super.approveRequest(request);
	}

	protected boolean checkSourceAndTargetForRTConnector(CreateRelationshipRequest createRelationshipRequest) {
		EObject source = createRelationshipRequest.getSource();
		EObject target = createRelationshipRequest.getTarget();

		if (source != null) {
			if (ElementTypeUtils.matches(source, IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID)) {
				return false; // cannot connect a connector to a SAP
			} else if (ElementTypeUtils.matches(source, IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID)) {
				return false;
			}
		}

		if (target != null) {
			if (ElementTypeUtils.matches(target, IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID)) {
				return false; // cannot connect a connector to a SAP
			} else if (ElementTypeUtils.matches(target, IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID)) {
				return false;
			}
		}

		return true;
	}
	/**
	 * @see org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice#getBeforeConfigureCommand(org.eclipse.gmf.runtime.emf.type.core.requests.ConfigureRequest)
	 *
	 * @param request
	 * @return
	 */
	@Override
	protected ICommand getBeforeConfigureCommand(ConfigureRequest request) {
		final Class capsule = (Class) request.getElementToConfigure();

		return new ConfigureElementCommand(request) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {
				capsule.setIsActive(true);

				createCompositeDiagram();

				return CommandResult.newOKCommandResult(capsule);
			}

			protected Diagram createCompositeDiagram() {
				ServicesRegistry registry;
				try {
					registry = ServiceUtilsForEObject.getInstance().getServiceRegistry(capsule);
				} catch (ServiceException ex) {
					Activator.log.error(ex);
					return null;
				}
				ModelSet modelSet;
				try {
					modelSet = registry.getService(ModelSet.class);
				} catch (ServiceException ex) {
					Activator.log.error(ex);
					return null;
				}

				Collection<ViewPrototype> prototypes = PolicyChecker.getCurrent().getPrototypesFor(capsule);
				for (ViewPrototype prototype : prototypes) {
					PapyrusView view = prototype.getConfiguration();
					if (view != null && CapsuleUtils.UML_RT_CAPSULE_DIAGRAM.equals(view.getName())) {
						return new CreateHeadlessCapsuleDiagramCommand().createDiagram(modelSet, capsule, capsule, prototype, null, false);
					}
				}

				return null;

			}

		};
	}

	/**
	 * Specific implementation for the creation of the diagram, to avoid setting a name using a popup
	 */
	public static class CreateHeadlessCapsuleDiagramCommand extends CreateCompositeDiagramCommand {

		/**
		 * {@inheritDoc}
		 */
		@Override
		protected CommandResult doEditDiagramName(ViewPrototype prototype, String name) {
			// overrides dialog creation to edit the name
			return CommandResult.newOKCommandResult(null);
		}

	}
}
