/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine JANSSENS (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.locator;

import java.util.Iterator;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.papyrus.uml.diagram.common.figure.node.SubCompartmentLayoutManager;
import org.eclipse.papyrus.uml.diagram.common.locator.PortPositionLocator;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Port;

/**
 * The Class RTPortPositionLocator is a specific Port Position Locator for the Real Time Context,
 * to allow RTPort to be inside a ClassComposite.
 */
public class RTPortPositionLocator extends PortPositionLocator {


	/** The port. */
	private Element port;

	/**
	 * Instantiates a new RT port position locator.
	 *
	 * @param droppedElement
	 *            the dropped element
	 * @param parentFigure
	 *            the parent figure
	 * @param preferredSide
	 *            the preferred side
	 */
	public RTPortPositionLocator(Element droppedElement, IFigure parentFigure, int preferredSide) {
		super(parentFigure, preferredSide);

		this.port = droppedElement;
	}

	/**
	 * Constructor.
	 *
	 * @param hostFigure
	 * @param none
	 */
	public RTPortPositionLocator(IFigure hostFigure, int side) {
		super(hostFigure, side);
	}

	/**
	 * Calculates the preferred location according to the RTPort type.
	 * Only Internal and SAP Kind can be located internally.
	 * 
	 * @see org.eclipse.papyrus.uml.diagram.common.locator.PortPositionLocator#getPreferredLocation(org.eclipse.draw2d.geometry.Rectangle)
	 */
	@Override
	public Rectangle getPreferredLocation(Rectangle proposedLocation) {

		Rectangle newLocation = null;
		if (port instanceof Port) {
			RTPortKindEnum kind = RTPortUtils.getKind((Port) port);
			if (null != kind) {
				switch (kind) {

				case EXTERNAL:
				case RELAY:
				case SPP:
					newLocation = getPreferredBorderedLocation(proposedLocation);
					break;
				case INTERNAL:
				case SAP:
					newLocation = getPreferredInternalLocation(proposedLocation);
					break;
				default:
					newLocation = getPreferredBorderedLocation(proposedLocation);
					break;
				}

			} else {
				newLocation = getPreferredBorderedLocation(proposedLocation);
			}
		} else {
			newLocation = getPreferredBorderedLocation(proposedLocation);
		}
		return newLocation;
	}

	/**
	 * Gets the preferred internal location. As a Standard port is not internal,
	 * this method calculates the right position based on the Compartment bounds and the target Position.
	 * 
	 *
	 * @param proposedLocation
	 *            the proposed location
	 * @return the preferred internal location
	 */
	private Rectangle getPreferredInternalLocation(Rectangle proposedLocation) {

		// Initialize port location with proposed location
		// and resolve the bounds of it graphical parent
		Rectangle realLocation = new Rectangle(proposedLocation);

		// Retrieve the bound of the Compartment to avoid the Name Label area.
		IFigure compartment = getCompositeCompartmentFigure(parentFigure);
		Rectangle compartmentBounds;
		if (null != compartment) {
			compartmentBounds = compartment.getBounds().getCopy();

			// Calculate Max position around the graphical parent (1/2 size or the port around
			// the graphical parent bounds.
			// this is an intra rectangle
			int xMin = compartmentBounds.x + borderItemOffset;
			int xMax = compartmentBounds.x + compartmentBounds.width - borderItemOffset;
			int yMin = compartmentBounds.y + borderItemOffset;
			int yMax = compartmentBounds.y + compartmentBounds.height - borderItemOffset;

			// Modify Port location if MAX X or Y are exceeded
			if (realLocation.x < xMin) {
				realLocation.x = xMin;
			}

			if (realLocation.x > xMax) {
				realLocation.x = xMax;
			}

			if (realLocation.y < yMin) {
				realLocation.y = yMin;
			}

			if (realLocation.y > yMax) {
				realLocation.y = yMax;
			}
		}


		// Return constrained location
		return realLocation;




	}

	/**
	 * Retrieve the Compartment of the Composite Class.
	 *
	 * @param parentFigure
	 *            the parent figure
	 * @return the composite compartment figure
	 */
	private IFigure getCompositeCompartmentFigure(IFigure parentFigure) {
		IFigure composite = null;
		Iterator<?> childrenIterator = parentFigure.getChildren().iterator();
		while (null == composite && childrenIterator.hasNext()) {
			Object child = childrenIterator.next();
			if (child instanceof IFigure) {
				// The compartment figure should have the Compartment Layout Manager
				if (((IFigure) child).getLayoutManager() instanceof SubCompartmentLayoutManager) {
					composite = (IFigure) child;
				}

				if (null == composite) {
					composite = getCompositeCompartmentFigure((IFigure) child);
				}
			}

		}
		return composite;
	}

	/**
	 * Gets the preferred bordered location.
	 *
	 * @param proposedLocation
	 *            the proposed location
	 * @return the preferred bordered location
	 */
	private Rectangle getPreferredBorderedLocation(Rectangle proposedLocation) {
		return super.getPreferredLocation(proposedLocation);
	}

}
