/*****************************************************************************
 * Copyright (c) 2014 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import java.util.Collection;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.IClientContext;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.jface.window.Window;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.services.edit.internal.context.TypeContext;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.uml.properties.creation.UMLPropertyEditorFactory;
import org.eclipse.papyrus.views.properties.creation.CreateInDialog;
import org.eclipse.papyrusrt.umlrt.tooling.ui.Activator;
import org.eclipse.swt.widgets.Control;

/**
 * Abstract class for element type based value factories
 */
public abstract class AbstractElementTypeBasedValueFactory extends UMLPropertyEditorFactory {
	public AbstractElementTypeBasedValueFactory(EReference eReference) {
		super(eReference);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<Object> validateObjects(Collection<Object> objectsToValidate) {
		return objectsToValidate;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected EObject createObjectInDifferentContainer(Control widget) {
		EObject protocolContainer = chooseNewContainer(widget);
		if (protocolContainer == null) {
			return null;
		}

		// can only create protocol...
		IElementType elementType = chooseElementType(widget);

		if (elementType == null) {
			return null;
		}

		IElementEditService service = ElementEditServiceUtils.getCommandProvider(protocolContainer);
		if (service == null) {
			// Something isn't right ...
			return null;
		}

		IClientContext clientContext = null;
		try {
			clientContext = TypeContext.getContext();
		} catch (ServiceException e) {
			Activator.log.error(e);
		}
		if (clientContext == null) {
			// Something isn't right ...
			return null;
		}

		ICommand editCommand = service.getEditCommand(new CreateElementRequest(protocolContainer, elementType));
		if (editCommand != null && editCommand.canExecute()) {
			IStatus status;
			try {
				status = editCommand.execute(new NullProgressMonitor(), null);
				if (status != null && status.isOK()) {
					CommandResult result = editCommand.getCommandResult();
					if (result.getReturnValue() instanceof EObject) {
						return (EObject) result.getReturnValue();
					}
				}
			} catch (ExecutionException e) {
				Activator.log.error(e);
			}
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected EObject simpleCreateObject(Control widget) {
		EObject container = chooseNewContainer(widget);
		if (container == null) {
			return null;
		}
		
		// can only create protocol...
		IElementType elementType = chooseElementType(widget);

		if(elementType ==null) {
			return null;
		}
		
		IElementEditService service = ElementEditServiceUtils.getCommandProvider(container);
		if (service == null) {
			// Something isn't right ...
			return null;
		}

		IClientContext clientContext = null;
		try {
			clientContext = TypeContext.getContext();
		} catch (ServiceException e) {
			Activator.log.error(e);
		}
		if (clientContext == null) {
			// Something isn't right ...
			return null;
		}

		ICommand editCommand = service.getEditCommand(new CreateElementRequest(container, elementType));
		if (editCommand != null && editCommand.canExecute()) {
			IStatus status;
			try {
				status = editCommand.execute(new NullProgressMonitor(), null);
				if (status != null && status.isOK()) {
					CommandResult result = editCommand.getCommandResult();
					if (result.getReturnValue() instanceof EObject) {
						return (EObject) result.getReturnValue();
					}
				}
			} catch (ExecutionException e) {
				Activator.log.error(e);
			}

		}
		return null;
	}

	/**
	 * Selection of the container of the newly created Protocol
	 * 
	 * @param widget
	 *            the widget to get access to the shell
	 * @return
	 */
	protected EObject chooseNewContainer(Control widget) {
		EObject virtualElement = getVirtualElementToCreate();
		CreateInDialog dialog = new CreateInDialog(widget.getShell(), virtualElement);
		dialog.setProviders(containerContentProvider, referenceContentProvider, containerLabelProvider, referenceLabelProvider);
		dialog.setTitle(getCreationDialogTitle());
		int result = dialog.open();
		if (result != Window.OK) {
			return null;
		}
		return dialog.getContainer();
	}

	protected abstract EObject getVirtualElementToCreate();

	protected abstract IElementType chooseElementType(Control widget);


}
