/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine JANSSENS (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.IObserving;
import org.eclipse.core.databinding.observable.value.ValueDiff;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.papyrus.commands.wrappers.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.UMLRealTimePackage;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * The Class PortRTKindObservableValue.
 */
public class PortRTKindObservableValue extends PapyrusObservableValue implements IObserving {

	/** The port. */
	private Port port;

	/** The stereotype application. */
	private EObject stereotypeApplication;

	/** The service. */
	private boolean service;

	/** The wired. */
	private boolean wired;

	/** The behavior. */
	private boolean behavior;

	/** The publish. */
	private boolean publish;

	/** The conjugated. */
	private boolean conjugated;

	/** The notification. */
	private boolean notification;

	/**
	 * Constructor.
	 *
	 * @param RTPortApplication
	 *            the RT port application
	 * @param domain
	 *            the domain
	 */
	public PortRTKindObservableValue(final EObject RTPortApplication, final TransactionalEditingDomain domain) {
		super(UMLUtil.getBaseElement(RTPortApplication), UMLUtil.getBaseElement(RTPortApplication).eContainingFeature(), domain);
		port = (Port) UMLUtil.getBaseElement(RTPortApplication);
		if (UMLUtil.getBaseElement(RTPortApplication) instanceof Port) {
			port = (Port) UMLUtil.getBaseElement(RTPortApplication);

			this.stereotypeApplication = RTPortApplication;
			stereotypeApplication.eAdapters().add(getListener());
			port.eAdapters().add(getListener());

			setUMLPropertiesValue();
			setUMLRTPropertiesValue();
		}
	}

	/**
	 * Retrieve Listener to put on the Stereotype Application.
	 *
	 * @return the listener
	 */
	protected AdapterImpl getListener() {
		return new AdapterImpl() {

			@Override
			public void notifyChanged(Notification notification) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();
				EList<EStructuralFeature> realTimeFeatureList = stereotypeApplication.eClass().getEStructuralFeatures();
				EList<EStructuralFeature> umlFeatureList = port.eClass().getEStructuralFeatures();

				// Case of StereotypeApplication that is changed
				if ((notifier == stereotypeApplication) && (type == Notification.SET)) {
					if (realTimeFeatureList.contains(feature)) {
						fireDiff(notification);
					}
					// Case if it is the UML Port that is modified
				} else if ((notifier == port) && (type == Notification.SET)) {
					if (umlFeatureList.contains(feature)) {
						fireDiff(notification);
					}
				}
			}

			private void fireDiff(Notification notification) {
				final ValueDiff diff = Diffs.createValueDiff(notification.getOldValue(), notification.getNewValue());
				getRealm().exec(new Runnable() {
					@Override
					public void run() {
						fireValueChange(diff);
					}
				});
			}

		};
	}


	/**
	 * Set the Properties relative to the RT profile.
	 */
	private void setUMLRTPropertiesValue() {
		wired = RTPortUtils.isWired(port);
		notification = RTPortUtils.isNotification(port);
		publish = RTPortUtils.isPublish(port);
	}

	/**
	 * Set properties relative to UML.
	 */
	private void setUMLPropertiesValue() {
		service = RTPortUtils.isService(port);
		behavior = RTPortUtils.isBehavior(port);
		conjugated = RTPortUtils.isConjugated(port);
	}

	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#getValueType()
	 *
	 * @return
	 */
	@Override
	public Object getValueType() {
		return RTPortKindEnum.class;
	}

	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#getObserved()
	 *
	 * @return
	 */
	@Override
	public Object getObserved() {
		return port;
	}



	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#doGetValue()
	 *
	 * @return
	 */
	@Override
	protected Object doGetValue() {
		RTPortKindEnum kind = RTPortKindEnum.RELAY;

		setUMLPropertiesValue();
		setUMLRTPropertiesValue();

		if (service && wired && behavior) {
			kind = RTPortKindEnum.EXTERNAL;
		} else if (service && behavior && !wired) {
			kind = RTPortKindEnum.SPP;
		} else if (wired && behavior && !service) {
			kind = RTPortKindEnum.INTERNAL;
		} else if (service && wired && !behavior) {
			kind = RTPortKindEnum.RELAY;
		} else if (behavior && !wired && !service) {
			kind = RTPortKindEnum.SAP;
		}

		return kind;
	}




	/**
	 * @see org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue#getCommand(java.lang.Object)
	 *
	 * @param value
	 * @return
	 */
	@Override
	public Command getCommand(Object value) {
		Command command = null;
		if (value instanceof RTPortKindEnum) {
			// For Each Kind of Port, set the property accordingly
			switch ((RTPortKindEnum) value) {
			case EXTERNAL:
				command = getCommandForRTPort(true, true, true);
				break;
			case INTERNAL:
				command = getCommandForRTPort(true, false, true);
				break;
			case RELAY:
				command = getCommandForRTPort(false, true, true);
				break;
			case SAP:
				command = getCommandForRTPort(true, false, false);
				break;
			case SPP:
				command = getCommandForRTPort(true, true, false);
				break;
			default:
				// Relay Port by Default
				command = getCommandForRTPort(false, true, true);
				break;
			}
		}
		return command;
	}

	/**
	 * Get the Command to set the RT Port properties.
	 *
	 * @param behavior
	 *            Value of isBehavior attribute
	 * @param service
	 *            Value of isService attribute
	 * @param wired
	 *            Value of isWired attribute
	 * @return the command for rt port
	 */
	protected Command getCommandForRTPort(boolean behavior, boolean service, boolean wired) {
		CompoundCommand command = new CompoundCommand();
		command.append(getSetBehaviorCommand(behavior));
		command.append(getSetServiceCommand(service));
		command.append(getSetWiredCommand(wired));
		return command;
	}

	/**
	 * Gets the Command to set the publish parameter.
	 *
	 * @param publish
	 *            The publish value to be set
	 * @return The Command to set the isPublish parameter
	 */
	protected Command getSetPublishCommand(final boolean publish) {
		IElementEditService serviceEdit = ElementEditServiceUtils.getCommandProvider(stereotypeApplication);
		EStructuralFeature feature = UMLRealTimePackage.eINSTANCE.getRTPort_IsPublish();
		SetRequest request = new SetRequest(stereotypeApplication, feature, publish);
		ICommand gmfCommand = serviceEdit.getEditCommand(request);
		Command cmd = GMFtoEMFCommandWrapper.wrap(gmfCommand);

		return cmd;
	}


	/**
	 * Gets the Command to set the wired parameter.
	 *
	 * @param wired
	 *            The wired value to be set
	 * @return The Command to set the isWired parameter
	 */
	protected Command getSetWiredCommand(final boolean wired) {
		IElementEditService serviceEdit = ElementEditServiceUtils.getCommandProvider(stereotypeApplication);
		EStructuralFeature feature = UMLRealTimePackage.eINSTANCE.getRTPort_IsWired();
		SetRequest request = new SetRequest(stereotypeApplication, feature, wired);
		ICommand gmfCommand = serviceEdit.getEditCommand(request);
		Command cmd = GMFtoEMFCommandWrapper.wrap(gmfCommand);

		return cmd;
	}

	/**
	 * Gets the Command to set the service parameter.
	 *
	 * @param service
	 *            The service value to be set
	 * @return The Command to set the isService parameter
	 */
	protected Command getSetServiceCommand(final boolean service) {
		Command cmd = null;
		IElementEditService serviceEdit = ElementEditServiceUtils.getCommandProvider(port);
		EStructuralFeature feature = UMLPackage.eINSTANCE.getPort_IsService();
		SetRequest request = new SetRequest(port, feature, service);
		ICommand gmfCommand = serviceEdit.getEditCommand(request);
		cmd = GMFtoEMFCommandWrapper.wrap(gmfCommand);

		return cmd;
	}

	/**
	 * Gets the Command to set the behavior parameter.
	 *
	 * @param behavior
	 *            The behavior value to be set
	 * @return The Command to set the isBehavior parameter
	 */
	protected Command getSetBehaviorCommand(final boolean behavior) {
		Command cmd = null;
		IElementEditService serviceEdit = ElementEditServiceUtils.getCommandProvider(port);
		EStructuralFeature feature = UMLPackage.eINSTANCE.getPort_IsBehavior();
		SetRequest request = new SetRequest(port, feature, behavior);
		ICommand gmfCommand = serviceEdit.getEditCommand(request);
		cmd = GMFtoEMFCommandWrapper.wrap(gmfCommand);

		return cmd;
	}


}
