/*****************************************************************************
 * Copyright (c) 2014 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.types.advice;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.CompositeCommand;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.commands.ConfigureElementCommand;
import org.eclipse.gmf.runtime.emf.type.core.commands.SetValueCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.ConfigureRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateRelationshipRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.papyrus.commands.wrappers.EMFtoGMFCommandWrapper;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.core.utils.ProtocolUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTPort;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.UMLRealTimePackage;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.VisibilityKind;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Edit Helper Advice for {@link RTPort}
 */
public class RTPortEditHelperAdvice extends AbstractEditHelperAdvice {

	@Override
	public boolean approveRequest(IEditCommandRequest request) {
		boolean result = false;
		// check that the type of the RTPort is a protocol, and nothing else
		if (request instanceof SetRequest) {
			SetRequest setRequest = (SetRequest) request;
			EStructuralFeature feature = setRequest.getFeature();
			if (UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature)) {
				// new value should be a protocol or null
				Object newValue = ((SetRequest) request).getValue();
				if (newValue instanceof EObject) {
					if (ProtocolUtils.isProtocol((EObject) newValue)) {
						result = true;
					} else {
						result = false;
					}
				} else {
					result = false;
				}
			} else {
				result = super.approveRequest(setRequest);
			}
		} else if (request instanceof CreateRelationshipRequest) {
			IElementType type = ((CreateRelationshipRequest) request).getElementType();
			if (type != null && ElementTypeUtils.isTypeCompatible(type, ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.RT_CONNECTOR_ID))) {
				result = checkSourceAndTarget(((CreateRelationshipRequest) request));
			} else {
				result = super.approveRequest(request);
			}
		} else {
			result = super.approveRequest(request);
		}

		return result;
	}

	protected boolean checkSourceAndTarget(CreateRelationshipRequest createRelationshipRequest) {
		EObject source = createRelationshipRequest.getSource();
		EObject target = createRelationshipRequest.getTarget();

		if (source != null) {
			if (ElementTypeUtils.matches(source, IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID)) {
				return false; // cannot connect a connector to a SAP
			} else if (ElementTypeUtils.matches(source, IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID)) {
				return false;
			}
		}

		if (target != null) {
			if (ElementTypeUtils.matches(target, IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID)) {
				return false; // cannot connect a connector to a SAP
			} else if (ElementTypeUtils.matches(target, IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID)) {
				return false;
			}
		}

		return true;
	}
	
	@Override
	protected ICommand getBeforeConfigureCommand(ConfigureRequest request) {
		final Port port = (Port) request.getElementToConfigure();

		return new ConfigureElementCommand(request) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {
				port.setIsOrdered(true);

				return CommandResult.newOKCommandResult(port);
			}
		};
	}

	@Override
	protected ICommand getBeforeCreateRelationshipCommand(CreateRelationshipRequest request) {
		return super.getBeforeCreateRelationshipCommand(request);
	}

	@Override
	protected ICommand getAfterSetCommand(SetRequest request) {
		CompositeCommand compositeCommand = new CompositeCommand("Set RTPort");

		EStructuralFeature feature = request.getFeature();
		Port portToEdit = (Port) request.getElementToEdit();
		RTPort rtPortToEdit = UMLUtil.getStereotypeApplication(portToEdit, RTPort.class);

		if (UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature)) {
			// if element is unnamed, set a name according to the new Type name
			Object newValue = request.getValue();
			EObject elementToEdit = request.getElementToEdit();
			if (newValue instanceof Type && elementToEdit instanceof Port && ((Port) elementToEdit).getName() == null) {
				String name = ((Type) newValue).getName();
				if (name != null && !name.isEmpty()) {
					String newName = Character.toLowerCase(name.charAt(0)) + name.substring(1);
					compositeCommand.add(new SetValueCommand(new SetRequest(request.getElementToEdit(), UMLPackage.eINSTANCE.getNamedElement_Name(), newName)));
				}
			}
		} else if (UMLPackage.eINSTANCE.getPort_IsService().equals(feature)) {


			ICommand visibilityCommand = getVisibilityCommand(request, portToEdit);
			if (null != visibilityCommand && visibilityCommand.canExecute()) {
				compositeCommand.add(visibilityCommand);
			}

			// isPublish = (!isWired && isService)
			ICommand publishedCommand = getPublishCommand(request, rtPortToEdit);
			if (null != publishedCommand && publishedCommand.canExecute()) {
				compositeCommand.add(publishedCommand);
			}

		}


		ICommand setKindCommand = getSetKindCommand(request);
		if (null != setKindCommand) {
			compositeCommand.add(setKindCommand);
		}

		return compositeCommand.isEmpty() ? super.getAfterSetCommand(request) : compositeCommand;
	}

	protected ICommand getVisibilityCommand(SetRequest request, Port portToEdit) {
		// The visibility of the port depends on the isService Value
		boolean isService = (Boolean) request.getValue();

		// set visibility of the port
		EAttribute visibilityFeature = UMLPackage.eINSTANCE.getNamedElement_Visibility();
		VisibilityKind visibility;

		if (isService) {
			// isService => public
			visibility = VisibilityKind.PUBLIC_LITERAL;
		} else {
			// !isService => protected
			visibility = VisibilityKind.PROTECTED_LITERAL;
		}

		IElementEditService serviceEdit = ElementEditServiceUtils.getCommandProvider(portToEdit);
		SetRequest setVisibilityRequest = new SetRequest(portToEdit, visibilityFeature, visibility);
		ICommand visibilityCommand = serviceEdit.getEditCommand(setVisibilityRequest);

		return visibilityCommand;
	}

	protected ICommand getPublishCommand(SetRequest request, RTPort rtPortToEdit) {
		// isPublish = (!isWired && isService)
		boolean isWired = rtPortToEdit.isWired();
		boolean isService = (Boolean) request.getValue();
		boolean isPublish = !isWired && isService;
		IElementEditService serviceEdit = ElementEditServiceUtils.getCommandProvider(rtPortToEdit);
		EStructuralFeature featurePublish = UMLRealTimePackage.eINSTANCE.getRTPort_IsPublish();
		SetRequest requestPublish = new SetRequest(rtPortToEdit, featurePublish, isPublish);
		ICommand gmfCommand = serviceEdit.getEditCommand(requestPublish);
		return gmfCommand;
	}


	protected ICommand getSetKindCommand(SetRequest request) {
		ICommand cmd = null;
		final EObject object = request.getElementToEdit();
		final Object kindParameter = request.getParameter(RTPortUtils.RTPORT_KIND_REQUEST_PARAMETER);
		if (object instanceof Port && (null != kindParameter) && kindParameter instanceof RTPortKindEnum) {
			final Port rtPort = (Port) object;
			RecordingCommand command = new RecordingCommand(TransactionUtil.getEditingDomain(object)) {
				@Override
				protected void doExecute() {

					if (RTPortKindEnum.EXTERNAL.equals(kindParameter)) {
						setExternalPort(rtPort);
					}
					if (RTPortKindEnum.INTERNAL.equals(kindParameter)) {
						setInternalPort(rtPort);
					}
					if (RTPortKindEnum.RELAY.equals(kindParameter)) {
						setRelayPort(rtPort);
					}
					if (RTPortKindEnum.SAP.equals(kindParameter)) {
						setSAPPort(rtPort);
					}
					if (RTPortKindEnum.SPP.equals(kindParameter)) {
						setSPPPort(rtPort);
					}

				}
			};
			cmd = EMFtoGMFCommandWrapper.wrap(command);
		}

		return cmd;
	}

	/**
	 * Sets the RT port as a external Behavior Port.
	 *
	 * @param externalBehaviorPort
	 *            the new RT port as a external Behavior Port
	 */
	protected void setExternalPort(final Port externalBehaviorPort) {
		// isService: true
		externalBehaviorPort.setIsService(true);

		// isBehavior: true
		externalBehaviorPort.setIsBehavior(true);

		// isWired: true
		RTPort stereotype = UMLUtil.getStereotypeApplication(externalBehaviorPort, RTPort.class);
		stereotype.setIsWired(true);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: public
		externalBehaviorPort.setVisibility(VisibilityKind.PUBLIC_LITERAL);
	}

	/**
	 * Sets the RT port as internal Behavior Port.
	 *
	 * @param internalBehaviorPort
	 *            the new RT port as internal Behavior Port
	 */
	protected void setInternalPort(final Port internalBehaviorPort) {
		// isService: false
		internalBehaviorPort.setIsService(false);

		// isBehavior: true
		internalBehaviorPort.setIsBehavior(true);

		// isWired: true
		RTPort stereotype = UMLUtil.getStereotypeApplication(internalBehaviorPort, RTPort.class);
		stereotype.setIsWired(true);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: protected
		internalBehaviorPort.setVisibility(VisibilityKind.PROTECTED_LITERAL);
	}

	/**
	 * Sets the RT port as a relay port.
	 *
	 * @param relayPort
	 *            the new RT port as a relay port
	 */
	protected void setRelayPort(final Port relayPort) {
		// isService: true
		relayPort.setIsService(true);

		// isBehavior: false
		relayPort.setIsBehavior(false);

		// isWired: true
		RTPort stereotype = UMLUtil.getStereotypeApplication(relayPort, RTPort.class);
		stereotype.setIsWired(true);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: public
		relayPort.setVisibility(VisibilityKind.PUBLIC_LITERAL);
	}

	/**
	 * Sets the RT port properties as an service Access Point.
	 *
	 * @param serviceAccessPoint
	 *            the new RT port as an service Access Point
	 */
	protected void setSAPPort(final Port serviceAccessPoint) {
		// isService: false
		serviceAccessPoint.setIsService(false);

		// isBehavior: true
		serviceAccessPoint.setIsBehavior(true);

		// isWired: false
		RTPort stereotype = UMLUtil.getStereotypeApplication(serviceAccessPoint, RTPort.class);
		stereotype.setIsWired(false);

		// isPublish: false
		stereotype.setIsPublish(false);

		// Visibility: protected
		serviceAccessPoint.setVisibility(VisibilityKind.PROTECTED_LITERAL);
	}

	/**
	 * Sets the RT port as service provision point.
	 *
	 * @param serviceProvisionPoint
	 *            the new RT port as service provision point
	 */
	protected void setSPPPort(final Port serviceProvisionPoint) {
		// isService: true
		serviceProvisionPoint.setIsService(true);

		// isBehavior: true
		serviceProvisionPoint.setIsBehavior(true);

		// isWired: false
		RTPort stereotype = UMLUtil.getStereotypeApplication(serviceProvisionPoint, RTPort.class);
		stereotype.setIsWired(false);

		// isPublish: true
		stereotype.setIsPublish(true);

		// Visibility: public
		serviceProvisionPoint.setVisibility(VisibilityKind.PUBLIC_LITERAL);
	}

}
