/*****************************************************************************
 * Copyright (c) 2014 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.types.advice;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.commands.ConfigureElementCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.ConfigureRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.infra.services.edit.utils.GMFCommandUtils;
import org.eclipse.papyrus.uml.service.types.element.UMLElementTypes;
import org.eclipse.papyrusrt.umlrt.core.types.UMLRTElementTypesEnumerator;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.Region;

/**
 * Edit Helper Advice for State machines
 */
public class RTRegionEditHelperAdvice extends AbstractEditHelperAdvice {

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ICommand getBeforeConfigureCommand(final ConfigureRequest request) {
		// create the region and an initial state in the region
		ICommand command = new ConfigureElementCommand(request) {

			/**
			 * {@inheritDoc}
			 */
			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) {
				EObject elementToConfigure = request.getElementToConfigure();
				if (!(elementToConfigure instanceof Region)) {
					return CommandResult.newErrorCommandResult("Element to configure is not a region");
				}
				Region region = (Region) elementToConfigure;
				try {
					Pseudostate initialState = createInitialState(region, monitor, info, null);
					return CommandResult.newOKCommandResult(initialState);
				} catch (ExecutionException e) {
					return CommandResult.newErrorCommandResult(e);
				}
			}
		};

		ICommand superCommand = super.getBeforeConfigureCommand(request);
		if (superCommand != null) {
			// draw the diagram, with initial state displayed
			return command.compose(superCommand).reduce();
		}
		return command.reduce();

	}

	/**
	 * Creates the initial state in the given region
	 * 
	 * @param region
	 *            the region in which to create the pseudo state
	 * @param monitor
	 *            the progress monitor for the creation command exexcution
	 * @param info
	 *            the IAdaptable (or <code>null</code>) provided by the
	 *            caller in order to supply UI information for prompting the
	 *            user if necessary. When this parameter is not
	 *            <code>null</code>, it should minimally contain an adapter
	 *            for the org.eclipse.swt.widgets.Shell.class.
	 * @param pseudoStateName
	 *            the name of the new pseudo state or null if the name should not be set
	 * @return the newly created pseudo state
	 * @throws ExecutionException
	 *             exception thrown if the command execution encouter issues
	 */
	protected Pseudostate createInitialState(Region region, IProgressMonitor monitor, IAdaptable info, String pseudoStateName) throws ExecutionException {
		EObject newElement = null;
		CreateElementRequest createElementRequest = new CreateElementRequest(region, UMLRTElementTypesEnumerator.RT_PSEUDO_STATE_INITIAL);
		// get command from edit service
		IElementEditService provider = ElementEditServiceUtils.getCommandProvider(region);
		if (provider == null) {
			throw new ExecutionException("Impossible to get the provider from " + region);
		}

		ICommand createGMFCommand = provider.getEditCommand(createElementRequest);
		if (createGMFCommand != null) {
			if (createGMFCommand.canExecute()) {
				IStatus status = createGMFCommand.execute(monitor, info);
				if (status.isOK()) {
					newElement = GMFCommandUtils.getCommandEObjectResult(createGMFCommand);

					if (!(newElement instanceof Pseudostate)) {
						throw new ExecutionException("Element creation problem for " + UMLElementTypes.PSEUDOSTATE.getDisplayName() + ".");
					}

					((Pseudostate) newElement).setName(pseudoStateName);
					return ((Pseudostate) newElement);

				} else {
					throw new ExecutionException("Impossible to create the pseudo state");
				}
			} else {
				throw new ExecutionException("Command to create the pseudo state is not executable");
			}
		} else {
			throw new ExecutionException("Impossible to find a command to create the Pseudostate");
		}
		
	}
}
