/*****************************************************************************
 * Copyright (c) 2014 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.types.advice;

import java.util.Collection;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.commands.ConfigureElementCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.ConfigureRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.papyrus.commands.Activator;
import org.eclipse.papyrus.infra.core.resource.ModelSet;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.infra.services.edit.utils.GMFCommandUtils;
import org.eclipse.papyrus.infra.viewpoints.configuration.PapyrusView;
import org.eclipse.papyrus.infra.viewpoints.policy.PolicyChecker;
import org.eclipse.papyrus.infra.viewpoints.policy.ViewPrototype;
import org.eclipse.papyrus.uml.diagram.statemachine.CreateStateMachineDiagramCommand;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.core.types.UMLRTElementTypesEnumerator;
import org.eclipse.papyrusrt.umlrt.core.utils.StateMachineUtils;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.StateMachine;

/**
 * Edit Helper Advice for State machines
 */
public class RTStateMachineEditHelperAdvice extends AbstractEditHelperAdvice {

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ICommand getBeforeConfigureCommand(final ConfigureRequest request) {
		// create the region and an initial state in the region
		ICommand command = new ConfigureElementCommand(request) {

			/**
			 * {@inheritDoc}
			 */
			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) {
				EObject elementToConfigure = request.getElementToConfigure();
				if (!(elementToConfigure instanceof StateMachine)) {
					return CommandResult.newErrorCommandResult("Element to configure is not a state machine");
				}
				StateMachine stateMachine = (StateMachine) elementToConfigure;
				stateMachine.setIsReentrant(false);
				stateMachine.getContext().setClassifierBehavior(stateMachine);
				try {
					Region region = createRegion(stateMachine, monitor, info, "Region");
					return CommandResult.newOKCommandResult(region);
				} catch (ExecutionException e) {
					return CommandResult.newErrorCommandResult(e);
				}
			}

		};

		ICommand superCommand = super.getBeforeConfigureCommand(request);
		if (superCommand != null) {
			// draw the diagram, with initial state displayed
			return command.compose(superCommand).reduce();
		}
		return command.reduce();

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ICommand getAfterConfigureCommand(final ConfigureRequest request) {

		return new ConfigureElementCommand(request) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {

				/* @noname String name = (request.getElementToConfigure() instanceof NamedElement) ? ((NamedElement) request.getElementToConfigure()).getName() : "smDiagram"; */
				createStateMachineDiagram(null);
				return CommandResult.newOKCommandResult(request.getElementToConfigure());
			}

			protected Diagram createStateMachineDiagram(String name) {
				ServicesRegistry registry;
				try {
					registry = ServiceUtilsForEObject.getInstance().getServiceRegistry(request.getElementToConfigure());
				} catch (ServiceException ex) {
					Activator.log.error(ex);
					return null;
				}
				ModelSet modelSet;
				try {
					modelSet = registry.getService(ModelSet.class);
				} catch (ServiceException ex) {
					Activator.log.error(ex);
					return null;
				}

				Collection<ViewPrototype> prototypes = PolicyChecker.getCurrent().getPrototypesFor(request.getElementToConfigure());
				for (ViewPrototype prototype : prototypes) {
					PapyrusView view = prototype.getConfiguration();
					if (view != null && StateMachineUtils.UMLRT_STATE_MACHINE_DIAGRAM.equals(view.getName())) {
						return new CreateHeadlessStateMachineDiagramCommand().createDiagram(modelSet, request.getElementToConfigure(), request.getElementToConfigure(), prototype, name, false);
					}
				}

				return null;

			}

		};
	}

	/**
	 * Specific implementation for the creation of the diagram, to avoid setting a name using a popup
	 */
	public static class CreateHeadlessStateMachineDiagramCommand extends CreateStateMachineDiagramCommand {

		/**
		 * {@inheritDoc}
		 */
		@Override
		protected CommandResult doEditDiagramName(ViewPrototype prototype, String name) {
			// overrides dialog creation to edit the name
			return CommandResult.newOKCommandResult(name);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		protected void initializeDiagram(EObject diagram) {
			super.initializeDiagram(diagram);

			// enforce name = null
			((Diagram) diagram).setName(null);
		}

	}

	/**
	 * Create the region for the given state machine
	 * 
	 * @param stateMachine
	 *            state machine in which created region will be added
	 * @param monitor
	 *            progress monitor for the command execution
	 * @param info
	 * @param regionName
	 *            name of the region to create
	 * @return the newly created region
	 * @throws ExecutionException
	 *             execution thrown in case of issues happening in the command execution
	 */
	protected Region createRegion(StateMachine stateMachine, IProgressMonitor monitor, IAdaptable info, String regionName) throws ExecutionException {
		EObject newElement = null;
		CreateElementRequest createElementRequest = new CreateElementRequest(stateMachine, ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.RT_REGION_ID));
		// get command from edit service
		IElementEditService provider = ElementEditServiceUtils.getCommandProvider(stateMachine);
		if (provider == null) {
			throw new ExecutionException("Impossible to get the provider from " + stateMachine);
		}

		ICommand createGMFCommand = provider.getEditCommand(createElementRequest);
		if (createGMFCommand != null) {
			if (createGMFCommand.canExecute()) {
				IStatus status = createGMFCommand.execute(monitor, info);
				if (status.isOK()) {
					newElement = GMFCommandUtils.getCommandEObjectResult(createGMFCommand);

					if (!(newElement instanceof Region)) {
						throw new ExecutionException("Element creation problem for " + UMLRTElementTypesEnumerator.RT_REGION.getDisplayName() + ".");
					}
					((Region) newElement).setName(regionName);
					return ((Region) newElement);
				} else {
					throw new ExecutionException("Impossible to create the region");
				}
			} else {
				throw new ExecutionException("Command to create the region is not executable");
			}
		} else {
			throw new ExecutionException("Impossible to find a command to create the region");
		}
	}
}
