/*****************************************************************************
 * Copyright (c) 2015 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.core.utils;

import java.util.Collections;
import java.util.List;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.uml.tools.utils.PackageUtil;
import org.eclipse.papyrusrt.umlrt.core.Activator;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTMessageKind;
import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.PackageableElement;

/**
 * Utility class for UMLRT::Protocols
 */
public class ProtocolUtils {

	/**
	 * Returns the package that corresponds to the protocol container. There should be a check here for the applied stereotype on the package.
	 * 
	 * @param protocol
	 *            the collaboration for which the protocol container is searched
	 * @return the collaboration for which the protocol container is searched or <code>null</code> if none is found
	 */
	public static Package getProtocolContainer(Collaboration protocol) {
		return protocol.getNearestPackage();
	}

	/**
	 * Returns <code>true</code> if the context element is a Protocol (Collaboration stereotyped by "protocol")
	 * 
	 * @param context
	 *            the eobject to test
	 * @return <code>true</code> if the context element is a Protocol (Collaboration stereotyped by "protocol")
	 */
	public static Boolean isProtocol(EObject context) {
		return ElementTypeUtils.matches(context, IUMLRTElementTypes.PROTOCOL_ID);
	}

	/**
	 * @param in
	 * @return
	 */
	public static List<Operation> getRTMessages(Collaboration protocol, RTMessageKind direction, boolean showInherited) {
		Package protocolContainer = getProtocolContainer(protocol);
		if(protocolContainer ==null) {
			Activator.log.error("Impossible to get the root protocol container", null);
			return Collections.emptyList();
		}
		
		return ProtocolContainerUtils.getRTMessages(protocolContainer, direction, showInherited);
	}

	/**
	 * @param editContext
	 * @return
	 */
	public static Interface getMessageSet(Collaboration editContext, RTMessageKind direction) {
		Package protocolContainer = getProtocolContainer(editContext);

		if (protocolContainer != null) {
			return ProtocolContainerUtils.getMessageSet(protocolContainer, direction);
		}
		return null;
	}

	/**
	 * @param editContext
	 * @return
	 */
	public static Interface getMessageSetIn(Collaboration editContext) {
		return getMessageSet(editContext, RTMessageKind.IN);
	}

	/**
	 * @param editContext
	 * @return
	 */
	public static Interface getMessageSetOut(Collaboration editContext) {
		return getMessageSet(editContext, RTMessageKind.OUT);
	}

	/**
	 * @param editContext
	 * @return
	 */
	public static Interface getMessageSetInOut(Collaboration editContext) {
		return getMessageSet(editContext, RTMessageKind.IN_OUT);
	}

	/**
	 * Get all protocols that are associated with a protocol, in particular
	 * "inherited" protocols. Currently, inheritance is not supported, but
	 * all protocols implicitly "inherit" a base protocol.
	 * 
	 * @param protocol
	 * @return all protocols.
	 */
	public static EList<Package> getAllProtocols(Package protocol) {
		EList<Package> allProtocols = new BasicEList<Package>();
		allProtocols.add(protocol);
		Package baseProtocol = getBaseProtocol(protocol);
		if ((baseProtocol != null) && (protocol != baseProtocol)) {
			allProtocols.add(baseProtocol);
		}
		return allProtocols;
	}
	
	/**
	 * Check whether a given protocol candidate is a common protocol in a set of protocols (including inherited protocols)
	 * @param protocolSet a list of protocols for which we examine
	 * @param protocolCandidate a candidate for a protocol that may be common
	 * @return true, iff the protocol is a common protocol
	 */
	public static boolean isCommonProtocol(EList<Package> protocolSet, Package protocolCandidate) {
		for (Package protocol : protocolSet) {
			if (!ProtocolUtils.getAllProtocols(protocol).contains(protocolCandidate)) {
				return false;
			}
		}
		return true;
	}
	
	/**
	 * Obtain the base protocol (UMLRTBaseCommProtocol) from the RTS library
	 *
	 * @param anElement an arbitrary element from a model that imports the RTS library.
	 *  
	 * @return the base protocol
	 */
	public static Package getBaseProtocol(Element anElement) {
		if (anElement != null) {
			Package root = PackageUtil.getRootPackage(anElement);
			if (anElement != null) {
				// the code assumes that the UML/RT RTS package has been imported. Thus,
				// the protocols package is directly visible in the root package.
				if (root instanceof Package) {
					NamedElement protocols = ((Package) root).getMember("Protocols"); //$NON-NLS-1$
					if (protocols instanceof Package) {
						PackageableElement baseProtocolCandidate = ((Package) protocols).getPackagedElement("UMLRTBaseCommProtocol"); //$NON-NLS-1$
						if (baseProtocolCandidate instanceof Package) {
							return (Package) baseProtocolCandidate;
						}
					}
				}
			}
		}
		return null;
	}
}
