/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Mickael ADAM (ALL4TEC) mickael.adam@all4tec.net - add RTPORT_KIND_REQUEST_PARAMETER 
 * Celine Janssens (ALL4TEC) celine.janssens@all4tec.net  - Bug 472884
 *   
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.utils;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.UnexecutableCommand;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.common.core.command.CompositeCommand;
import org.eclipse.gmf.runtime.emf.type.core.commands.SetValueCommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.papyrus.commands.wrappers.GMFtoEMFCommandWrapper;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTPort;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.UMLRealTimePackage;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Utility class on {@link Operation} that are RTMessage.
 */
public class RTPortUtils {

	/** The Constant RTPORT_KIND_REQUEST_PARAMETER. Used as key on creation request to define which kind of RTPort we want to create. */
	public static final String RTPORT_KIND_REQUEST_PARAMETER = "RTPORT_KIND";

	/**
	 * Checks if is a RT port.
	 *
	 * @param eObject
	 *            the e object
	 * @return true, if is a RT port
	 */
	public static boolean isRTPort(EObject eObject) {
		if (eObject instanceof Port) {
			// get Owner of the operation, and check if this is a messageSET
			Port port = (Port) eObject;
			return (null != getStereotypeApplication(port));

		}
		return false;
	}

	/**
	 * Returns <code>true</code> if a connector is connected to the specified Port
	 * 
	 * @param port
	 *            the port to check
	 * @return <code>true</code> if a connector is connected to the specified Port
	 */
	public static boolean isConnected(Port port) {
		return port != null && !port.getEnds().isEmpty();
	}

	public static boolean isConnectedInside(Port port) {
		if (!isConnected(port)) { // it should be at least connected
			return false;
		}

		// check the connected elements
		for (ConnectorEnd connectorEnd : port.getEnds()) {
			Element connectorEndContainer = connectorEnd.getOwner(); // should be a connector
			if (connectorEndContainer instanceof Connector) {
				Element connectorContainer = ((Connector) connectorEndContainer).getOwner();
				if (connectorContainer != null && connectorContainer.equals(port.getClass_())) {
					return true;
				}
			}
		}
		return false;
	}

	public static boolean isConnectedOutside(Port port) {
		if (!isConnected(port)) { // it should be at least connected
			return false;
		}

		// check the connected elements
		for (ConnectorEnd connectorEnd : port.getEnds()) {
			Element connectorEndContainer = connectorEnd.getOwner(); // should be a connector
			if (connectorEndContainer instanceof Connector) {
				Element connectorContainer = ((Connector) connectorEndContainer).getOwner();
				if (connectorContainer != null && !connectorContainer.equals(port.getClass_())) {
					return true;
				}
			}
		}
		return false;
	}

	public static RTPort getStereotypeApplication(Port port) {
		return UMLUtil.getStereotypeApplication(port, RTPort.class);

	}


	public static boolean isWired(Port port) {

		return getStereotypeApplication(port) == null ? false : (Boolean) getStereotypeApplication(port).isWired();
	}

	public static boolean isPublish(Port port) {
		return getStereotypeApplication(port) == null ? false : (Boolean) getStereotypeApplication(port).isPublish();
	}

	public static boolean isNotification(Port port) {
		return getStereotypeApplication(port) == null ? false : (Boolean) getStereotypeApplication(port).isNotification();
	}

	public static boolean isBehavior(Port port) {
		return port.isBehavior();
	}

	public static boolean isService(Port port) {
		return port.isService();
	}

	public static boolean isConjugated(Port port) {
		return port.isConjugated();
	}

	/**
	 * Gets the kind of the RT Port.
	 *
	 * @param port
	 *            the port
	 * @return the kind
	 */
	public static RTPortKindEnum getKind(Port port) {
		RTPortKindEnum kind = null;
		if (isRTPort(port)) {
			if (isService(port) && isWired(port) && isBehavior(port) && !isPublish(port)) {
				kind = RTPortKindEnum.EXTERNAL;
			} else if (isBehavior(port) && isPublish(port) && !isWired(port)) {
				kind = RTPortKindEnum.SPP; // isService won't be checked here => Cf. bug 477033
			} else if (isWired(port) && isBehavior(port) && !isService(port) && !isPublish(port)) {
				kind = RTPortKindEnum.INTERNAL;
			} else if (isService(port) && isWired(port) && !isBehavior(port) && !isPublish(port)) {
				kind = RTPortKindEnum.RELAY;
			} else if (isBehavior(port) && !isWired(port) && !isPublish(port)) {
				kind = RTPortKindEnum.SAP;// isService won't be checked here => Cf. bug 477033
			}
		}

		return kind;
	}

	/**
	 * Returns the command that sets all parameters of the specified port to turn him into the new kind, and checks initially that the command is possible
	 * 
	 * @param port
	 *            the port to set
	 * @param newKind
	 *            the new kind of the RTPort
	 * @param checkInitialStatus
	 *            <code>true</code> if the original status of the port should be checked to see if the command can be executed or not
	 * @return the command that sets all parameters of the specified port to turn him into the new kind or <code>null</code>
	 */
	public static Command getChangeKindCommand(Port port, RTPortKindEnum newKind, boolean checkInitialStatus) {
		if (!checkInitialStatus) {
			return getChangeKindCommand(port, newKind);
		}
		switch (getKind(port)) {
		case EXTERNAL:
			if (isConnectedOutside(port)) {
				if (!(RTPortKindEnum.RELAY == newKind || RTPortKindEnum.EXTERNAL == newKind)) { // can only go from EXTERNAL to RELAY when connected outside
					return UnexecutableCommand.INSTANCE;
				}
			} else if (isConnected(port)) {
				if (RTPortKindEnum.SAP == newKind || RTPortKindEnum.SPP == newKind) {
					return UnexecutableCommand.INSTANCE; // no connection on SAP and SPP
				}
			}
			break;
		case INTERNAL:
			if (isConnectedInside(port)) {
				if (!(RTPortKindEnum.RELAY == newKind || RTPortKindEnum.INTERNAL == newKind)) { // can only go from INTERNAL to RELAY when connected inside
					return UnexecutableCommand.INSTANCE;
				}
			}
			if (isConnected(port)) {
				if (RTPortKindEnum.SAP == newKind || RTPortKindEnum.SPP == newKind) {
					return UnexecutableCommand.INSTANCE; // no connection on SAP and SPP
				}
			}
			break;
		case RELAY:
			if (isConnectedInside(port)) {
				if (!(RTPortKindEnum.RELAY == newKind || RTPortKindEnum.INTERNAL == newKind)) { // can only go from RELAY to INTERNAL when connected inside
					return UnexecutableCommand.INSTANCE;
				}
			}
			if (isConnectedOutside(port)) {
				if (!(RTPortKindEnum.RELAY == newKind || RTPortKindEnum.EXTERNAL == newKind)) {// can only go from RELAY to EXTERNAL when connected outside
					return UnexecutableCommand.INSTANCE;
				}
			}
			if (isConnected(port)) {
				if (RTPortKindEnum.SAP == newKind || RTPortKindEnum.SPP == newKind) {
					return UnexecutableCommand.INSTANCE; // no connection on SAP and SPP
				}
			}
			break;
		case SAP:
			break;
		case SPP:
			break;
		default:
			break;
		}
		return getChangeKindCommand(port, newKind);
	}

	/**
	 * Returns the command that sets all parameters of the specified port to turn him into the new kind
	 * 
	 * @param port
	 *            the port to set
	 * @param newKind
	 *            the new kind of the RTPort
	 * @return the command that sets all parameters of the specified port to turn him into the new kind or <code>null</code>
	 */
	public static Command getChangeKindCommand(Port port, RTPortKindEnum newKind) {
		Command command = null;
		switch (newKind) {
		case EXTERNAL:
			command = getRTPortKindChangeCommand(port, true, true, true);
			break;
		case INTERNAL:
			command = getRTPortKindChangeCommand(port, false, true, true);
			break;
		case RELAY:
			command = getRTPortKindChangeCommand(port, true, true, false);
			break;
		case SAP:
			command = getRTPortKindChangeCommand(port, false, false, true);
			break;
		case SPP:
			command = getRTPortKindChangeCommand(port, true, false, true);
			break;
		default:
			// Relay Port by Default
			command = getRTPortKindChangeCommand(port, true, true, false);
			break;
		}
		return command;
	}

	protected static Command getRTPortKindChangeCommand(Port port, boolean service, boolean wired, boolean behavior) {
		RTPort rtPort = getStereotypeApplication(port);
		if (rtPort == null) {
			return UnexecutableCommand.INSTANCE;
		}

		CompositeCommand command = new CompositeCommand("Setting RTPort Kind"); // //$NON-NLS-1$
		command.add(new SetValueCommand(new SetRequest(port, UMLPackage.eINSTANCE.getPort_IsBehavior(), behavior)));
		command.add(new SetValueCommand(new SetRequest(port, UMLPackage.eINSTANCE.getPort_IsService(), service)));
		command.add(new SetValueCommand(new SetRequest(rtPort, UMLRealTimePackage.eINSTANCE.getRTPort_IsWired(), wired)));
		command.add(new SetValueCommand(new SetRequest(rtPort, UMLRealTimePackage.eINSTANCE.getRTPort_IsPublish(), !wired && service)));
		return GMFtoEMFCommandWrapper.wrap(command);
	}

	/**
	 * Returns <code>true</code> if specified port is a legacy SPP port
	 * <ul>
	 * <li>isService = false</li>
	 * <li>isWired = false</li>
	 * <li>isBehavior = true</li>
	 * <li>isPublish = true</li>
	 * </ul>
	 * 
	 * @param port
	 *            the port to check
	 * @return <code>true</code> if the port is a legacy SPP port, false otherwise
	 */
	public static boolean isLegacySpp(Port port) {
		if (RTPortKindEnum.SPP.equals(getKind(port))) { // it should be at least a "true" SPP port
			return !port.isService(); // isService should be false
		}
		return false;
	}

	/**
	 * Returns <code>true</code> if specified port is a legacy SAP port
	 * <ul>
	 * <li>isService = true</li>
	 * <li>isWired = false</li>
	 * <li>isBehavior = true</li>
	 * <li>isPublish = false</li>
	 * </ul>
	 * 
	 * @param port
	 *            the port to check
	 * @return <code>true</code> if the port is a legacy SAP port, false otherwise
	 */
	public static boolean isLegacySap(Port port) {
		if (RTPortKindEnum.SAP.equals(getKind(port))) { // it should be at least a "true" SAP port
			return port.isService(); // isService should be true
		}
		return false;
	}

}
