/*****************************************************************************
 * Copyright (c) 2015 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Celine Janssens (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.editpolicies;

import java.util.Iterator;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gef.commands.Command;
import org.eclipse.gmf.runtime.diagram.core.commands.DeleteCommand;
import org.eclipse.gmf.runtime.diagram.core.listener.DiagramEventBroker;
import org.eclipse.gmf.runtime.diagram.core.listener.NotificationListener;
import org.eclipse.gmf.runtime.diagram.core.preferences.PreferencesHint;
import org.eclipse.gmf.runtime.diagram.core.util.ViewUtil;
import org.eclipse.gmf.runtime.diagram.ui.editparts.GraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequest.ViewDescriptor;
import org.eclipse.gmf.runtime.emf.core.util.EObjectAdapter;
import org.eclipse.gmf.runtime.emf.type.core.IHintedType;
import org.eclipse.gmf.runtime.gef.ui.internal.editpolicies.GraphicalEditPolicyEx;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.infra.gmfdiag.common.model.NotationUtils;
import org.eclipse.papyrus.uml.diagram.common.util.CommandUtil;
import org.eclipse.papyrus.uml.diagram.composite.providers.UMLElementTypes;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsulePartUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsuleUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.Capsule;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.utils.NamedStyleProperties;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * This EditPolicy is used in the Real Time Context and allows to display Service Port canonically on CapsulePart.
 * To activate or deactivate the canonical behavior, the CSS property can be defined: {@link NamedStyleProperties.DISPLAY_SERVICE_PORT}.
 * 
 * @author Céline JANSSENS
 *
 */
public class CanonicalPortDisplayEditPolicy extends GraphicalEditPolicyEx implements NotificationListener {

	/**
	 * Role key of the Canonical Edit Policy
	 */
	public static final String CANONICAL_PORT_DISPLAY_ROLE = "CanonicalPortDisplay";// $NON-NLS-0$


	/**
	 * NamedStyle property to display all the service port of a Capsule
	 */
	private static final String DISPLAY_SERVICE_PORT = NamedStyleProperties.DISPLAY_SERVICE_PORT;

	/**
	 * The view of the host EditPart.
	 */
	private View hostView;

	/**
	 * The Capsule as uml Class
	 */
	private org.eclipse.uml2.uml.Class capsule;

	/**
	 * The CapsulePart as uml Property
	 */
	private Property capsulePart;

	/**
	 * The CSS Value of {@link DISPLAY_SERVICE_PORT}
	 */
	private boolean displayServicePort;


	/**
	 * @see org.eclipse.gef.editpolicies.AbstractEditPolicy#activate()
	 *
	 */
	@Override
	public void activate() {
		initialisation();
		getDiagramEventBroker().addNotificationListener(capsule, this);
		super.activate();
	}

	/**
	 * @see org.eclipse.gef.editpolicies.AbstractEditPolicy#deactivate()
	 *
	 */
	@Override
	public void deactivate() {
		getDiagramEventBroker().removeNotificationListener(capsule, this);
		super.deactivate();
	}

	/**
	 * Gets the diagram event broker from the editing domain.
	 *
	 * @return the diagram event broker
	 */
	protected DiagramEventBroker getDiagramEventBroker() {
		DiagramEventBroker eventBorker = null;
		TransactionalEditingDomain theEditingDomain = ((IGraphicalEditPart)getHost()).getEditingDomain();
		if(null != theEditingDomain) {
			eventBorker = DiagramEventBroker.getInstance(theEditingDomain);
		}
		return eventBorker;
	}


	/**
	 * Initialization of Members
	 */
	protected void initialisation() {
		if(getHost() instanceof GraphicalEditPart) {
			hostView = ((GraphicalEditPart)getHost()).getNotationView();
			displayServicePort = NotationUtils.getBooleanValue(hostView, DISPLAY_SERVICE_PORT, true);

			if(hostView.getElement() instanceof Property) {
				capsulePart = (Property)hostView.getElement();
			}

			if(CapsulePartUtils.isCapsulePart(capsulePart)) {
				Object type = capsulePart.getType();

				if((type instanceof Class) && (null != UMLUtil.getStereotypeApplication((Class)type, Capsule.class))) {
					capsule = (Class)type;
				}

			}
		}
	}


	/**
	 * @see org.eclipse.gmf.runtime.gef.ui.internal.editpolicies.GraphicalEditPolicyEx#refresh()
	 *      Create the Port to be created and delete the Orphan port
	 */
	@Override
	public void refresh() {
		createPortView();
		deletePortView();
		super.refresh();
	}

	/**
	 * Delete all the view that should not be there
	 */
	private void deletePortView() {
		for(Object child : hostView.getChildren()) {
			if((child instanceof Node)) {
				Object childElement = ((Node)child).getElement();
				// if the port element doesn't exist in the capsule, delete the view in the CapsulePart
				if((childElement instanceof Port) && (!capsule.getOwnedPorts().contains(childElement))) {
					deletePortView((Port)childElement);
				} else if(!displayServicePort) {
					deletePortView((Port)childElement);
				}
			}
		}

	}

	/**
	 * Creates the port view.
	 */
	private void createPortView() {
		initialisation();

		if((displayServicePort) && (null != capsule)) {
			for(Port port : capsule.getOwnedPorts()) {
				// Add listener on the UML Element
				getDiagramEventBroker().addNotificationListener(port, this);
				if(RTPortUtils.isService(port)) {// only service port should be displayed
					if(null == getPortView(port)) {// if the view of the port doesn't exist in the Capsule Part create it
						displayPortView(port);
					}
				}
			}

		}
	}

	/**
	 * Retrieve the View associated to the port for the Host
	 * 
	 * @param port
	 *        the UML element of the view we are looking for
	 * @return The view of the port into the host view children
	 */
	private View getPortView(Port port) {
		View portView = null;
		Iterator<?> iterator = hostView.getChildren().iterator();
		while(null == portView && iterator.hasNext()) {
			Object child = iterator.next();
			if((child instanceof Node) && ((Node)child).getElement().equals(port)) {
				portView = (Node)child;
			}
		}
		return portView;
	}


	/**
	 * Create the view of the port
	 * 
	 * @param port
	 */
	protected void displayPortView(Port port) {
		CreateViewRequest request = getCreationViewRequest(port);
		Command command = getHost().getCommand(request);

		CommandUtil.executeCommand(command, ((IGraphicalEditPart)getHost()));
	}

	/**
	 * Get the Creation View request for the Port
	 * 
	 * @param port
	 * @return
	 */
	protected CreateViewRequest getCreationViewRequest(Port port) {

		String semanticHint = ((IHintedType)UMLElementTypes.Port_3069).getSemanticHint();
		IAdaptable elementAdapter = new EObjectAdapter(port);
		ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, semanticHint, ViewUtil.APPEND, false, getDiagramPreferencesHint());
		CreateViewRequest createViewRequest = new CreateViewRequest(descriptor);

		return createViewRequest;
	}

	/**
	 * Gets the diagram preferences hint.
	 *
	 * @return the diagram preferences hint
	 */
	protected PreferencesHint getDiagramPreferencesHint() {
		return ((IGraphicalEditPart)getHost()).getDiagramPreferencesHint();
	}


	/**
	 * @see org.eclipse.gmf.runtime.diagram.core.listener.NotificationListener#notifyChanged(org.eclipse.emf.common.notify.Notification)
	 *
	 * @param notification
	 */
	@Override
	public void notifyChanged(Notification notification) {
		int type = notification.getEventType();
		Object notifier = notification.getNotifier();
		Object newValue = notification.getNewValue();
		Object oldValue = notification.getOldValue();

		// If a new port is created in the Capsule then create the new view accordingly into the CapsulePart
		if((type == Notification.ADD) && (newValue instanceof Port)) {
			if((notifier instanceof Class) && (CapsuleUtils.isCapsule((Class)notifier))) {
				createPortView();
			}
		}

		// If a port is deleted in the Capsule then delete the view accordingly into the CapsulePart
		if((type == Notification.REMOVE) && (oldValue instanceof Port)) {
			if((notifier instanceof Class) && (CapsuleUtils.isCapsule((Class)notifier))) {
				deletePortView((Port)oldValue);
			}
		}

		// If a port is deleted in the Capsule then delete the view accordingly into the CapsulePart
		if((type == Notification.SET) && (notifier instanceof Port)) {
			Object feature = notification.getFeature();

			if((null != feature) && (UMLPackage.PORT__IS_SERVICE == ((EStructuralFeature)feature).getFeatureID())) {
				if((newValue instanceof Boolean) && ((Boolean)newValue)) {
					createPortView();
				} else {
					deletePortView((Port)notifier);
				}
			}


		}


	}

	/**
	 * Delete the view of the Port in the CapsulePart visualization
	 * 
	 * @param port
	 *        The Port Element corresponding to the view to delete
	 */
	private void deletePortView(Port port) {
		View viewPort = getPortView(port);
		if(null != viewPort) {
			DeleteCommand command = new DeleteCommand(viewPort);
			CommandUtil.executeUnsafeCommand(command, getHost());
		}
	}

}
