/*****************************************************************************
 * Copyright (c) 2010, 2015 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Onder GURCAN (CEA LIST) onder.gurcan@cea.fr - Initial API and implementation
 *
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.modelelement;

import static org.eclipse.papyrus.uml.tools.util.MultiplicityParser.ONE;
import static org.eclipse.papyrus.uml.tools.util.MultiplicityParser.OPTIONAL;

import java.lang.reflect.InvocationTargetException;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.core.databinding.observable.ChangeEvent;
import org.eclipse.core.databinding.observable.IChangeListener;
import org.eclipse.core.databinding.observable.IObservable;
import org.eclipse.emf.databinding.FeaturePath;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.papyrus.infra.emf.providers.EMFGraphicalContentProvider;
import org.eclipse.papyrus.infra.emf.providers.EMFLabelProvider;
import org.eclipse.papyrus.infra.emf.utils.HistoryUtil;
import org.eclipse.papyrus.infra.emf.utils.ProviderHelper;
import org.eclipse.papyrus.infra.widgets.creation.ReferenceValueFactory;
import org.eclipse.papyrus.infra.widgets.providers.IStaticContentProvider;
import org.eclipse.papyrus.infra.widgets.providers.StaticContentProvider;
import org.eclipse.papyrus.uml.properties.creation.UMLPropertyEditorFactory;
import org.eclipse.papyrus.uml.properties.modelelement.UMLModelElement;
import org.eclipse.papyrus.uml.tools.providers.UMLContainerContentProvider;
import org.eclipse.papyrus.uml.tools.providers.UMLFilteredLabelProvider;
import org.eclipse.papyrus.views.properties.providers.FeatureContentProvider;
import org.eclipse.papyrusrt.umlrt.core.utils.IRealTimeConstants;
import org.eclipse.papyrusrt.umlrt.core.utils.MessageSetUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.CapsulePart;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTMessageKind;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTMessageSet;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTPort;
import org.eclipse.papyrusrt.umlrt.tooling.ui.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.ui.internal.modelelement.ChangeListenerUtils;
import org.eclipse.papyrusrt.umlrt.tooling.ui.widgets.CapsulePartExtendedObservableValue;
import org.eclipse.papyrusrt.umlrt.tooling.ui.widgets.CapsulePartTypeValueFactory;
import org.eclipse.papyrusrt.umlrt.tooling.ui.widgets.MessageSetOwnedProtocolMessageValueFactory;
import org.eclipse.papyrusrt.umlrt.tooling.ui.widgets.RTPortTypeValueFactory;
import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.DirectedRelationship;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * A UMLRTModelElement provider. In particular, it will take care of UMLRT protocols which reference provided, required and prov/required interfaces.
 * These can not be specified by means of a property path, since they depend on implemented or used interfaces which are not directly provided.
 * The idea of this class is to delegate to UMLModelElement belonging to these interfaces
 */
public class UMLRTExtModelElement extends UMLModelElement {

	private Hashtable<Element, UMLModelElement> delegationModelElements;

	protected static final String OWNED_OPERATION = "ownedOperation"; //$NON-NLS-1$

	/**
	 * Constructor.
	 */
	public UMLRTExtModelElement(EObject source) {
		super(source, TransactionUtil.getEditingDomain(source));
		delegationModelElements = new Hashtable<Element, UMLModelElement>();
	}

	/**
	 * Constructor.
	 */
	public UMLRTExtModelElement(EObject source, EditingDomain domain) {
		super(source, domain);
		delegationModelElements = new Hashtable<Element, UMLModelElement>();
	}

	public class UMLRTExtModelElementChangeListener implements IChangeListener {

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void handleChange(ChangeEvent event) {
			try {
				ChangeListenerUtils.fireDataSourceChanged(dataSource);
			} catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException | RuntimeException e) {
				Activator.log.debug(e.getMessage());
			}
		}
	}

	/**
	 * Get the delegating model element
	 * 
	 * @param element
	 * @return
	 */
	public UMLModelElement getDelegationModelElement(Element element) {
		UMLModelElement delegationModelElement = delegationModelElements.get(element);
		if (delegationModelElement == null) {
			delegationModelElement = new UMLRTExtModelElement(element, this.getDomain());
			delegationModelElements.put(element, delegationModelElement);
		}
		return delegationModelElement;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IStaticContentProvider getContentProvider(String propertyPath) {
		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).getContentProvider(OWNED_OPERATION);
		}
		if (propertyPath.contains(IRealTimeConstants.CAPSULE_PART_MULTIPLICITY)) {

			return new StaticContentProvider(new String[] { ONE, OPTIONAL });

		}

		return super.getContentProvider(propertyPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOrdered(String propertyPath) {
		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).isOrdered(OWNED_OPERATION);
		}
		return super.isOrdered(propertyPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isMandatory(String propertyPath) {
		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).isMandatory(OWNED_OPERATION);
		}
		return super.isMandatory(propertyPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ILabelProvider getLabelProvider(String propertyPath) {
		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).getLabelProvider(OWNED_OPERATION);
		}
		return super.getLabelProvider(propertyPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EStructuralFeature getFeature(String propertyPath) {
		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).getFeature(OWNED_OPERATION);
		}
		return super.getFeature(propertyPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FeaturePath getFeaturePath(String propertyPath) {
		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).getFeaturePath(OWNED_OPERATION);
		}
		return super.getFeaturePath(propertyPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IObservable doGetObservable(String propertyPath) {
		IObservable observable = null;

		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);

		if (intf != null) {
			observable = getDelegationModelElement(intf).doGetObservable(OWNED_OPERATION);
		} else if (IRealTimeConstants.CAPSULE_PART_MULTIPLICITY.equals(propertyPath)) {
			observable = new CapsulePartExtendedObservableValue(source, domain);
		} else {
			observable = super.doGetObservable(propertyPath);
		}

		if (getSource() instanceof RTPort) {
			observable.addChangeListener(new UMLRTExtModelElementChangeListener());
		}

		return observable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ReferenceValueFactory getValueFactory(String propertyPath) {

		Interface intf = getInOrInoutOrOutMessageSet(propertyPath);
		if (intf != null) {
			return getDelegationModelElement(intf).getValueFactory(propertyPath);
		}

		if (propertyPath.endsWith("type")) {
			UMLPropertyEditorFactory factory = null;
			EReference reference = UMLPackage.eINSTANCE.getTypedElement_Type();
			EClass type = reference.getEReferenceType();
			if (source instanceof Port) {
				factory = new RTPortTypeValueFactory(reference);
			} else if (source instanceof Property) {
				factory = new CapsulePartTypeValueFactory(reference);
			} else {
				return super.getValueFactory(propertyPath);
			}

			factory.setContainerLabelProvider(new UMLFilteredLabelProvider());
			factory.setReferenceLabelProvider(new EMFLabelProvider());
			ITreeContentProvider contentProvider = new UMLContainerContentProvider(source, reference);

			ResourceSet rs = source == null ? null : source.eResource() == null ? null : source.eResource().getResourceSet();
			EMFGraphicalContentProvider provider = ProviderHelper.encapsulateProvider(contentProvider, rs, HistoryUtil.getHistoryID(source, reference, "container"));

			factory.setContainerContentProvider(provider);
			factory.setReferenceContentProvider(new FeatureContentProvider(type));
			return factory;
		} else if (propertyPath.endsWith("Incoming") || propertyPath.endsWith("Outgoing") || propertyPath.endsWith("InOut")) {
			UMLPropertyEditorFactory factory = null;
			EReference reference = UMLPackage.eINSTANCE.getInterface_OwnedOperation();
			EClass type = reference.getEReferenceType();
			if (source instanceof Interface && MessageSetUtils.isRTMessageSet((Interface) source)) {
				factory = new MessageSetOwnedProtocolMessageValueFactory(reference);
			} else {
				return super.getValueFactory(propertyPath);
			}
			factory.setContainerLabelProvider(new UMLFilteredLabelProvider());
			factory.setReferenceLabelProvider(new EMFLabelProvider());
			ITreeContentProvider contentProvider = new UMLContainerContentProvider(source, reference);

			ResourceSet rs = source == null ? null : source.eResource() == null ? null : source.eResource().getResourceSet();
			EMFGraphicalContentProvider provider = ProviderHelper.encapsulateProvider(contentProvider, rs, HistoryUtil.getHistoryID(source, reference, "container"));

			factory.setContainerContentProvider(provider);
			factory.setReferenceContentProvider(new FeatureContentProvider(type));
			return factory;
		}

		return super.getValueFactory(propertyPath);
	}

	/**
	 * return the message set (interface) that is providing the IN, the out or the inout list of protocol messages (operation), depending on propertyPath
	 * 
	 * @param propertyPath
	 * @return the message set with the right direction
	 */
	protected Interface getInOrInoutOrOutMessageSet(String propertyPath) {
		Interface result = null;
		if (source instanceof Collaboration) {
			if (propertyPath.endsWith("Incoming")) { //$NON-NLS-1$
				result = getInterface(RTMessageKind.IN);
			} else if (propertyPath.endsWith("Outgoing")) { //$NON-NLS-1$
				result = getInterface(RTMessageKind.OUT);
			} else if (propertyPath.endsWith("InOut")) { //$NON-NLS-1$
				result = getInterface(RTMessageKind.IN_OUT);
			}
		}
		return result;
	}

	@Override
	public boolean forceRefresh(String propertyPath) {
		return true;
	}

	@Override
	protected boolean isFeatureEditable(String propertyPath) {
		boolean editable = false;
		EStructuralFeature feature = getFeature(propertyPath);

		if (source instanceof Port) {
			Port port = (Port) source;

			if (UMLPackage.eINSTANCE.getPort_IsConjugated().equals(feature)) {
				editable = super.isFeatureEditable(propertyPath);
			} else if (IRealTimeConstants.PROTOCOL_TYPE.equals(propertyPath)) {
				editable = super.isFeatureEditable(propertyPath);
			} else if (RTPortUtils.isRTPort(source)) {
				RTPort rtPort = RTPortUtils.getStereotypeApplication(port);
				if (UMLPackage.eINSTANCE.getPort_IsService().equals(feature)) {
					editable = !RTPortUtils.isConnected(port) && port.isBehavior();
				} else if (UMLPackage.eINSTANCE.getPort_IsBehavior().equals(feature)) {
					editable = !(!(rtPort.isWired())
							|| (rtPort.isWired() && !port.isService())
							|| (rtPort.isWired() && port.isService() && !port.isBehavior() && RTPortUtils.isConnectedInside(port)));
					// isBehavior disabled if !iswired || (isWired && !isService) || (isWired && isService && !isBehavior && isConnectedInside)
				}
			} else {
				editable = super.isFeatureEditable(propertyPath);
			}
		}

		// Rules defined by the Client
		else if (source instanceof Property) {
			if (null != UMLUtil.getStereotypeApplication((Element) source, CapsulePart.class)) {
				editable = true;
			} else {
				editable = super.isFeatureEditable(propertyPath);
			}
		} else if (source instanceof Collaboration) {
			editable = super.isFeatureEditable(propertyPath);
		}

		return editable;
	}


	/**
	 * Get the incoming interfaces. Don't use getImplementedInterfaces, since it only captures
	 * the interface realization and not the realization relationship.
	 * 
	 * @return list of required interfaces
	 */
	protected Interface getInterface(RTMessageKind rtMessageKind) {
		Interface result = null;

		Collaboration protocol = (Collaboration) source;
		Iterator<DirectedRelationship> relationshipIterator = protocol.getSourceDirectedRelationships().iterator();
		while (relationshipIterator.hasNext() && (result == null)) {
			DirectedRelationship directedRelation = relationshipIterator.next();
			if (directedRelation instanceof Dependency) { // Realization or Usage
				result = getInterfaceFromDepencies(rtMessageKind, directedRelation);
			}
		}

		return result;
	}


	/**
	 * Get the Interface from the dependencies and Kind of Message.
	 * 
	 * 
	 * @return list of required interfaces
	 */
	protected Interface getInterfaceFromDepencies(RTMessageKind rtMessageKind, DirectedRelationship directedRelation) {
		Interface matchingInterface = null;
		Dependency dependency = (Dependency) directedRelation;
		Iterator<NamedElement> dependencyIterator = dependency.getSuppliers().iterator();

		while (dependencyIterator.hasNext() && (null == matchingInterface)) {
			NamedElement supplier = dependencyIterator.next();
			if (supplier instanceof Interface) {
				Interface interfaceImpl = (Interface) supplier;
				RTMessageSet rtMessageSet = UMLUtil.getStereotypeApplication(interfaceImpl, RTMessageSet.class);
				if ((null != rtMessageSet) && (rtMessageSet.getRtMsgKind() == rtMessageKind)) {
					matchingInterface = (Interface) supplier;

				}
			}
		}
		return matchingInterface;
	}


}
