/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine JANSSENS (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.IObserving;
import org.eclipse.core.databinding.observable.value.ValueDiff;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsulePartKindEnum;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.CapsulePart;
import org.eclipse.uml2.uml.AggregationKind;
import org.eclipse.uml2.uml.MultiplicityElement;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * The Class CapsulePartKindObservableValue.
 */
public class CapsulePartKindObservableValue extends PapyrusObservableValue implements IObserving {

	/** The Constant LABEL_MULTIPLICITY. */
	private static final String LABEL_MULTIPLICITY = "Set Multiplicity";

	/** The Constant LABEL_AGGREGATION. */
	private static final String LABEL_AGGREGATION = "Set Aggregation";

	/** The capsule part element. */
	private Property capsulePartElement;

	/** The aggregation Enum value. */
	private AggregationKind aggregation;

	/** The multiplicity Element. */
	private MultiplicityElement multiplicity;

	/**
	 * Instantiates a new capsule part kind observable value.
	 *
	 * @param RTApplication
	 *            the RT application
	 * @param domain
	 *            the domain
	 */
	public CapsulePartKindObservableValue(final EObject RTApplication, final TransactionalEditingDomain domain) {

		super(UMLUtil.getBaseElement(RTApplication), UMLUtil.getBaseElement(RTApplication).eContainingFeature(), domain);

		if (RTApplication instanceof CapsulePart) {
			capsulePartElement = (Property) UMLUtil.getBaseElement(RTApplication);

			capsulePartElement.eAdapters().add(getListener());
			capsulePartElement.getUpperValue().eAdapters().add(getListener());
			capsulePartElement.getLowerValue().eAdapters().add(getListener());

			setUMLPropertiesValue();
		}
	}



	/**
	 * Sets the uml properties value.
	 */
	protected void setUMLPropertiesValue() {
		if (capsulePartElement instanceof Property) {
			multiplicity = capsulePartElement;
			aggregation = capsulePartElement.getAggregation();
		}
	}

	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#doGetValue()
	 *
	 * @return
	 */
	@Override
	protected Object doGetValue() {
		CapsulePartKindEnum kind = null;
		setUMLPropertiesValue();

		if (multiplicity.getLower() == multiplicity.getUpper()) {
			kind = CapsulePartKindEnum.FIXED;
		} else if (multiplicity.getLower() == 0) {
			if (AggregationKind.COMPOSITE_LITERAL.equals(aggregation)) {
				kind = CapsulePartKindEnum.OPTIONAL;
			} else if (AggregationKind.SHARED_LITERAL.equals(aggregation)) {
				kind = CapsulePartKindEnum.PLUGIN;
			}
		}


		return kind;
	}

	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#getObserved()
	 *
	 * @return
	 */
	@Override
	public Object getObserved() {
		return capsulePartElement;
	}


	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#getValueType()
	 *
	 * @return
	 */
	@Override
	public Object getValueType() {
		return CapsulePartKindEnum.class;
	}

	/**
	 * @see org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue#getCommand(java.lang.Object)
	 *
	 * @param value
	 * @return
	 */
	@Override
	public Command getCommand(Object value) {
		Command command = null;
		if (value instanceof CapsulePartKindEnum) {
			// For Each Kind of Capsule Part, set the property accordingly
			switch ((CapsulePartKindEnum) value) {
			case FIXED:
				command = getCommandForCapsulePart(AggregationKind.COMPOSITE_LITERAL, multiplicity.getUpper(), multiplicity.getUpper());
				break;
			case OPTIONAL:
				command = getCommandForCapsulePart(AggregationKind.COMPOSITE_LITERAL, 0, multiplicity.getUpper());
				break;
			case PLUGIN:
				command = getCommandForCapsulePart(AggregationKind.SHARED_LITERAL, 0, multiplicity.getUpper());
				break;

			default:
				// Fixed Capsule Part by Default
				command = getCommandForCapsulePart(AggregationKind.COMPOSITE_LITERAL, multiplicity.getUpper(), multiplicity.getUpper());
				break;
			}
		}
		return command;
	}

	/**
	 * Gets the command for capsule part.
	 *
	 * @param aggregation
	 *            the aggregation
	 * @param lower
	 *            the lower
	 * @param upper
	 *            the upper
	 * @return the command for capsule part
	 */
	protected Command getCommandForCapsulePart(AggregationKind aggregation, int lower, int upper) {
		CompoundCommand command = new CompoundCommand();
		command.append(getSetAggregationCommand(aggregation));
		command.append(getSetMultiplicityCommand(lower, upper));

		return command;
	}


	/**
	 * Gets the sets the multiplicity command.
	 *
	 * @param lower
	 *            the lower
	 * @param upper
	 *            the upper
	 * @return the sets the multiplicity command
	 */
	protected Command getSetMultiplicityCommand(final int lower, final int upper) {
		RecordingCommand multiplicityCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				capsulePartElement.setLower(lower);
				capsulePartElement.setUpper(upper);
			}

			@Override
			public boolean canExecute() {
				return true;
			}

			@Override
			public String getLabel() {
				return LABEL_MULTIPLICITY;
			}
		};

		return multiplicityCommand;
	}


	/**
	 * Retrieve Listener to put on the Stereotype Application.
	 *
	 * @return the listener
	 */
	protected AdapterImpl getListener() {
		return new AdapterImpl() {

			@Override
			public void notifyChanged(Notification notification) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();

				if ((notifier == capsulePartElement) && (type == Notification.SET)) {
					if (UMLPackage.eINSTANCE.getProperty_Aggregation().equals(feature)) {
						fireDiff(notification);
					}
				} else if ((notifier == capsulePartElement.getLowerValue()) && (type == Notification.SET)) {

					fireDiff(notification);

				} else if ((notifier == capsulePartElement.getUpperValue()) && (type == Notification.SET)) {

					fireDiff(notification);

				}
			}

			private void fireDiff(Notification notification) {
				final ValueDiff diff = Diffs.createValueDiff(notification.getOldValue(), notification.getNewValue());
				getRealm().exec(new Runnable() {
					@Override
					public void run() {
						fireValueChange(diff);
					}
				});
			}
		};
	}

	/**
	 * Gets the sets the aggregation command.
	 *
	 * @param wishedAggregation
	 *            the wished aggregation
	 * @return the sets the aggregation command
	 */
	protected Command getSetAggregationCommand(final AggregationKind wishedAggregation) {
		RecordingCommand aggregationCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				capsulePartElement.setAggregation(wishedAggregation);
			}

			@Override
			public boolean canExecute() {
				return true;
			}

			@Override
			public String getLabel() {
				return LABEL_AGGREGATION;
			}
		};

		return aggregationCommand;
	}



}




