/*******************************************************************************
 * Copyright (c) 2016, 2017 Zeligsoft (2009) Limited, Christian W. Damus, and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  Young-Soo Roh - Initial API and implementation
 *  Christian W. Damus - bug 510315
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.core.utils;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.uml.tools.utils.CustomUMLUtil.StereotypeApplicationHelper;
import org.eclipse.papyrusrt.umlrt.core.defaultlanguage.IDefaultLanguage;
import org.eclipse.papyrusrt.umlrt.core.defaultlanguage.IDefaultLanguageService;
import org.eclipse.papyrusrt.umlrt.profile.statemachine.UMLRTStateMachines.UMLRTStateMachinesPackage;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTGuard;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTTrigger;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * Utility class for handing user code related actions.
 * 
 * @author Young-Soo Roh
 *
 */
public final class CodeSnippetUtils {

	/**
	 * 
	 * Constructor.
	 *
	 */
	private CodeSnippetUtils() {

	}

	/**
	 * Get default language.
	 * 
	 * @param context
	 *            element
	 * @return Default language or null if not found
	 */
	public static IDefaultLanguage getDefaultLanguage(Element context) {
		IDefaultLanguage language = null;

		try {
			IDefaultLanguageService service = ServiceUtilsForEObject.getInstance().getService(IDefaultLanguageService.class, context);
			language = service.getActiveDefaultLanguage(context);
		} catch (ServiceException e) {
			org.eclipse.papyrusrt.umlrt.core.Activator.log.error(e);
		}
		return language;
	}

	/**
	 * Create guard.
	 * 
	 * @param owner
	 *            owner
	 * @return guard
	 */
	@SuppressWarnings("unchecked")
	public static Constraint createGuard(Namespace owner) {
		return createGuard(owner, Collections.EMPTY_LIST);
	}

	/**
	 * Ceate Guard.
	 * 
	 * @param owner
	 *            owner
	 * @param constrainedElements
	 *            constrained elements
	 * @return guard
	 */
	public static Constraint createGuard(Namespace owner, List<Element> constrainedElements) {
		Constraint result = owner.createOwnedRule(null);
		result.getConstrainedElements().addAll(constrainedElements);
		result.setSpecification(createOpaqueExpressionWithDefaultLanguage(owner));
		StereotypeApplicationHelper.getInstance(owner).applyStereotype(result,
				UMLRTStateMachinesPackage.Literals.RT_GUARD);
		return result;
	}

	/**
	 * Get trigger guard.
	 * 
	 * @param trigger
	 *            trigger
	 * @return Trigger guard or null if not found
	 */
	public static Constraint getTriggerGuard(Trigger trigger) {
		Optional<UMLRTTrigger> facade = Optional.ofNullable(UMLRTTrigger.getInstance(trigger));
		return facade.map(UMLRTTrigger::getGuard).map(UMLRTGuard::toUML).orElse(null);
	}

	/**
	 * Create Opauqe Expression with default language.
	 * 
	 * @param context
	 *            context
	 * @return Opaque Expression
	 */
	public static OpaqueExpression createOpaqueExpressionWithDefaultLanguage(Element context) {
		OpaqueExpression result = (OpaqueExpression) ModelUtils.create(context, UMLPackage.Literals.OPAQUE_EXPRESSION);
		result.getBodies().add("");
		IDefaultLanguage language = getDefaultLanguage(context);

		if (language != null) {
			result.getLanguages().add(language.getName());
		}
		return result;
	}


	/**
	 * Create Opauqe Expression with default language.
	 * 
	 * @param context
	 *            context
	 * @return Opaque Expression
	 */
	public static OpaqueBehavior createOpaqueBehaviorWithDefaultLanguage(Element context) {
		OpaqueBehavior result = (OpaqueBehavior) ModelUtils.create(context, UMLPackage.Literals.OPAQUE_BEHAVIOR);
		result.getBodies().add("");
		IDefaultLanguage language = getDefaultLanguage(context);

		if (language != null) {
			result.getLanguages().add(language.getName());
		}
		return result;
	}
}
