/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.editpolicies.stereotype;

import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Set;

import org.eclipse.emf.common.util.ECollections;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.papyrus.infra.tools.util.TypeUtils;
import org.eclipse.papyrus.uml.diagram.stereotype.edition.editpolicies.AppliedStereotypeCommentEditPolicy;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTFactory;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTNamedElement;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Stereotype;

/**
 * Customization of the applied-stereotype comment edit-policy that accounts for
 * inheritance of views from other diagrams by not creating the comment views for
 * the transient views of inherited elements.
 */
public class RTAppliedStereotypeCommentEditPolicy extends AppliedStereotypeCommentEditPolicy {
	private static final Class<?>[] GET_VALUE_SIGNATURE = { Stereotype.class, String.class };
	private static final Class<?>[] SET_VALUE_SIGNATURE = { Stereotype.class, String.class, Object.class };

	private Element realHostElement;
	private Element stereotypelessHostElement;

	public RTAppliedStereotypeCommentEditPolicy() {
		super();
	}

	@Override
	protected Element getUMLElement() {
		IGraphicalEditPart host = (IGraphicalEditPart) getHost();
		Element result = TypeUtils.as(host.resolveSemanticElement(), Element.class);

		if (result instanceof NamedElement) {
			UMLRTNamedElement rtElement = UMLRTFactory.create((NamedElement) result);
			if ((rtElement != null) && rtElement.isVirtualRedefinition()) {
				// Provide a view of the element that has no stereotypes
				if (stereotypelessHostElement != null) {
					// Already have one. Is it the correct one?
					if (realHostElement != result) {
						// Nope, wrong one
						stereotypelessHostElement = null;
					}
				}
				if (stereotypelessHostElement == null) {
					createStereotypelessHostElement(result);
				}
				result = stereotypelessHostElement;
			}
		}

		return result;
	}

	private void createStereotypelessHostElement(Element element) {
		realHostElement = element;
		Class<?>[] interfaces = getAllInterfaces(element.getClass());
		stereotypelessHostElement = (Element) Proxy.newProxyInstance(getClass().getClassLoader(), interfaces,
				(proxy, method, args) -> {
					Object result;

					switch (method.getName()) {
					case "getAppliedStereotypes":
					case "getStereotypeApplications":
						result = ECollections.EMPTY_ELIST;
						break;
					case "getValue":
						if (Arrays.equals(method.getParameterTypes(), GET_VALUE_SIGNATURE)) {
							throw new IllegalArgumentException(String.valueOf(args[0]));
						} else {
							result = method.invoke(element, args);
						}
						break;
					case "setValue":
						if (Arrays.equals(method.getParameterTypes(), SET_VALUE_SIGNATURE)) {
							throw new IllegalArgumentException(String.valueOf(args[0]));
						} else {
							result = method.invoke(element, args);
						}
						break;
					default:
						result = method.invoke(element, args);
						break;
					}

					return result;
				});
	}

	static Class<?>[] getAllInterfaces(Class<?> class_) {
		Set<Class<?>> result = new LinkedHashSet<>();

		for (; (class_ != Object.class) && (class_ != null); class_ = class_.getSuperclass()) {
			result.addAll(Arrays.asList(class_.getInterfaces()));
		}

		return result.toArray(new Class<?>[result.size()]);
	}
}
