/*****************************************************************************
 * Copyright (c) 2017 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.databinding.facade;

import java.util.Arrays;
import java.util.Optional;
import java.util.function.Function;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.value.ValueDiff;
import org.eclipse.core.databinding.property.INativePropertyListener;
import org.eclipse.core.databinding.property.ISimplePropertyListener;
import org.eclipse.core.databinding.property.SimplePropertyEvent;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTCapsule;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTClassifier;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTFactory;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * A property for a {@link UMLRTClassifier}'s general (supertype) feature.
 */
class UMLRTClassifierGeneralProperty<S extends UMLRTClassifier> extends FacadeValueProperty<S, S> {

	UMLRTClassifierGeneralProperty(Class<S> elementType, Function<? super S, ? extends S> accessor) {
		super(elementType, Arrays.asList(UMLPackage.Literals.CLASSIFIER__GENERALIZATION, UMLPackage.Literals.GENERALIZATION__GENERAL), accessor);
	}

	@Override
	public String getPropertyName() {
		return (getValueType() == UMLRTCapsule.class) ? "Superclass" : "Supertype";
	}

	@Override
	S fromModel(Object o) {
		return (o == null)
				? null
				: (o instanceof Generalization)
						? fromModel(((Generalization) o).getGeneral())
						: super.fromModel(UMLRTFactory.create((Classifier) o));
	}

	@Override
	Object toModel(S value) {
		return (value == null) ? null : value.toUML();
	}

	@Override
	public INativePropertyListener<S> adaptListener(ISimplePropertyListener<S, ValueDiff<? extends S>> listener) {
		class NativeListener extends AdapterImpl implements INativePropertyListener<S> {

			private S source;

			@Override
			public void addTo(S source) {
				EObject listOwner = getValueOwner(source);
				listOwner.eAdapters().add(this);
				if (listOwner instanceof Classifier) {
					Classifier classifier = (Classifier) listOwner;
					if (!classifier.getGeneralizations().isEmpty()) {
						classifier.getGeneralizations().get(0).eAdapters().add(this);
					}
				}
				this.source = source;
			}

			@Override
			public void removeFrom(S source) {
				if (source != null) {
					EObject listOwner = getValueOwner(source);
					if (listOwner instanceof Classifier) {
						Classifier classifier = (Classifier) listOwner;
						if (!classifier.getGeneralizations().isEmpty()) {
							classifier.getGeneralizations().get(0).eAdapters().remove(this);
						}
					}
					if (listOwner != null) {
						listOwner.eAdapters().remove(this);
					}
					source = null;
				}
			}

			@Override
			public void notifyChanged(Notification msg) {
				ValueDiff<S> diff = null;

				if (msg.getFeature() == UMLPackage.Literals.GENERALIZATION__GENERAL) {
					switch (msg.getEventType()) {
					case Notification.SET:
					case Notification.UNSET:
					case Notification.RESOLVE:
						diff = Diffs.createValueDiff(
								fromModel(msg.getOldValue()),
								fromModel(msg.getNewValue()));
						break;
					}
				} else if (msg.getFeature() == UMLPackage.Literals.CLASSIFIER__GENERALIZATION) {
					switch (msg.getEventType()) {
					case Notification.ADD:
					case Notification.REMOVE:
					case Notification.SET:
					case Notification.RESOLVE:
						if (msg.getPosition() == 0) {
							Generalization oldValue = (Generalization) msg.getOldValue();
							Generalization newValue = (Generalization) msg.getNewValue();

							diff = Diffs.createValueDiff(fromModel(oldValue), fromModel(newValue));

							if (oldValue != null) {
								oldValue.eAdapters().remove(this);
							}
							if (newValue != null) {
								newValue.eAdapters().add(this);
							}
						}

						break;
					}
				}

				if (diff != null) {
					listener.handleEvent(new SimplePropertyEvent<>(SimplePropertyEvent.CHANGE,
							source, UMLRTClassifierGeneralProperty.this, diff));
				}
			}

		}

		return new NativeListener();
	}

	@Override
	protected ICommand getSetCommand(TransactionalEditingDomain domain, S source, Optional<S> newValue) {
		return new AbstractTransactionalCommand(domain, "Set " + getPropertyName(), null) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
				source.setGeneral(newValue.orElse(null));
				return CommandResult.newOKCommandResult();
			}
		};
	}

}
