/*****************************************************************************
 * Copyright (c) 2016, 2017 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.internal.handlers;

import static org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper.wrap;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.papyrus.infra.tools.util.PlatformHelper;
import org.eclipse.papyrusrt.umlrt.core.commands.ExclusionCommand;
import org.eclipse.ui.commands.IElementUpdater;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.menus.UIElement;
import org.eclipse.uml2.uml.Element;

/**
 * Default handler for the exclude and reinherit commands.
 */
public class ExclusionHandler extends AbstractHandler implements IElementUpdater {

	private final boolean doUpdate;

	/**
	 * Initializes me.
	 */
	public ExclusionHandler() {
		this(false);
	}

	/**
	 * Initializes me with the option of updating the text of the command that I
	 * handle to indicate exclusion, instead.
	 * 
	 * @param doUpdate
	 *            whether to update my command to indicate exclusion semantics
	 */
	public ExclusionHandler(boolean doUpdate) {
		super();

		this.doUpdate = doUpdate;
	}

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		Object result = null;

		IStructuredSelection sel = HandlerUtil.getCurrentStructuredSelection(event);
		List<Element> elements = ((List<?>) sel.toList()).stream()
				.map(e -> PlatformHelper.getAdapter(e, EObject.class))
				.filter(Element.class::isInstance).map(Element.class::cast)
				.collect(Collectors.toList());
		if (!elements.isEmpty()) {
			TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(elements.get(0));
			if (domain != null) {
				boolean exclude = event.getCommand().getId().contains("exclude") //$NON-NLS-1$
						|| event.getCommand().getId().contains("delete"); //$NON-NLS-1$

				Command command = wrap(ExclusionCommand.getExclusionCommand(domain, elements, exclude));
				domain.getCommandStack().execute(command);
				result = command.getResult().isEmpty()
						? null
						: command.getResult().iterator().next();
			}
		}

		return result;
	}

	@Override
	public void updateElement(UIElement element, @SuppressWarnings("rawtypes") Map parameters) {
		if (doUpdate) {
			element.setText("Exclude Element");
			element.setTooltip("Undefine the inherited element in the inheriting namespace");
		}
	}
}
