/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.dialogs;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ResourceSelectionDialog;

import org.eclipse.pop.ssme.compilation.Scenario;

/**
 * Class describing the dialog to find the scenario to execute
 */
public class FindCompilationScenarioDialog extends Dialog {
	/** Dialog area */
	private Composite	dialog;
	/** Full path of the selected file */
	private Text		scenarioTxt;
	/** Button to find the scenario */
	private Button		scenarioBtn;
	/** Scenario to execute */
	private Scenario	scenario;
	/** Comment if you select no or more than one file */
	private Label		comment;
	
	/**
	 * Create the Dialog window that handles the string editing
	 * 
	 * @param parentShell
	 *        the parent Shell
	 */
	public FindCompilationScenarioDialog(Shell parentShell) {
		super(parentShell);
		setBlockOnOpen(true);
		setShellStyle(getShellStyle() | SWT.RESIZE);
		scenario = null;
	}
	
	/**
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	protected void configureShell(Shell newShell) {
		newShell.setText("Select the compilation scenario...");
		newShell.setMinimumSize(400, 200);
		
		super.configureShell(newShell);
	}
	
	/**
	 * Create the Dialog area
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent) {
		dialog = (Composite) super.createDialogArea(parent);
		dialog.setLayout(new GridLayout(3, false));
		dialog.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// Selection of the scenario file
		Label modelLbl = new Label(dialog, SWT.NONE);
		modelLbl.setText("Scenario file : ");
		
		scenarioTxt = new Text(dialog, SWT.BORDER);
		scenarioTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		scenarioTxt.setEnabled(false);
		
		scenarioBtn = new Button(dialog, SWT.PUSH);
		scenarioBtn.setText(ELLIPSIS);
		scenarioBtn.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent event) {
				selectFile();
			}
			
			public void widgetDefaultSelected(SelectionEvent event) {
				selectFile();
			}
		});
		
		comment = new Label(dialog, SWT.NONE);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.horizontalSpan = 3;
		comment.setLayoutData(data);
		
		return dialog;
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.Dialog#cancelPressed()
	 */
	@Override
	protected void cancelPressed() {
		scenario = null;
		
		super.cancelPressed();
	}
	
	/**
	 * Returns the selected scenario, or null if no file are selected or the selected file does not contain a scenario
	 * 
	 * @return a compilation scenario or null.
	 */
	public Scenario getScenario() {
		return scenario;
	}
	
	/**
	 * Method describing the selection of the scenario file in the dialog
	 */
	private void selectFile() {
		// The OK button will be enabled only when a correct file is selected.
		getButton(IDialogConstants.OK_ID).setEnabled(false);
		ResourceSelectionDialog dialog = new ResourceSelectionDialog(getShell(), ResourcesPlugin.getWorkspace()
				.getRoot(), "Choose a compilation scenario file :");
		if (dialog.open() == Window.OK) {
			Object[] results = dialog.getResult();
			if (results.length == 1 && results[0] instanceof IFile) {
				scenarioTxt.setText(((IFile) results[0]).getFullPath().toString());
				checkFile((IFile) results[0]);
			}
			else {
				comment.setText("Select one and only file");
				getButton(IDialogConstants.OK_ID).setEnabled(false);
			}
		}
	}
	
	/**
	 * This method checks if the selected file is a valid scenario
	 * 
	 * @param file
	 *        the selected file supposed to be a scenario
	 */
	private void checkFile(IFile file) {
		scenario = null;
		ResourceSet resourceSet = new ResourceSetImpl();
		URI fileURI = URI.createFileURI(file.getLocation().toOSString());
		Resource resource = resourceSet.getResource(fileURI, true);
		
		for (EObject obj : resource.getContents()) {
			if (obj instanceof Scenario)
				scenario = (Scenario) obj;
		}
		
		if (scenario == null) {
			comment.setText("The selected file does not contain a compilation scenario.");
			getButton(IDialogConstants.OK_ID).setEnabled(false);
		}
		else {
			comment.setText("You select the scenario named : " + scenario.getDescription());
			getButton(IDialogConstants.OK_ID).setEnabled(true);
		}
	}
}
