/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.jobs;

import java.io.IOException;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.pop.ssme.compilation.utils.CompilationUtilsPlugin;
import org.eclipse.pop.ssme.compilation.utils.popup.actions.CompilationUtils;

import org.eclipse.pop.ssme.RootModel;
import org.eclipse.pop.ssme.compilation.Functionality;
import org.eclipse.pop.ssme.compilation.FunctionalityKind;
import org.eclipse.pop.ssme.compilation.Generator;
import org.eclipse.pop.ssme.compilation.GeneratorKind;
import org.eclipse.pop.ssme.compilation.Scenario;
import org.eclipse.pop.ssme.compilation.ScenarioItem;
import org.eclipse.pop.ssme.polychrony.EnvironmentConstants;
import org.eclipse.pop.ssme.polychrony.ExitStatus;
import org.eclipse.pop.ssme.polychrony.GraphValue;
import org.eclipse.pop.ssme.polychrony.PKTrees;
import org.eclipse.pop.ssme.polychrony.PkPlugin;
import org.eclipse.pop.ssme.polychrony.traceability.SSMENode;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyConsole;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyDiagnostic;

public class ExecuteScenarioJob extends Job {
	/** The scenario that will be applied */
	private Scenario			scenario;
	/** The instance of the class containing the methods used to execute the scenario */
	private CompilingMethods	compilingMethods;
	/** The source file : SSME EMF model or SIGNAL textual file */
	private IFile				sourceFile;
	/** Boolean set to true if a clustering is applied */
	private boolean				clustersApplied;
	/** Boolean set to true if an abstraction is applied */
	private boolean				abstractionApplied;
	/** The instance of the class containing compilation utilities */
	private CompilationUtils	compilationUtils;
	
	public ExecuteScenarioJob(IFile file, Scenario scenario, CompilationUtils compilationUtils) {
		super("Execute scenario: " + scenario);
		this.sourceFile = file;
		this.scenario = scenario;
		this.compilationUtils = compilationUtils;
		compilingMethods = new CompilingMethods();
		clustersApplied = false;
		abstractionApplied = false;
		// protect the workspace during the job
		this.setRule(ResourcesPlugin.getWorkspace().getRoot());
	}
	
	/**
	 * Initialization of Polychrony Services
	 */
	private void initialize() {
		compilationUtils.printBanner();
		PolychronyConsole.printInfo("Input file : " + sourceFile.getFullPath() + "\n\n");
		PolychronyDiagnostic.clear();	 
		PkPlugin.getServices().resetMachine();
	}
	
	/**
	 * Method describing the end of a Polychrony operation
	 * 
	 * @param monitor
	 *        the object that monitors the progress of the compilation
	 * @param status
	 *        the object representing the outcome of the precedent operation
	 * @return the final status of the operation
	 */
	private IStatus finalize(IProgressMonitor monitor, IStatus status) {
		MultiStatus mStatus = new MultiStatus(CompilationUtilsPlugin.PLUGIN_ID, 0, "Execute scenario action status",
			null);
		mStatus.add(status);
		PkPlugin.getServices().setEnv(EnvironmentConstants.LIBRARY_PATH, compilationUtils.getSaveLibraryPath());
		if (!monitor.isCanceled()) {
			monitor.setTaskName("Markers creation");
			// creates the markers from the validation results if there is errors or warnings
			final Diagnostic diagnostic = PolychronyDiagnostic.getUniqueDiagnostics();
			if (sourceFile != null) {
				compilationUtils.refreshErrors(sourceFile);
				if (diagnostic.getSeverity() == Diagnostic.ERROR) {
					mStatus.add(compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_GENERATION));
				}
			}
		}
		monitor.done();
		PkPlugin.getServices().resetMachine();
		return mStatus;
	}
	
	@Override
	protected IStatus run(IProgressMonitor monitor) {
		try {
			compilationUtils.setSaveLibraryPath(PkPlugin.getServices().getEnv(EnvironmentConstants.LIBRARY_PATH));
			
			if (sourceFile == null) return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_NO_FILE_SELECTED));
			 
			if (scenario == null) return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_NO_SCENARIO_SELECTED));
			 
			
			// We get the extension of the input file
			boolean isSignalTextualFile = (sourceFile.getFileExtension().equalsIgnoreCase(CompilationUtils.EXTENSION_SIGNAL_TEXTUAL_FILE));
			boolean isSignalEMFModel = (sourceFile.getFileExtension().equalsIgnoreCase(CompilationUtils.EXTENSION_SIGNAL_EMF_MODEL));
			boolean isSignalTextOrEMFModel = ((isSignalTextualFile) || (isSignalEMFModel));
			
			if (!isSignalTextOrEMFModel)
				return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_FILE_EXTENSION));
			 
			
			initialize();
			
			monitor.beginTask("Compilation scenario execution", 100);
			
			// We get the input file
			IPath fileLocation = sourceFile.getLocation();
			IPath fileDirectory = fileLocation.removeFileExtension();
			
			int monitorStep = 0;
			long tree = PKTrees.ERROR_NODE;
			
			if (isSignalEMFModel) {
				/*
				 * Gets the root Model Element : there must be only one root element, and its type must be
				 * ModelDeclaration or Module.
				 */
				ResourceSet resourceSet = new ResourceSetImpl();
				URI fileURI = URI.createFileURI(fileLocation.toOSString());
				Resource resource = resourceSet.getResource(fileURI, true);
				
				if (resource.getContents().size() != 1)  
					return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_MORE_THAN_ONE_ROOT_ELEMENT));
				 
				
				RootModel rootModel = null;
				EObject obj = resource.getContents().get(0);
				
				if (!(obj instanceof RootModel)) 
					return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_NO_ROOT_MODEL));
				 
				
				rootModel = (RootModel) obj;
				
				
				// Checks if there is no EMF validation error on the model
				monitor.subTask("Model validation");
				if (!compilationUtils.validModel(resource))  
					return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_INVALID_MODEL));
				 
				monitor.worked(10);
				
				// Creates the AST from the root model element
				monitor.subTask("AST creation");
				// the interface hashmap between SSME and SIGNAL is cleared before every new execution
				PkPlugin.getTraceabilityAPI().clearMap();
				tree = rootModel.makeAST();
				PkPlugin.getTraceabilityAPI().initializeSSMENodesList();
				PkPlugin.getTraceabilityAPI().printInformations(false);
				PkPlugin.getTraceabilityAPI().printTree(tree, 0, false);
				
				monitorStep = 10;
			}
			else if (isSignalTextualFile) {
				// Create the AST from the SIG file
				monitor.subTask("AST creation");
				tree = PkPlugin.getParser().parseSrcfile(fileLocation.toOSString());
				monitorStep = 20;
			}
			
			if (tree == PKTrees.ERROR_NODE) 
				return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_TREE));
			 
			monitor.worked(monitorStep);
			
			// We create a boolean to know when to generate the graph or not.
			boolean generateGraph = false;
			
			if (!scenario.getElements().isEmpty()) {
				for (ScenarioItem scit1 : scenario.getElements()) {
					if (scit1 instanceof Generator) {
						Generator generator = (Generator) scit1;
						int generationType = generator.getKind().getValue();
						
						/*
						 * If the scenario contains only "SSME Generation" or "Signal Generation", we don't need the
						 * graph : but if the scenario contains another generator, for example "C Code", the boolean
						 * generateGraph needs to be set to true.
						 */

						boolean graphGenerationNeeded = ((generationType != GeneratorKind.SSME) && (generationType != GeneratorKind.SIGNAL));
						if (graphGenerationNeeded) generateGraph = true;
					}
				}
			}
			// if (!isSignalTextualFile) compilingMethods.generateSignal(tree, fileDirectory);
			compilationUtils.addPathSignalLibraryPath(fileDirectory);
			GraphValue graphValue = null;
			if (generateGraph) {
				// System.out.println("----> Calling tree2DCG (ExecuteScenario)");
				monitor.subTask("Tree to Graph transformation");
				graphValue = PkPlugin.getServices().tree2DCG(tree, true, PkPlugin.getTreeAPI().getNilTree(), true,
						fileDirectory.toOSString(), false, isSignalEMFModel);
				if (! graphValue.isOk()) {
					PkPlugin.getTraceabilityAPI().printErrorsOnModel(tree, false);
					return finalize(monitor, compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_GRAPH
							+ ExitStatus.get(graphValue.getReturnCode()).getLiteral()));
				}
				monitor.worked(10);
			}
			MultiStatus status = new MultiStatus(CompilationUtilsPlugin.PLUGIN_ID, 0, "Scenario status", null);
			if (!scenario.getElements().isEmpty()) {
				// Displays or not the warnings
				PkPlugin.getServices().displayWarnings(scenario.isDisplayWarnings());
				
				IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 60);
				subMonitor.beginTask("Scenario execution", scenario.getElements().size() * 10);
				for (ScenarioItem scit : scenario.getElements()) {
					subMonitor.subTask(scit.toString());
					if (scit instanceof Functionality) // case of a functionality (ex : clock calculus)
					{
						Functionality func = (Functionality) scit;
						// Stop the scenario when a functionality execution returns false
						IStatus res = executeFunctionality(func.getKind(), graphValue, tree);
						status.add(res);
						if (!res.isOK()) {
							break;
						}
					}
					else if (scit instanceof Generator) // case of a generator (ex : generate C)
					{
						Generator gen = (Generator) scit;
						if (gen.getKind().getValue() == GeneratorKind.SSME
								|| gen.getKind().getValue() == GeneratorKind.SIGNAL
								|| gen.getKind().getValue() == GeneratorKind.SIGNAL_LIS) {
							/*
							 * to generate the SSME file or the SIGNAL and SIGNAL LIS textual forms, we need the abstract
							 * syntax tree. We put 0 as graph argument because in these 3 cases (SSME, Signal & Signal
							 * LIS) we don't need the graph.
							 */
							status.add(executeGenerator(gen.getKind(), 0, tree, fileDirectory));
						}
						else {
							/*
							 * for other generation commands (for example C or C++), we need the graph. We put 0 as tree
							 * argument because we don't need the tree in these cases.
							 */
							status.add(executeGenerator(gen.getKind(), graphValue.getGraph(), 0, fileDirectory));
						}
					}
					else {
						status.add(compilingMethods.getErrorCompilationStatus(CompilationUtils.ERROR_NOT_IMPLEMENTED
								+ scit.toString()));
					}
					subMonitor.worked(10);
				}
				subMonitor.done();
			}
			return finalize(monitor, status);
		}
		catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	/**
	 * Method executing a given functionality (ex : clock calculus)
	 * 
	 * @param kind
	 *        the kind of the functionality
	 * @param graph
	 *        the graph generated from the abstract tree
	 * @param tree
	 *        the abstract syntax tree
	 * @return the status of the operation
	 * @throws IOException
	 */
	private IStatus executeFunctionality(FunctionalityKind kind, GraphValue graph, long tree) throws IOException {
		IStatus returnStatus = Status.OK_STATUS;
		int functionality = kind.getValue();
		
		if (functionality == FunctionalityKind.ABSTRACTION) {
			returnStatus = compilingMethods.abstraction(graph);
			abstractionApplied = returnStatus.isOK();
		}
		else if (functionality == FunctionalityKind.BOOLEAN2_EVENTS) {
			returnStatus = compilingMethods.boolean2Events(graph);
		}
		else if (functionality == FunctionalityKind.CLOCK_CALCULUS) {
			returnStatus = compilingMethods.clockCalculus(graph);
		}
		else if (functionality == FunctionalityKind.CLUSTERING) {
			returnStatus = compilingMethods.cluster(graph);
			clustersApplied = returnStatus.isOK();
		}
		else if (functionality == FunctionalityKind.EVENTS2_BOOLEAN) {
			returnStatus = compilingMethods.DCGPoly2DCGBool(graph);
		}
		else if (functionality == FunctionalityKind.FLATTENING) {
			returnStatus = compilingMethods.Flattening(graph);
		}
		else if (functionality == FunctionalityKind.RETIMING) {
			returnStatus = compilingMethods.retiming(graph);
		}
		else if (functionality == FunctionalityKind.SEQUENTIALIZING) {
			returnStatus = compilingMethods.sequentializing(graph, tree, clustersApplied);
		}
		else if (functionality == FunctionalityKind.SIGNAL_UNIFICATIONS) {
			returnStatus = compilingMethods.unify(graph);
		}
		return returnStatus;
	}
	
	/**
	 * Method used to generate code (ex : C, C++...)
	 * 
	 * @param kind
	 *        the kind of the generator
	 * @param graph
	 *        the graph generated from the abstract tree
	 * @param tree
	 *        tree the abstract syntax tree
	 * @param fileDir
	 *        the path of the source file
	 * @throws IOException
	 */
	private IStatus executeGenerator(GeneratorKind kind, long graph, long tree, IPath fileDir) throws IOException {
		IStatus returnStatus = Status.OK_STATUS;
		int generator = kind.getValue();
		
		if (generator == GeneratorKind.CANSI) {
			returnStatus = compilingMethods.generateC(graph, abstractionApplied);
		}
		else if (generator == GeneratorKind.CPP) {
			returnStatus = compilingMethods.generateCPP(graph, abstractionApplied);
		}
		else if (generator == GeneratorKind.JAVA) {
			returnStatus = compilingMethods.generateJava(graph, fileDir,
					abstractionApplied, compilationUtils);
		}
		else if (generator == GeneratorKind.LUSTRE) {
			returnStatus = compilingMethods.generateLustre(graph);
		}
		else if (generator == GeneratorKind.PROFILING) {
			returnStatus = compilingMethods.generateProfiling(graph);
		}
		else if (generator == GeneratorKind.SIGALI) {
			returnStatus = compilingMethods.generateSigali(graph);
		}
		else if (generator == GeneratorKind.SIGNAL) {
			returnStatus = compilingMethods.generateSignal(tree, fileDir);
		}
		else if (generator == GeneratorKind.SIGNAL_ABSTRACTION) {
			returnStatus = compilingMethods.generateSignalAbstraction(graph);
		}
		else if (generator == GeneratorKind.SSME) {
			returnStatus = compilingMethods.generateSSME(tree, fileDir);
		}
		else if (generator == GeneratorKind.SME) {
			returnStatus = compilingMethods.generateSSME(tree, fileDir);
		}
		else if (generator == GeneratorKind.SYNDEX) {
			returnStatus = compilingMethods.generateSyndex(graph);
		}
		else if (generator == GeneratorKind.SIGNAL_LIS) {
			returnStatus = compilingMethods.generateSignalLIS(tree);
		}
		else if (generator == GeneratorKind.SIGNAL_TRA) {
			returnStatus = compilingMethods.generateSignalTRA(graph);
		}
		return returnStatus;
	}
	
	/**
	 * This method allows access to the SSME nodes list in an extern plugin
	 * 
	 * @return The SSME/Signal interface Map of the current execution
	 */
	public List<SSMENode> getSSMENodesList() {
		return PkPlugin.getTraceabilityAPI().getSMENodesList();
	}
}
