/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.popup.actions;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.validation.model.EvaluationMode;
import org.eclipse.emf.validation.service.IBatchValidator;
import org.eclipse.emf.validation.service.ModelValidationService;
import org.eclipse.pop.ssme.compilation.utils.CompilationUtilsPlugin;
import org.eclipse.swt.widgets.Display;
import org.eclipse.pop.ssme.compilation.Scenario;
import org.eclipse.pop.ssme.polychrony.EnvironmentConstants;
import org.eclipse.pop.ssme.polychrony.PkPlugin;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyConsole;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyDiagnostic;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyMarkerHelper;

/**
 * This class contains useful methods that can eventually be called from other plugins
 */
public class CompilationUtils {
	public static final String EXTENSION_SIGNAL_EMF_MODEL				= "SSME";
	public static final String EXTENSION_SIGNAL_TEXTUAL_FILE			= "SIG";
	public static final String ERROR_SIGNALTOOLBOX_NOT_LOADED		= "Cannot access to the Signal ToolBox (not loaded or bad version).";
	public static final String ERROR_GENERATION						= "Some errors has been detected during the generation. Check the Problem view and correct them.";
	public static final String ERROR_SCENARIO							= "Execute scenario error";
	public static final String ERROR_CODE_GENERATION					= "Code Generation error";
	public static final String ERROR_NO_FILE_SELECTED					= "No file selected";
	public static final String ERROR_NO_SCENARIO_SELECTED				= "No compilation scenario selected";
	public static final String ERROR_FILE_EXTENSION					= "The file is neither a Signal model nor a Signal textual file.";
	public static final String ERROR_MORE_THAN_ONE_ROOT_ELEMENT		= "The sme file must only have one single root element.";
	public static final String ERROR_NO_ROOT_MODEL						= "No root model was found in the Signal model file.";
	public static final String ERROR_INVALID_MODEL						= "The model is not valid. Correct these errors before calling compiler functionalities.";
	public static final String ERROR_TREE								= "Error during the creation of the abstract syntax tree: too many errors to correct before compiling.";
	public static final String ERROR_GRAPH								= "Error during the graph transformation: ";
	public static final String ERROR_NOT_IMPLEMENTED					= "Feature not yet implemented:";
	public static final String ERROR_ABSTRACTION						= "Error during the abstraction building step";
	public static final String ERROR_BOOLEAN_TO_EVENTS					= "Error during the boolean to events transformation step";
	public static final String ERROR_CLOCK_CALCULUS					= "Error during the clock calculus step";
	public static final String ERROR_BOOLEANIZING						= "Error during the booleanizing operation";
	public static final String ERROR_CYCLE_DETECTION					= "Error during the cycle detection operation";
	public static final String ERROR_EVENTS_TO_BOOLEANS				= "Error during the events to booleans pruning operation";
	public static final String ERROR_FLATTENING						= "Error during the flattening step";
	public static final String ERROR_RETIMING							= "Error during the retiming step";
	public static final String ERROR_SEQUENTIALIZING					= "Error during the sequentializing operation";
	public static final String ERROR_SIGNAL_UNIFICATION				= "Error during the signal unification step";
	public static final String ERROR_C_GENERATION						= "Error during the C code generation";
	public static final String ERROR_CPP_GENERATION					= "Error during the CPP code generation";
	public static final String ERROR_JAVA_GENERATION					= "Error during the Java code generation";
	public static final String ERROR_LUSTRE_GENERATION					= "Error during the Lustre code generation";
	public static final String ERROR_PROFILING							= "Error during the profiling";
	public static final String ERROR_SIGALI_GENERATION					= "Error during the Sigali code generation";
	public static final String ERROR_SIGNAL_GENERATION					= "Error during the Signal code generation";
	public static final String ERROR_SIGNAL_ABSTRACTION_GENERATION		= "Error during the Signal Abstraction generation";
	public static final String ERROR_SIGNAL_MODEL_GENERATION			= "Error during the generation of the Signal model file.";
	public static final String ERROR_SYNDEX_GENERATION					= "Error during the SynDEx code generation";
	public static final String ERROR_DELAY_TREE_GENERATION				= "Error during the delay Tree generation";
	public static final String ERROR_EVENTS_TO_BOOLEANS_BOOLEANIZING	= "Error during the events to Booleans Booleanizing operation";
	public static final String ERROR_CLOCK_ANALYSIS_ARRAYS_ITERATE		= "Error during the analysis of the clocks of the arrays/iterates";
	public static final String WARNING_CLUSTERING_TITLE				    = "Force the clustering ?";
	public static final String WARNING_CLUSTERING_TEXT					= "There are some unresolved constraints. Do you want to force the clustering operation ?";
	public static final String WARNING_SEQUENTIALIZING_TITLE			= "Force the sequentializing ?";
	public static final String WARNING_SEQUENTIALIZING_TEXT			    = "There are some unresolved constraints. Do you want to force the sequentializing operation ?";
	public static final String INFO_WAITED_VZERSION                     =  "Waited Signal ToolBox version = ";
	public static final String INFO_SIGNALTOOLBOX_NOT_LOADED			= "See Help-> Help contents --> Pop documentation->user guide";
	
	/** A String containing the paths from the library path */
	private String				saveLibraryPath;
	
	/** A List containing the resources to refresh */
	private List<IResource>		resources;
	
	public String getSaveLibraryPath() {
		return saveLibraryPath;
	}
	
	public void setSaveLibraryPath(String saveLibraryPath) {
		this.saveLibraryPath = saveLibraryPath;
	}
	
	public List<IResource> getResources() {
		return resources;
	}
	
	public void setResources(List<IResource> resources) {
		this.resources = resources;
	}
	
	/**
	 * This method refreshes the contents of the resources List.
	 */
	public void refreshContent() {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				try {
					for (IResource res : resources) {
						res.refreshLocal(IResource.DEPTH_INFINITE, null);
					}
				}
				catch (CoreException ce) {
					CompilationUtilsPlugin.getDefault().getLog().log(
						new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, 0, "Refreshing error.", ce));
				}
			}
		});
	}
	
	/**
	 * This method refreshes errors on a selected file
	 * 
	 * @param file
	 *        the file whose compilation has failed
	 */
	public void refreshErrors(final IFile file) {
		final Diagnostic diagnostic = PolychronyDiagnostic.getUniqueDiagnostics();
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				PolychronyMarkerHelper.INSTANCE.deleteMarkers(file, true, IResource.DEPTH_ZERO);
				try {
					for (IResource res : getResources()) {
						res.refreshLocal(IResource.DEPTH_INFINITE, null);
					}
				}
				catch (CoreException ce) {
					CompilationUtilsPlugin.getDefault().getLog().log(
						new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, 0, "Refreshing error.", ce));
				}
				if (diagnostic.getSeverity() != Diagnostic.OK) {
					for (Diagnostic childDiagnostic : diagnostic.getChildren()) {
						PolychronyMarkerHelper.INSTANCE.createMarkers(file, childDiagnostic);
					}
				}
			}
		});
		PolychronyDiagnostic.clear();
	}
	
	/**
	 * This method returns the Scenario contained in a file
	 * 
	 * @param file
	 *        the file containing the scenario
	 * @return the scenario contained in the file
	 */
	public Scenario checkFile(File file) {
		Scenario scenario = null;
		ResourceSet resourceSet = new ResourceSetImpl();
		URI fileURI = URI.createFileURI(file.getAbsolutePath());
		Resource resource = resourceSet.getResource(fileURI, true);
		for (EObject obj : resource.getContents()) {
			if (obj instanceof Scenario) {
				scenario = (Scenario) obj;
			}
		}
		return scenario;
	}
	
	/**
	 * With this method, we obtain the path of the workspace of a given plugin.
	 * 
	 * @param pluginName
	 *        the name of the plugin whose path we want
	 * @return the path of the workspace of this plugin
	 */
	public String getPluginPath(String pluginName) {
		String returnValue = "";
		URL pluginRootUrl = Platform.getBundle(pluginName).getEntry("/src");
		try {
			returnValue = new File(FileLocator.toFileURL(pluginRootUrl).getFile()).getAbsolutePath();
		}
		catch (IOException e) {
			e.printStackTrace();
		}
		return returnValue;
	}
	
	/**
	 * This method adds the given path to the Signal Library Path
	 * 
	 * @param dir
	 *        the path to add
	 */
	public void addPathSignalLibraryPath(IPath dir) {
		String value = saveLibraryPath;
		if (!("".equals(value)))  value += File.pathSeparator;
		value += dir.toOSString();
		PkPlugin.getServices().setEnv(EnvironmentConstants.LIBRARY_PATH, value);
	}
	
	/**
	 * Validate a model through EMF
	 * 
	 * @param resource
	 *        the resource to validate
	 * @return true if valid
	 */
	public boolean validModel(Resource resource) {
		IBatchValidator validator = (IBatchValidator) ModelValidationService.getInstance().newValidator(EvaluationMode.BATCH);
		validator.setIncludeLiveConstraints(true);
		
		boolean result = true;
		for (EObject obj : resource.getContents()) {
			IStatus status = validator.validate(obj);
			result = status.isOK() && result;
		}
		return result;
	}
	
	/**
	 * Method displaying the version information
	 */
	public void printBanner() {
		PolychronyConsole.printInfo("\n=============================================\n");
	}
}
