/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.popup.actions;

import java.io.File;
import java.util.LinkedList;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.pop.ssme.compilation.utils.dialogs.FindCompilationScenarioDialog;
import org.eclipse.pop.ssme.compilation.utils.jobs.CompilingMethods;
import org.eclipse.pop.ssme.compilation.utils.jobs.ExecuteScenarioJob;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionDelegate;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

import org.eclipse.pop.ssme.RootModel;
import org.eclipse.pop.ssme.compilation.Scenario;
import org.eclipse.pop.ssme.polychrony.EnvironmentConstants;
import org.eclipse.pop.ssme.polychrony.ExitStatus;
import org.eclipse.pop.ssme.polychrony.GraphValue;
import org.eclipse.pop.ssme.polychrony.PKTrees;
import org.eclipse.pop.ssme.polychrony.PkPlugin;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyConsole;

public class GenerateAction implements IObjectActionDelegate {
	
	// The following strings must me the same than in the plugin.xml file.
	private static final String SCENARIO_EXECUTION = "Execute a compilation scenario";
	private static final String C_GENERATION = "Generate C files";
	private static final String CPP_GENERATION = "Generate CPP files";
	private static final String JAVA_GENERATION = "Generate Java files";
	private static final String SIG_GENERATION = "Generate Signal textual file";
	private static final String SIG_LIS_GENERATION = "Generate Signal textual (LIS) file";
	private static final String SIG_TRA_GENERATION = "Generate Signal textual (TRA) file";
	private static final String SIGNAL_MODEL_GENERATION = "Generate Signal model(SSME) file";
	private static final String INTERACTIVE_COMPILATION = "Interactive Compiling";
	private static final String BATCH_COMPILING = "Batch Compiling";
 

	/** The file on which the transformations will be applied */
	private IFile sourceFile;
 
	
	public GenerateAction() {
		super();
	}

	/**
	 * @see IObjectActionDelegate#setActivePart(IAction, IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
	}

	/**
	 * @see IActionDelegate#run(IAction)
	 */
	public void run(IAction action) {
		CompilationUtils compilationUtils = new CompilationUtils();
		String actionID = action.getText();
		compilationUtils.setResources(new LinkedList<IResource>());
	 if	 (! PkPlugin.getServices().isSignalToolBoxLoaded())
		 {
		// The SignalToolBox is not loaded. The compilation is aborted.		
		 	PolychronyConsole.printDetectedError(CompilationUtils.ERROR_SIGNALTOOLBOX_NOT_LOADED);
		 	PolychronyConsole.printInfo(CompilationUtils.INFO_SIGNALTOOLBOX_NOT_LOADED);
		 	PolychronyConsole.printNewLine();
		 	PolychronyConsole.printInfo( CompilationUtils.INFO_WAITED_VZERSION + PkPlugin.getServices().getWaitedVersion());
		 	PolychronyConsole.printNewLine();
		 }
	 else
		 
		if (sourceFile != null) {
			// the directory of the file is added to the list of resources to be refreshed
			compilationUtils.getResources().add(sourceFile.getParent());

			// case of the scenario execution (the .ssc file written in the Eclipse "SME Scenario" tab)
			if (actionID.equals(SCENARIO_EXECUTION)) {
				FindCompilationScenarioDialog dialog = new FindCompilationScenarioDialog(new Shell());
				if (dialog.open() == Window.OK) {
					Scenario scenario = dialog.getScenario();
					if (scenario != null)
						executeScenario(scenario, compilationUtils);
					else
						MessageDialog.openError(new Shell(),CompilationUtils.ERROR_SCENARIO, CompilationUtils.ERROR_NO_SCENARIO_SELECTED);
				}
			} else {
				String projectPath = compilationUtils.getPluginPath("org.eclipse.pop.ssme.compilation.utils");
				projectPath += GenerateAction.class.getResource("").getPath();
				boolean isSimpleScenario = true;

				File file = null;
				if (actionID.equals(C_GENERATION))
					file = new File(projectPath + "CGeneration.ssc");
				else if (actionID.equals(CPP_GENERATION))
					file = new File(projectPath + "CPPGeneration.ssc");
				else if (actionID.equals(JAVA_GENERATION))
					file = new File(projectPath + "JavaGeneration.ssc");
				else if (actionID.equals(SIG_GENERATION))
					file = new File(projectPath + "SIGGeneration.ssc");
				else if (actionID.equals(SIG_LIS_GENERATION))
					file = new File(projectPath + "SIGLISGeneration.ssc");
				else if (actionID.equals(SIG_TRA_GENERATION))
					file = new File(projectPath + "SIGTRAGeneration.ssc");
				else if (actionID.equals(SIGNAL_MODEL_GENERATION))
					file = new File(projectPath + "SSMEGeneration.ssc");
				else
					isSimpleScenario = false;

				// case of a simple scenario (C, C++...)
				if (isSimpleScenario) {
					Scenario scenario = compilationUtils.checkFile(file);
					executeScenario(scenario, compilationUtils);
				}
				// interactive and general compilation
				else if ((actionID.equals(INTERACTIVE_COMPILATION))	|| (actionID.equals(BATCH_COMPILING))   )
				{
					CompilingMethods compilingMethods = new CompilingMethods();
					long tree = 0;
					boolean isSignalTextualFile = (sourceFile.getFileExtension().equalsIgnoreCase(CompilationUtils.EXTENSION_SIGNAL_TEXTUAL_FILE));
					boolean isSignalEMFModel = (sourceFile.getFileExtension().equalsIgnoreCase(CompilationUtils.EXTENSION_SIGNAL_EMF_MODEL));

					compilationUtils.printBanner();
					PolychronyConsole.printInfo("Input file : "	+ sourceFile.getFullPath() + "\n\n");

					// reset of the string memory
					// PkPlugin.getServices().resetMachine();  remove April 2014 (LBesnard).
					compilationUtils.setSaveLibraryPath(PkPlugin.getServices().getEnv(EnvironmentConstants.LIBRARY_PATH));

					IPath fileLocation = sourceFile.getLocation();
					IPath fileDirectory = fileLocation.removeFileExtension();
					if (isSignalTextualFile)
						tree = PkPlugin.getParser().parseSrcfile(fileLocation.toOSString());
					else if (isSignalEMFModel) 
					{
						ResourceSet resourceSet = new ResourceSetImpl();
						URI fileURI = URI.createFileURI(fileLocation.toOSString());
						Resource resource = resourceSet.getResource(fileURI, true);
						if (resource.getContents().size() == 1) {
							EObject obj = resource.getContents().get(0);
							if (obj instanceof RootModel) // The root of the EMF model
							{
								RootModel rootModel = (RootModel) obj;
								if (compilationUtils.validModel(resource)) {
									// the SSME/Signal interface hashmap is cleared before every new execution
									PkPlugin.getTraceabilityAPI().clearMap();
									tree = rootModel.makeAST();
									PkPlugin.getTraceabilityAPI().initializeSSMENodesList();

									/*
									 * Information printed in the console if the booleans are true. 
									 * 	Please refer to the documentation of these methods.
									 */
									PkPlugin.getTraceabilityAPI().printInformations(false);
									PkPlugin.getTraceabilityAPI().printTree(tree, 0, false);
								} else
									PolychronyConsole.printDetectedError(CompilationUtils.ERROR_INVALID_MODEL);
							} else
								PolychronyConsole.printDetectedError(CompilationUtils.ERROR_NO_ROOT_MODEL);
						} else
							PolychronyConsole.printDetectedError(CompilationUtils.ERROR_MORE_THAN_ONE_ROOT_ELEMENT);
					}

					if (tree != PKTrees.ERROR_NODE) {
						compilingMethods.generateSignal(tree, fileDirectory);
						compilationUtils.addPathSignalLibraryPath(fileDirectory);
						// Tree to Graph transformation
						GraphValue graphValue = PkPlugin.getServices().tree2DCG(tree, true, PkPlugin.getTreeAPI().getNilTree(),true, 
								fileDirectory.toOSString(), false, isSignalEMFModel);
						// If there is no problem in the graph production, the interactive or the batch compiler is called.	 
						if (graphValue.isOk())
						{
							if (actionID.equals(INTERACTIVE_COMPILATION))							 
								 new InteractiveCompiling(compilingMethods, tree,	sourceFile, graphValue, compilationUtils);					 
							else if (actionID.equals(BATCH_COMPILING))							 
								 
								 new BatchCompiling(compilingMethods, tree, sourceFile, graphValue, compilationUtils, fileDirectory);
							compilationUtils.refreshContent();	
						}	
						else
						{
							PkPlugin.getTraceabilityAPI().printErrorsOnModel(tree, false);
							PolychronyConsole.printDetectedError((CompilationUtils.ERROR_GRAPH + ExitStatus.get(graphValue.getReturnCode()).getLiteral()));
							compilationUtils.refreshErrors(sourceFile);
						}	
					} else
						PolychronyConsole.printDetectedError(CompilationUtils.ERROR_TREE);
				}
			}
		} else
			MessageDialog.openError(new Shell(),  CompilationUtils.ERROR_CODE_GENERATION, CompilationUtils.ERROR_NO_FILE_SELECTED);
	}

	
	/**
	 * Method displaying the version information
	 */
	public void printBanner(boolean bannouncepk) {
		PolychronyConsole.printInfo("\n=============================================\n");
		
//		PolychronyConsole.printInfo( "SSME Platform " + SSME_VERSION + " (SIGNAL Toolbox " + SIGNAL_VERSION +")");
//		PolychronyConsole.printInfo("\n=============================================\n\n");
	}
	/**
	 * Method used to start the execution of a scenario
	 * 
	 * @param scenario
	 *            the scenario to execute
	 */
	public void executeScenario(Scenario scenario,
			CompilationUtils compilationUtils) {
		ExecuteScenarioJob execute = new ExecuteScenarioJob(sourceFile,	scenario, compilationUtils);
		execute.setUser(true);
		execute.schedule();
	}

	/**
	 * @see IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		if (selection instanceof StructuredSelection) {
			/*
			 * We know here that there is one and only one source file because
			 * it is specified in the plugin.xml file, at the following line :
			 * enablesFor="1"
			 */
			sourceFile = (IFile) ((StructuredSelection) selection).getFirstElement();
		}
	}
}
