/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony;

/**
 * This file contains all SignalTreeManager functions concerning the build of a Signal Tree. Conversions of type between
 * C and Java
 */
public class SignalTreeBuilting {
	public synchronized native void n_addDeclarationToInterface(long modelNode, long declNode, int where);
	
	/**
	 * Adds a declaration to a model.
	 * 
	 * @param modelNode
	 *        node referring a model (assumed).
	 * @param declNode
	 *        node referring a declaration of a signal/parameter.
	 * @param kind
	 *        indicates the kind of identifier(input, output, or parameter)
	 */
	public synchronized void addDeclarationToInterface(long modelNode, long declNode, IdentifierKind kind) {
		n_addDeclarationToInterface(modelNode, declNode, kind.getValue());
	}
	
	/**
	 * Adds a declaration to a list of declarations. For typed identifier, it searches the existency of such type for
	 * adding, otherwise, it adds at the head of the list.
	 * 
	 * @param listNode
	 *        the current list of declarations
	 * @param declNode
	 *        the declaration node to add
	 * @return the new list of declarations.
	 */
	public synchronized native long addDeclarationToList(long listNode, long declNode);
	
	/**
	 * Adds an empty interface to a process.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @return the interface created node.
	 */
	public synchronized native long addEmptyInterface(long modelNode);
	
	/**
	 * Adds a declaration to the end of the list of declarations.
	 * 
	 * @param listNode
	 *        the current list of declarations
	 * @param declNode
	 *        the declaration node to add
	 * @return the new list of declarations.
	 */
	public synchronized native long addEndDeclarationToList(long listNode, long declNode);
	
	/**
	 * Adds a process expression to a process model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param procExprNode
	 *        node referring to a process expression.
	 */
	public synchronized native void addExpressionOfProcessToaModel(long modelNode, long procExprNode);
	
	/**
	 * Adds an element to a composition list.
	 * 
	 * @param compListNode
	 *        node referring to a composition list.
	 * @param procExprNode
	 *        node referring to a process expression (assumed).
	 * @return a node referring to the new composition list.
	 */
	public synchronized native long addExprToComposition_Post(long compListNode, long procExprNode);
	
	/**
	 * Adds an element at the head of a composition list.
	 * 
	 * @param compListNode
	 *        node referring to a composition list.
	 * @param procExprNode
	 *        node referring to a process expression (assumed).
	 * @return a node referring to the new composition list.
	 */
	public synchronized native long addExprToComposition_Pre(long compListNode, long procExprNode);
	
	/**
	 * Inserts in a list of declarations the declaration of a signal/constant. The unicity is guaranteed.
	 * 
	 * @param declListNode
	 *        node referring to a declaration list.
	 * @param typeNode
	 *        node referring to a type.
	 * @param declNode
	 *        node referring to a signal/constant declaration.
	 */
	public synchronized native void addFlowDeclaration(long declListNode, long typeNode, long declNode);
	
	/**
	 * Declares a list of identifiers as labels in a model process.
	 * 
	 * @param idListNode
	 *        node referring to a list of identifiers.
	 * @param modelNode
	 *        node referring to the SIGNAL abstract tree representation of a process (assumed).
	 */
	public synchronized native void addLabelsInModel(long idListNode, long modelNode);
	
	/**
	 * Adds a "where" area in second child of a model process.
	 * 
	 * @param modelNode
	 *        node referring to a model process (assumed).
	 * @return the created local node
	 */
	public synchronized native long addLocalBlock(long modelNode);
	
	/**
	 * Adds a declaration as local to a model or a module.
	 * 
	 * @param modelNode
	 *        node referring to a model or a module.
	 * @param declNode
	 *        node referring to a declaration.
	 */
	public synchronized native void addLocalDeclaration(long modelNode, long declNode);
	
	/**
	 * Adds a pragma to a model process.
	 * 
	 * @param modelNode
	 *        node referring to a model.
	 * @param pragmaNode
	 *        node referring to a pragma.
	 */
	public synchronized native void addPragma(long modelNode, long pragmaNode);
	
	/**
	 * Adds a process declaration to a model/module.
	 * 
	 * @param modelNode
	 *        node referring to a model or a module.
	 * @param modelDeclNode
	 *        node referring to a model declaration.
	 */
	public synchronized native void addProcessDeclaration(long modelNode, long modelDeclNode);
	
	/**
	 * Adds a specification object to a process model.
	 * 
	 * @param modelNode
	 *        node referring to a model.
	 * @param specNode
	 *        node referring to a specification expression.
	 */
	public synchronized native void addSpecification(long modelNode, long specNode);
	
	/**
	 * @param listNode
	 *        a node referring a list of synchronized signals.
	 * @param exprNode
	 *        a node referring to a SIGNAL expression.
	 * @return a node referring to the new modified list ({list} + {exprNode}).
	 */
	public synchronized native long addSynchroTree(long listNode, long exprNode);
	
	/**
	 * Adds a type declaration to a list of type declarations.
	 * 
	 * @param typeDeclListNode
	 *        node referring to a list of type declarations.
	 * @param typeDeclNode
	 *        node referring to a type declaration.
	 * @return the new list of type declaration.
	 */
	public synchronized native long addTypeToTypes(long typeDeclListNode, long typeDeclNode);
	
	/**
	 * @param node
	 *        a node
	 * @param copyAttr
	 *        boolean indicating if the attributes of the node are also copied.
	 * @return a copy of node with (resp. without) a copy of its attributes when copyattr is true (resp. false).
	 */
	public synchronized native long copy(long node, boolean copyAttr);
	
	/**
	 * Removes the parameters tree of a model.
	 * 
	 * @param modelNode
	 *        node referring to a model.
	 */
	public synchronized native void deleteParameters(long modelNode);
	
	/**
	 * Copy the interface of a model without the specification expressions.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @return a copy of the interface of modelNode without specifications.
	 */
	public synchronized native long duplicateInterfaceWithoutSpec(long modelNode);
	
	/**
	 * Inserts in a destination model a copy of all the local declarations of a source model.
	 * 
	 * @param modelNodeDst
	 *        node referring to a model (assumed).
	 * @param modelNodeSrc
	 *        node referring to a model (assumed).
	 */
	public synchronized native void importDeclarations(long modelNodeDst, long modelNodeSrc);
	
	/**
	 * Adds the declaration of identifiers of a process to another process. Not general : only Type declaration
	 * identifier.
	 * 
	 * @param idListNode
	 *        node referring to the list of (type) identifiers to copy
	 * @param modelNodeSrc
	 *        node referring to a model (assumed).
	 * @param modelNodeDst
	 *        node referring to another model (assumed).
	 */
	public synchronized native void importDeclsFromTo(long idListNode, long modelNodeSrc, long modelNodeDst);
	
	/**
	 * Modifies the body of the process model by adding masking for some signals.
	 * 
	 * @param modelNode
	 *        node referring to a process model (assumed).
	 * @param signalListNode
	 *        node referring to a list of signals (assumed).
	 */
	public synchronized native void maskSignals(long modelNode, long signalListNode);
	
	/**
	 * Appends the declarations of a source process model to the declarations of a destination one.
	 * 
	 * @param modelNodeDst
	 *        node referring to the destination process model(assumed).
	 * @param modelNodeSrc
	 *        node referring to the destination process model(assumed).
	 */
	public synchronized native void mergeDeclarations(long modelNodeDst, long modelNodeSrc);
	
	/**
	 * Adds the body of a source process model to the body of a destination one.
	 * 
	 * @param modelNodeDst
	 *        node referring to the destination process model(assumed).
	 * @param modelNodeSrc
	 *        node referring to the destination process model(assumed).
	 */
	public synchronized native void mergeExprProc(long modelNodeDst, long modelNodeSrc);
	
	/**
	 * Inserts the declarations of a source list of declarations in a destination list of declarations. The lists can
	 * contain models or/and signal/constant declarations. The unicity is not guaranteed.
	 * 
	 * @param declListNodeDst
	 *        node referring to the source list of declarations
	 * @param declListNodeSrc
	 *        node referring to the destination list of declarations
	 */
	public synchronized native void mergeLocalDecl(long declListNodeDst, long declListNodeSrc);
	
	/**
	 * Appends the body (resp. declarations) of a source process model to the body (resp. declarations) of a destination
	 * one .
	 * 
	 * @param modelNodeDst
	 *        node referring to the destination process model(assumed).
	 * @param modelNodeSrc
	 *        node referring to the destination process model(assumed).
	 */
	public synchronized native void mergeProcesses(long modelNodeDst, long modelNodeSrc);
	
	/**
	 * @return a nil tree.
	 */
	public synchronized native long mkAtomNilTree();
	
	/**
	 * Returns a process call to a model. The names of the formal declaration (parameters, intputs, outputs) are taken
	 * as effective names for the call.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @return a process call to a model.
	 */
	public synchronized native long mkCompleteModelCall(long process);
	
	/**
	 * Signal Grammar: (| exprNode |) where declNode end;
	 * 
	 * @param exprNode
	 *        node referring to a process expression.
	 * @param declNode
	 *        node referring to a declaration.
	 * @return an abtract tree representing a Signal confined process model.
	 */
	public synchronized native long mkConfinedExpression(long exprNode, long declNode);
	
	/**
	 * @param cstListNode
	 *        node referring to a list of definitions of constants
	 * @param typeNode
	 *        node referring to a type
	 * @return a node referring to a constant declaration with typeNode as type and cstListNode for list of definitions
	 *         of constants.
	 */
	public synchronized native long mkConstantDeclaration(long cstListNode, long typeNode);
	
	/**
	 * @param name
	 *        identifier for the type declaration
	 * @param description
	 *        SIGNAL expression representing the description of the type (bundle(...), enum(...), etc)
	 * @param vinit
	 *        SIGNAL expr corresponding to the initial value
	 * @return a type declaration description .
	 */
	public synchronized native long mkDclType(long name, long description, long vinit);
	
	/**
	 * Returns a dependence expression:
	 * <ul>
	 * <li>fromNode -> toNode</li>
	 * <li>fromNode -> toNode when clockNode</li>
	 * </ul>
	 * 
	 * @param fromNode
	 *        node referring to an identifier
	 * @param toNode
	 *        node referring to another identifier
	 * @param clockNode
	 *        node referring to a clock used as condition of the dependence
	 * @return a node referring to a dependence expression.
	 */
	public synchronized native long mkDependenceTree(long fromNode, long toNode, long clockNode);
	
	/**
	 * @return a node referring to an empty body tree (empty list of pragma, empty body).
	 */
	public synchronized native long mkEmptyBody();
	
	/**
	 * @return a node referring to an empty confined process.
	 */
	public synchronized native long mkEmptyConfinedExpr();
	
	/**
	 * @return a node referring to an empty string.
	 */
	public synchronized native long mkEmptyString();
	
	/**
	 * @return a node
	 */
	public synchronized native long mkEmptyTree();
	
	public synchronized native long n_mkEquation(long operatorCode, long leftExprNode, long rightExprNode);
	
	/**
	 * Builds an equation.
	 * 
	 * @param partialDef
	 *        true if we want to create a partial definition equation, false otherwise.
	 * @param leftExprNode
	 *        node referring to a SIGNAL expression used as left member of the equation
	 * @param rightExprNode
	 *        node referring to a SIGNAL expression used as right member of the equation
	 * @return a node referring the created equation.
	 */
	public synchronized long mkEquation(boolean partialDef, long leftExprNode, long rightExprNode) {
		if (partialDef) {
			return n_mkEquation(SignalOperators.opdefpartdef, leftExprNode, rightExprNode);
		}
		return n_mkEquation(SignalOperators.opdefsig, leftExprNode, rightExprNode);
	}
	
	public synchronized native long n_mkExternalImplLanguage(int langCode);
	
	/**
	 * @param langCode
	 *        code for a language.
	 * @return a node referring to an external "language".
	 */
	public synchronized long mkExternalImplLanguage(LanguageCode langCode) {
		return n_mkExternalImplLanguage(langCode.getValue());
	}
	
	/**
	 * @param valueNode
	 *        a node referring to a value for an external keyword
	 * @return a node referring to an "external" vstring keyword.
	 */
	public synchronized native long mkExternalString(long valueNode);
	
	/**
	 * @param name
	 *        name of the Signal to declare.
	 * @return a node referring to the declaration of name with an implicit type and without initialisation.
	 */
	public synchronized native long mkImplicitSignalDeclaration(String name);
	
	/**
	 * Builds an interface with parameters, input/output signals, and a specification area.
	 * 
	 * @param paramNode
	 *        node referring to the list of parameters.
	 * @param insNode
	 *        node referring to the list of inputs.
	 * @param outsNode
	 *        node referring to the list of outputs.
	 * @param specNode
	 *        node referring to the specification area.
	 * @return a node referring to the created interface.
	 */
	public synchronized native long mkInterface(long paramNode, long insNode, long outsNode, long specNode);
	
	/**
	 * Creates and returns an empty local declarations block of a model.
	 * 
	 * @param modelNode
	 *        a node referring to a model (assumed).
	 * @return a node referring to an empty local declarations block of modelNode.
	 */
	public synchronized native long mkLocalDeclaration(long modelNode);
	
	/**
	 * @param name
	 *        the name of the process model to call.
	 * @return a node referring to the calling process expression to name without parameters.
	 */
	public synchronized native long mkModelCall(String name);
	
	/**
	 * @param nameNode
	 *        a node referring the name of the new module.
	 * @return a node referring to an empty module called name.
	 */
	public synchronized native long mkModule(long nameNode);
	
	/**
	 * @return a node referring to "TYPE_NOT_IMPLEMENTED".
	 */
	public synchronized native long mkNOTIMPLEMENTED();
	
	/**
	 * Makes atomic node with op as operator.
	 * 
	 * @param codeOp
	 *        an operator.
	 * @return a node referring to the op operator.
	 */
	public synchronized native long mkNullaryTree(int codeOp);
	
	/**
	 * Creates a pragma without associated objects.
	 * 
	 * @param id
	 *        the identifier of the pragma.
	 * @param listObjectNode
	 *        node referring to a list of identifiers
	 * @param enonce
	 *        the enonce of the pragma
	 * @return a node referring to a pragma defined by id and enonce
	 */
	public synchronized native long mkPragma(String id, long listObjectNode, String enonce);
	
	/**
	 * Adds an empty pragma area for a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @return the created node (empty list of pragmas).
	 */
	public synchronized native long mkPragmasList(long modelNode);
	
	/**
	 * Makes a process.
	 * 
	 * @param nameNode
	 *        node referring to the name of the process.
	 * @param interfaceNode
	 *        node referring to the description of an interface.
	 * @param descNode
	 *        node referring to the description of a process.
	 * @return a node referring to a process called nameNode, with interface interfaceNode and with descNode as
	 *         description.
	 */
	public synchronized native long mkProcess(long nameNode, long interfaceNode, long descNode);
	
	/**
	 * @param modelIdNode
	 *        node referring to the identifier of a model.
	 * @return a node referring to a process call to modelIdNode.
	 */
	public synchronized native long mkReferenceToModel(long modelIdNode);
	
	/**
	 * @param parListNode
	 *        node referring to a list of parameters.
	 * @param modelNode
	 *        node referring to a model.
	 * @returna a node referring to a process call to modelNode with parListNode as parameters.
	 */
	public synchronized native long mkReferenceToModelWithParameter(long parListNode, long modelNode);
	
	/**
	 * In this case, vt is a copy of <enonce_pragma> value.
	 * 
	 * @param pragListNode
	 *        node referring to a list of pragmas.
	 * @param name
	 *        name of the pragma.
	 * @return a node referring to the enonce of the pragma if the pragma called name exists in the pragListNodetrue,
	 *         PKTrees.ERROR_NODE otherwise.
	 */
	public synchronized native long mkSelectedPragma(long pragListNode, String name);
	
	/**
	 * @param nameNode
	 *        a node referring to the name of the signal.
	 * @param typeNode
	 *        a node referring to a type.
	 * @param initNode
	 *        a node referring to a init SIGNAL expression?
	 * @return a node referring to a signal declaration (typeNode nameNode "init" initNode).
	 */
	public synchronized native long mkSignalDeclaration(long nameNode, long typeNode, long initNode);
	
	public synchronized native long n_mkSkeletonExternProcess(long modelIdNode, long langNode, long opCode);
	
	/**
	 * Makes the skeleton of an external SIGNAL process.
	 * 
	 * @param langNode
	 *        node referring to the denoting language implementation of the external process.
	 * @param kind
	 *        kind of model.
	 * @return a node referring the skeleton.
	 */
	public synchronized long mkSkeletonExternProcess(long modelIdNode, long langNode, ModelKind kind) {
		return n_mkSkeletonExternProcess(modelIdNode, langNode, kind.getValue());
	}
	
	/**
	 * @return a node referring to the skeletton of an interface.
	 */
	public synchronized native long mkSkeletonInterface();
	
	public synchronized native long n_mkSkeletonModel(long id, long opCode);
	
	/**
	 * Making Signal process/module skeleton.
	 * 
	 * @param idNode
	 *        node referring to the identifier used for the new created model/module.
	 * @param kind
	 *        kind of model
	 * @return a node referring the new skeleton of a model/module.
	 */
	public synchronized long mkSkeletonModel(long id, ModelKind kind) {
		return n_mkSkeletonModel(id, kind.getValue());
	}
	
	/**
	 * @param str
	 *        the string
	 * @return a node referring to the str string as value.
	 */
	public synchronized native long mkStringFromStr(String vstr);
	
	/**
	 * @param sigName
	 * @return a node referring to a list of synchronized signals initialized with sigName (this list must be completed
	 *         with at least one other signal).
	 */
	public synchronized native long mkSynchroIni(long sigName);
	
	/**
	 * @return a node referring to an "external" keyword.
	 */
	public synchronized native long mkUndefinedExternalValue();
	
	/**
	 * Moves the rank-th attribute of nodeSrc to nodeDst.
	 * 
	 * @param nodeSrc
	 *        a node
	 * @param nodeDst
	 *        an other node
	 * @param rank
	 *        rank in the list of attributes
	 */
	public synchronized native void moveAttribute(long nodeSrc, long nodeDst, int rank);
	
	/**
	 * Removes the declaration of identifiers from a model. Not general : only Type declaration identifier.
	 * 
	 * @param idListNode
	 *        list of (type) identifiers.
	 * @param modelNode
	 *        node referring to a model (assumed).
	 */
	public synchronized native void removeDeclsFrom(long idListNode, long modelNode);
	
	/**
	 * Removing NIL nodes from node sons of a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @return a node referring the modelNode without NIL sons.
	 */
	public synchronized native long removeNilTree(long modelNode);
	
	/**
	 * Replaces the inputs/outputs of a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param inOutNode
	 *        node containing the new inputs/outputs. The first child are inputs, and the second child are outputs.
	 */
	public synchronized native void replaceInputsOutputs(long modelNode, long inOutNode);
	
	/**
	 * Replaces:
	 * <ul>
	 * <li>the inputs of a model by a list of signal declarations if asInput is true,</li>
	 * <li>the inputs of a model by a list of signal declarations otherwise.</li>
	 * </ul>
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param sigListNode
	 *        node referring to a list of signal declarations.
	 * @param asInput
	 *        true for inputs, false for outputs
	 */
	public synchronized native void replaceInputsOutputsModel(long modelNode, long sigListNode, boolean asInput);
	
	/**
	 * Assigns an interface to a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param newInterfaceNode
	 *        node referring to an interface.
	 */
	public synchronized native void replaceInterface(long modelNode, long newInterfaceNode);
	
	/**
	 * Deletes all the types, constants, signals declarations of the model or module and replaces them by the elements
	 * of a declaration list. The new declarations are inserted at the beginning of the list.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param declListNode
	 *        node referring to a declaration list.
	 */
	public synchronized native void replaceLocalDecl(long modelNode, long declListNode);
	
	/**
	 * Replaces the rank-th sons of node1 by node2. If the rank is equal to 0, node1 is replace by node2.
	 * 
	 * @param node1
	 *        a node.
	 * @param node2
	 *        another node.
	 * @param rank
	 *        rank of the son to replace in node1.
	 */
	public synchronized native void replaceSon(long node1, long node2, int rank);
	
	/**
	 * Assigns the body of a model. The body of the process is replaced by a new expression (vexprpro).
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param bodyNode
	 *        node referring to a body expression (assumed).
	 */
	public synchronized native void setBody(long modelNode, long bodyNode);
	
	/**
	 * Assigns a description to a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param descNode
	 *        node referring to a description of a model (assumed).
	 */
	public synchronized native void setDescription(long modelNode, long descNode);
	
	/**
	 * Assigns the "external graph" of a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param egNode
	 *        node referring to an external graph (assumed).
	 */
	public synchronized native void setExternalGraph(long modelNode, long egNode);
	
	/**
	 * Assigns local declarations to a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param declNode
	 *        node referring to local declarations.
	 * @return a node referring to the previous local declaration of modelNode.
	 */
	public synchronized native long setLocalDeclarations(long modelNode, long declNode);
	
	/**
	 * Assigns a name to a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param nameNode
	 *        node referring to a name (assumed).
	 */
	public synchronized native void setNameModel(long modelNode, long nameNode);
	
	/**
	 * Assigns the parameters of a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param paramNode
	 *        node referring to parameter declarations (assumed).
	 */
	public synchronized native void setParameters(long modelNode, long paramNode);
	
	/**
	 * Assigns the pragmas of a model.
	 * 
	 * @param modelNode
	 *        node referring to a model (assumed).
	 * @param pragmasNode
	 *        node referring to pragmas (assumed).
	 */
	public synchronized native void setPragmas(long modelNode, long pragmasNode);
	
	/**
	 * Verifies the correction of the form of a module.
	 * 
	 * @param moduleNode
	 *        node referring to a module (assumed).
	 * @return the correct form of moduleNode
	 */
	public synchronized native long updateModule(long moduleNode);
	
	/**
	 * It imports to a model (p) the processes declared in the tree b. It is done for the confined expressions of b, not
	 * recursively.
	 * 
	 * @param model
	 *        node referring to a model (assumed).
	 * @param body
	 *        node referring to a body expression (assumed).
	 */
	public synchronized native void moveModelsFrom(long model, long body);
	
}
