/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony.utils;

import java.io.IOException;

import org.eclipse.pop.ssme.polychrony.EnvironmentConstants;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.console.ConsolePlugin;
import org.eclipse.ui.console.IConsole;
import org.eclipse.ui.console.IOConsole;
import org.eclipse.ui.console.IOConsoleOutputStream;

public class PolychronyConsole {
	private final static PolychronyConsole	INSTANCE		= new PolychronyConsole();
	private IOConsole						console			= null;
	private IOConsoleOutputStream			outputStream	= null;
	
	/**
	 * This basic constructor instantiates an IOConsole object, add the console instance to the eclipse console
	 * registry, and display it.
	 * 
	 * @param name
	 */
	private PolychronyConsole() {
		console = new IOConsole("Pop console (SSME version " + EnvironmentConstants.SSME_VERSION + ", Signal Toolbox " + System.getenv ("pK_VERSION") + ")" , null);
		ConsolePlugin.getDefault().getConsoleManager().addConsoles(new IConsole[] { console });
		ConsolePlugin.getDefault().getConsoleManager().showConsoleView(console);
		console.activate();
	}
	
	/**
	 * This is a lazy initialization.
	 * 
	 * @return
	 */
	private IOConsoleOutputStream getOutputStream() {
		if (outputStream == null) outputStream = console.newOutputStream();
		return outputStream;
	}
	
	/**
	 * @param color
	 * @param message
	 */
	public void print(Color color, String message) {
		if (message == null) return;
		if (color != null) changeColor(color);
		
		IOConsoleOutputStream stream = getOutputStream();
		try {
			stream.write(message);
		}
		catch (IOException exception) {
			exception.printStackTrace();
		}
	}
	
	public static void printNewLine() {
		INSTANCE.print(null, "\n");
	}
	
	/**
	 * @param message
	 */
	public static void printInfo(String message) {
		INSTANCE.print(new Color(Display.getCurrent(), 0, 0, 0), message);
	}
	
	/**
	 * @param message
	 */
	public static void printWarning(String message) {
		INSTANCE.print(new Color(Display.getCurrent(), 255, 170, 0), message);
	}
	
	/**
	 * @param message
	 */
	public static void printError(String message) {
		INSTANCE.print(new Color(Display.getCurrent(), 255, 0, 0), message);
	}
	
	/**
	 * @param message
	 */
	public static void printDetectedError(String message) {
		PolychronyConsole.printError("Error detected : "	+ message);
		PolychronyConsole.printNewLine();
	}
	
	/**
	 * Change the color for the following messages (this is because a simple change of current stream color, change the
	 * color for all messages, even previous ones ...)
	 * 
	 * @param c
	 */
	private void changeColor(Color c) {
		Color previousColor = getOutputStream().getColor();
		if (!c.equals(previousColor)) {
			// need to change to another stream for the new color
			outputStream = null; // resets the stream
			getOutputStream().setColor(c);
		}
	}
	
	public static void clear() {
		INSTANCE.console.clearConsole();
	}
	
 
	
	public void dispose() {
		try {
			outputStream.close();
		}
		catch (IOException exception) {
			exception.printStackTrace();
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#finalize()
	 */
	public void finalize() throws Throwable {
		super.finalize();
		if ((outputStream != null) && !outputStream.isClosed())
			outputStream.close();
	}
}
