/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.jobs;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.pop.ssme.compilation.utils.CompilationUtilsPlugin;
import org.eclipse.pop.ssme.compilation.utils.popup.actions.CompilationUtils;
import org.eclipse.pop.ssme.compilation.utils.simulation.CreateJavaProject;
import org.eclipse.swt.widgets.Display;

import org.eclipse.pop.ssme.polychrony.ExitStatus;
import org.eclipse.pop.ssme.polychrony.PkPlugin;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyConsole;
import org.eclipse.pop.ssme.polychrony.GraphValue;

/**
 * This class contains the methods and error messages used in the scenarios and in the interactive compilation function.
 */
public class CompilingMethods {
	
	public CompilingMethods() {
	}
	
	/**
	 * Method returning the error status
	 * 
	 * @param message
	 *        the error message
	 * @return the corresponding status
	 */
	public IStatus getErrorCompilationStatus(String message) {
		return new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, message);
	}
	
	/**
	 * Asking to the user for an operation (force code generation). The question given by the strings S1 and S2. The
	 * boolean answer is stored in the global boolean "generate".
	 * 
	 * @param windowTitle
	 *        the title of the window
	 * @param message
	 *        the message to display
	 * @return true if the user decides to generate (with the constraints)
	 */
	private boolean askUserForGeneration(final String windowTitle, final String message) {
		final List<Boolean> result = new LinkedList<Boolean>();
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				result.add(MessageDialog.openQuestion(Display.getDefault().getActiveShell(), windowTitle, message));
			}
		});
		return result.get(0);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus abstraction(GraphValue graph) {
		if (PkPlugin.getServices().separatedCompilingPoly(graph.getGraph(), true)) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_ABSTRACTION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus boolean2Events(GraphValue graph) {
		if (PkPlugin.getServices().transformBooleanClocks2Events(graph.getGraph())) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_BOOLEAN_TO_EVENTS);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus clockCalculus(GraphValue graph) {
		graph = PkPlugin.getServices().DCGBasic2DCGPoly(graph.getGraph());
		if (PkPlugin.getServices().setIntrinsicClocks(graph.getGraph())) {
			PkPlugin.getServices().nonProductiveSignalElimination(graph.getGraph());
			return Status.OK_STATUS;
		}
		return getErrorCompilationStatus(CompilationUtils.ERROR_CLOCK_CALCULUS);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus cluster(GraphValue graph) {
		PkPlugin.getServices().setIntrinsicClocks(graph.getGraph());
		graph = PkPlugin.getServices().DCGPoly2DCGBool(graph.getGraph());
		if (graph.getReturnCode() == ExitStatus.INIT_ERROR) return getErrorCompilationStatus(CompilationUtils.ERROR_BOOLEANIZING);
		PkPlugin.getServices().nonProductiveSignalElimination(graph.getGraph());
		PkPlugin.getServices().setClockDependences(graph.getGraph());
		boolean generate = !PkPlugin.getServices().hasUnsolvedClockConstraints(graph.getGraph());
		if (!generate) generate = askUserForGeneration(CompilationUtils.WARNING_CLUSTERING_TITLE, CompilationUtils.WARNING_CLUSTERING_TEXT);
		 
		if (!generate) return Status.CANCEL_STATUS;
		if (PkPlugin.getServices().hasStaticCycle(graph.getGraph(), false)) {
			PkPlugin.getServices().nonProductiveSignalElimination(graph.getGraph());
			PkPlugin.getServices().phylumClustering(graph.getGraph());
			PkPlugin.getServices().nonProductiveSignalElimination(graph.getGraph());
			return Status.OK_STATUS;
		}
		return getErrorCompilationStatus(CompilationUtils.ERROR_CYCLE_DETECTION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus DCGPoly2DCGBool(GraphValue graph) {
	   //	PkPlugin.getServices().setIntrinsicClocks(graph.getGraph());
		graph = PkPlugin.getServices().DCGPoly2DCGBool(graph.getGraph());
		if (graph.getReturnCode() == ExitStatus.INIT_ERROR) 
			return getErrorCompilationStatus(CompilationUtils.ERROR_EVENTS_TO_BOOLEANS_BOOLEANIZING);
		if (PkPlugin.getServices().nonProductiveSignalElimination(graph.getGraph()))
			return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_EVENTS_TO_BOOLEANS);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus Flattening(GraphValue graph) {
		graph = PkPlugin.getServices().DCGBool2DCGFlat(graph.getGraph());
		if (graph.getReturnCode() == ExitStatus.NO_ERROR) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_FLATTENING);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus retiming(GraphValue graph) {
		if (PkPlugin.getServices().moveIn2OutDelays(graph.getGraph())) return Status.OK_STATUS;		 
		return getErrorCompilationStatus(CompilationUtils.ERROR_RETIMING);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @param tree
	 *        the abstract syntax tree
	 * @param clustersApplied
	 *        boolean set to true if a clustering is applied
	 * @return the final status of the operation
	 */
	public IStatus sequentializing(GraphValue graph, long tree, boolean clustersApplied) {
		if (!clustersApplied) {
			if (!PkPlugin.getServices().delayTreeBuild(graph.getGraph()))
				return getErrorCompilationStatus(CompilationUtils.ERROR_DELAY_TREE_GENERATION);
			PkPlugin.getServices().nonProductiveSignalElimination(graph.getGraph());
			PkPlugin.getServices().setClockDependences(graph.getGraph());
			boolean generate = !PkPlugin.getServices().hasUnsolvedClockConstraints(graph.getGraph());
			if (!generate)
				generate = askUserForGeneration(CompilationUtils.WARNING_SEQUENTIALIZING_TITLE, CompilationUtils.WARNING_SEQUENTIALIZING_TEXT);
			
			if (!generate) return Status.CANCEL_STATUS;

			if (!PkPlugin.getServices().hasStaticCycle(graph.getGraph(), false)) 
				return getErrorCompilationStatus(CompilationUtils.ERROR_CYCLE_DETECTION);
		}
		
		graph = PkPlugin.getServices().Serializing(graph.getGraph(), true, true, true, true);
		if (graph.getReturnCode() == ExitStatus.NO_ERROR) return Status.OK_STATUS;		 
		return getErrorCompilationStatus(CompilationUtils.ERROR_SEQUENTIALIZING);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus unify(GraphValue graph) {
		if (PkPlugin.getServices().removeSignalClones(graph.getGraph()))  return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_UNIFICATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @param abstractionApplied
	 *        boolean set to true if an abstraction is applied
	 * @return the final status of the operation
	 */
	public IStatus generateC(long graph, boolean abstractionApplied) {
		boolean res;
		if (abstractionApplied)  
			res = PkPlugin.getServices().genCodeCSeparate(graph);
		else  
			res = PkPlugin.getServices().genCodeC(graph, true, true, true);
		if (res)  return Status.OK_STATUS;		 
		return getErrorCompilationStatus(CompilationUtils.ERROR_C_GENERATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @param abstractionApplied
	 *        boolean set to true if an abstraction is applied
	 * @return the final status of the operation
	 */
	public IStatus generateCPP(long graph, boolean abstractionApplied) {
		boolean res;
		if (abstractionApplied)  
			res = PkPlugin.getServices().genCodeCPPSeparate(graph);
		else  
			res = PkPlugin.getServices().genCodeCPP(graph, true, true, true);
		if (res) return Status.OK_STATUS;		 
		return getErrorCompilationStatus(CompilationUtils.ERROR_CPP_GENERATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @param targetDir
	 *        the path where the Java project will be created
	 * @param abstractionApplied
	 *        boolean set to true if an abstraction is applied
	 * @return the final status of the operation
	 */
	public IStatus generateJava(long graph, IPath targetDir, 
			boolean abstractionApplied, CompilationUtils compilationUtils) {
		// Extract from the zip file the structure of a Java project containing jar files of Polychrony Java libraries.
		PolychronyConsole.printInfo("\n   ==> Create the Java project: " + targetDir.lastSegment() + "\n");
		CreateJavaProject project = new CreateJavaProject("org.eclipse.pop.ssme.compilation.utils",
				"javalibs/simulation.zip", targetDir.lastSegment());
		IProject pr = project.create();
		
		// Add the project to the list of resources to refresh at the end
		compilationUtils.getResources().add(pr);
		
		IFolder dir = pr.getFolder("src");
		if (!dir.exists()) {
			try {
				dir.create(true, true, null);
				// Change the output directory to the generate Java files into the source directory of the created project
				PkPlugin.getPrinter().setOutputDirectory(dir.getLocation().toOSString());
			}
			catch (CoreException e) {
				CompilationUtilsPlugin.getDefault().getLog().log(
					new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, 0, e.getMessage(), e));
			}
		}
		
		boolean res;
		if (abstractionApplied) 
			res = PkPlugin.getServices().genCodeJavaSeparate(graph);
		else  
			res = PkPlugin.getServices().genCodeJava(graph, true, true, true);
		
		// Change the output directory to the initial one
		PkPlugin.getPrinter().setOutputDirectory(targetDir.toOSString());
		
		if (res)  return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_JAVA_GENERATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateLustre(long graph) {
		if (PkPlugin.getServices().LustreUnparse(graph)) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_LUSTRE_GENERATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateProfiling(long graph) {
		if (PkPlugin.getServices().applyTabularMorphism(graph))	return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_PROFILING);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateSigali(long graph) {
		if (PkPlugin.getServices().SigaliPrintExtraction(graph)) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_SIGALI_GENERATION);
	}
	
	/**
	 * @param tree
	 *        the abstract syntax tree
	 * @param fileDirectory
	 *        the output directory
	 * @return the final status of the operation
	 */
	public IStatus generateSignal(long tree, IPath fileDirectory) {
		PkPlugin.getPrinter().setOutputDirectory(fileDirectory.toOSString());
		if (!PkPlugin.getPrinter().printTree(tree)) return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_GENERATION);
		return Status.OK_STATUS;
	}
	
	/**
	 *  Print the Signal textual form of a tree (tree) in a file (AbsfileName).
	 *  No message to announce the generation: useful when the workbench is not created (ssme2signal for example).
	 * @param tree
	 *        the abstract syntax tree
	 * @param AbsfileName
	 *        the absolute pathname of the generated file.
	 * @return the final status of the operation
	 */
	public IStatus generateSignalFile(long tree, String AbsfileName) {
		if (!PkPlugin.getPrinter().printTreeFile(tree, AbsfileName)) return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_GENERATION);
		return Status.OK_STATUS;
	}
	/**
	 * @param tree
	 *        the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateSignalLIS(long tree) {
		boolean signalLISPrinted = PkPlugin.getServices().printSignalLIS(tree, 0);
		if (!signalLISPrinted) return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_GENERATION);
		return Status.OK_STATUS;
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateSignalTRA(long graph) {
		if (PkPlugin.getServices().unparse(graph)) return Status.OK_STATUS;		 
		return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_GENERATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateSignalAbstraction(long graph) {
		if (PkPlugin.getServices().unparseAbstraction(graph)) return Status.OK_STATUS;		 
		return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_ABSTRACTION_GENERATION);
	}
	
	/**
	 * @param tree
	 *        the abstract syntax tree
	 * @param fileDirectory
	 *        the output directory
	 */
	public IStatus generateSSME(long tree, IPath fileDirectory) {
		PkPlugin.getPrinter().setOutputDirectory(fileDirectory.toOSString());
		// System.out.println("generate SSME in " + fileDirectory.toOSString());
		if (PkPlugin.getServices().signal2SSME(tree, fileDirectory.toOSString())) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_SIGNAL_MODEL_GENERATION);
	}
	
	/**
	 * @param graph
	 *        the graph generated from the abstract syntax tree
	 * @return the final status of the operation
	 */
	public IStatus generateSyndex(long graph) {
		if (PkPlugin.getServices().SynDExUnparse(graph)) return Status.OK_STATUS;
		return getErrorCompilationStatus(CompilationUtils.ERROR_SYNDEX_GENERATION);
	}
}
