/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.popup.actions;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.pop.ssme.compilation.utils.jobs.CompilingMethods;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;

import org.eclipse.pop.ssme.polychrony.EnvironmentConstants;
import org.eclipse.pop.ssme.polychrony.GraphValue;
import org.eclipse.pop.ssme.polychrony.PkPlugin;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyConsole;

/**
 * This class describes the interactive compilation feature
 */
public class InteractiveCompiling {
	/** boolean set to true if a clustering is applied */
	private boolean clustersApplied;
	/** boolean set to true if an abstraction is applied */
	private boolean abstractionApplied;

	public InteractiveCompiling(final CompilingMethods compilingMethods,
			final long finalTree, final IFile sourceFile,
			final GraphValue graphValue, final CompilationUtils compilationUtils) {
		Display display = Display.getCurrent();
		final IPath fileLocation = sourceFile.getLocation();
		final IPath fileDirectory = fileLocation.removeFileExtension();
		final Shell shell = new Shell(display);
		shell.setLayout(new GridLayout());
		shell.setText("Interactive Compilation");

		// Buttons Definition

		Group groupOptions = createGroup(shell, "General Options");

		final Button events2BooleansButton = createButton(groupOptions,	"Events To Booleans");
		final Button unificationButton = createButton(groupOptions, "Signals Unification");
		final Button flatteningButton = createButton(groupOptions, "Flattening");
		final Button clockCalculusButton = createButton(groupOptions, "Clock Calculus");
		final Button sequentializingButton = createButton(groupOptions,	"Sequentializing");
		final Button retimingButton = createButton(groupOptions, "Retiming");
		final Button boolean2EventsButton = createButton(groupOptions, "Booleans To Events");
		final Button abstractionButton = createButton(groupOptions,	"Abstraction");
		final Button sequentialClusteringButton = createButton(groupOptions, "Sequential Clustering");
		final Button profilingButton = createButton(groupOptions, "Profiling");

		Group groupExportTools = createGroup(shell, "Export Tools");

		final Button lustreButton = createButton(groupExportTools, "Lustre Generation");
		final Button syndexButton = createButton(groupExportTools, "Syndex Generation");
		final Button sigaliGenButton = createButton(groupExportTools, "Z/3Z - Sigali");

		Group groupGeneration = createGroup(shell, "Code Generation");

		final Button cGenButton = createButton(groupGeneration, "C Generation");
		final Button cppGenButton = createButton(groupGeneration, "C++ Generation");
		final Button javaGenButton = createButton(groupGeneration, "Java Generation");

		Group groupSignalForms = createGroup(shell, "Signal Forms");

		final Button signalGenButton = createButton(groupSignalForms,"Signal Generation");
		final Button signalLISGenButton = createButton(groupSignalForms,"Signal LIS Generation");
		final Button signalTRAGenButton = createButton(groupSignalForms,"Signal TRA Generation");
		final Button signalAbstractionButton = createButton(groupSignalForms,"Signal Abstraction");
		final Button ssmeGenButton = createButton(groupSignalForms,"SSME Generation");

		final Button quitButton = new Button(shell, SWT.PUSH);
		quitButton.setText("Quit");
		GridData gridDataQuitButton = new GridData();
		gridDataQuitButton.horizontalAlignment = GridData.FILL;
		quitButton.setLayoutData(gridDataQuitButton);

		if (sourceFile.getFileExtension().equalsIgnoreCase("ssme"))
			ssmeGenButton.setEnabled(false);
		else if (sourceFile.getFileExtension().equalsIgnoreCase("sig"))
			signalGenButton.setEnabled(false);
		
		sequentialClusteringButton.setEnabled(false);
		sequentializingButton.setEnabled(false);
		flatteningButton.setEnabled(false);
		cGenButton.setEnabled(false);
		cppGenButton.setEnabled(false);
		javaGenButton.setEnabled(false);
		profilingButton.setEnabled(false);
		lustreButton.setEnabled(false);
		syndexButton.setEnabled(false);
		signalAbstractionButton.setEnabled(false);

		// Buttons Listeners

		events2BooleansButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.DCGPoly2DCGBool(graphValue);
					if (!status.equals(Status.OK_STATUS))
						PolychronyConsole.printDetectedError(status.getMessage());
					else {
						retimingButton.setEnabled(false);
						boolean2EventsButton.setEnabled(false);
						unificationButton.setEnabled(false);
						clockCalculusButton.setEnabled(false);
						events2BooleansButton.setEnabled(false);
						abstractionButton.setEnabled(false);
						sequentialClusteringButton.setEnabled(true);
						sequentializingButton.setEnabled(true);
						flatteningButton.setEnabled(true);
						profilingButton.setEnabled(true);
					}

				}
			}
		});

		unificationButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.unify(graphValue);
					if (isActionOk(status)) unificationButton.setEnabled(false);
				}
			}

			 
		});

		flatteningButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.Flattening(graphValue);
					if (isActionOk(status))  {
						sequentializingButton.setEnabled(false);
						flatteningButton.setEnabled(false);
						syndexButton.setEnabled(true);
					}
				}
			}
		});

		clockCalculusButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.clockCalculus(graphValue);
					if (isActionOk(status)) 
						 clockCalculusButton.setEnabled(false);
				}
			}
		});

		sequentializingButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.sequentializing(graphValue, finalTree, clustersApplied);
					if (isActionOk(status)) {
						sequentialClusteringButton.setEnabled(false);
						sequentializingButton.setEnabled(false);
						flatteningButton.setEnabled(false);
						cGenButton.setEnabled(true);
						cppGenButton.setEnabled(true);
						javaGenButton.setEnabled(true);
						lustreButton.setEnabled(true);
						// syndexButton.setEnabled(true);
					}
				}
			}
		});

		retimingButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.retiming(graphValue);
					if (isActionOk(status)) retimingButton.setEnabled(false);
				}
			}
		});

		boolean2EventsButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.boolean2Events(graphValue);
					if (isActionOk(status))
						boolean2EventsButton.setEnabled(false);
				}
			}
		});

		abstractionButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					abstractionApplied = true;
					IStatus status = compilingMethods.abstraction(graphValue);
					if (isActionOk(status)) {
						retimingButton.setEnabled(false);
						boolean2EventsButton.setEnabled(false);
						unificationButton.setEnabled(false);
						clockCalculusButton.setEnabled(false);
						events2BooleansButton.setEnabled(false);
						abstractionButton.setEnabled(false);
						sequentialClusteringButton.setEnabled(true);
						sequentializingButton.setEnabled(true);
						flatteningButton.setEnabled(true);
						profilingButton.setEnabled(true);
						signalAbstractionButton.setEnabled(true);
					}

				}
			}
		});

		sequentialClusteringButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.cluster(graphValue);
					if (isActionOk(status)) sequentialClusteringButton.setEnabled(false);
					clustersApplied = status.isOK();
				}
			}
		});

		profilingButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateProfiling(graphValue.getGraph());
					endActionrefreshContent(compilationUtils, status);	 
				}
			}
		});

		lustreButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateLustre(graphValue.getGraph());
					endActionrefreshContent(compilationUtils, status);					 
				}
			}
		});

		syndexButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSyndex(graphValue.getGraph());
					endActionrefreshContent(compilationUtils, status);
				}
			}

		});

		sigaliGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSigali(graphValue.getGraph());
					endActionrefreshContent(compilationUtils, status);					
				}
			}
		});

		cGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateC(graphValue.getGraph(), abstractionApplied);
					endActionrefreshContent(compilationUtils, status);
				}
			}
		});

		cppGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateCPP(graphValue.getGraph(), abstractionApplied);
					endActionrefreshContent(compilationUtils, status); 
				}
			}
		});

		javaGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateJava(graphValue.getGraph(), fileDirectory, abstractionApplied, compilationUtils);
					endActionrefreshContent(compilationUtils, status);
				}
			}
		});

		signalGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSignal(finalTree,	fileDirectory);
					endActionrefreshContent(compilationUtils, status); 
				}
			}
		});

		signalLISGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSignalLIS(finalTree);
					endActionrefreshContent(compilationUtils, status);
				}
			}
		});

		signalTRAGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSignalTRA(graphValue.getGraph());
					endActionrefreshContent(compilationUtils, status);
				}
			}
		});

		signalAbstractionButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSignalAbstraction(graphValue.getGraph());
					endActionrefreshContent(compilationUtils, status);
				}
			}
		});

		ssmeGenButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					IStatus status = compilingMethods.generateSSME(finalTree, fileDirectory);
					endActionrefreshContent(compilationUtils, status); 
				}
			}
		});

		quitButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (e.type == SWT.Selection) {
					PkPlugin.getServices().setEnv( EnvironmentConstants.LIBRARY_PATH, compilationUtils.getSaveLibraryPath());
					shell.dispose();
				}
			}
		});

		shell.pack();
		shell.open();

		while (!shell.isDisposed()) {
			if (!display.readAndDispatch()) {
				display.sleep();
			}
		}
	}

	protected boolean isActionOk(IStatus status) {
		if (status.equals(Status.OK_STATUS)) return true;
		PolychronyConsole.printDetectedError(status.getMessage());
		return false;
	}

	protected void endActionrefreshContent(CompilationUtils compilationUtils, IStatus status) {
		if (isActionOk(status)) 
			compilationUtils.refreshContent();		
	}

	/**
	 * This method creates a button and inserts it in a group
	 * 
	 * @param group
	 *            the group where the button will be inserted
	 * @param buttonName
	 *            the name of the button
	 * @return the button
	 */
	public Button createButton(Group group, String buttonName) {
		final Button button = new Button(group, SWT.PUSH);
		button.setText(buttonName);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		return button;
	}

	/**
	 * This method creates a group of buttons and inserts it in a shell
	 * 
	 * @param shell
	 *            the shell where the group of buttons will be inserted
	 * @param groupName
	 *            the name of the group
	 * @return the group
	 */
	public Group createGroup(Shell shell, String groupName) {
		Group group = new Group(shell, SWT.NONE);
		group.setLayout(new GridLayout(2, true));
		group.setText(groupName);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		group.setLayoutData(gridData);
		return group;
	}
}
