/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.simulation;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.pop.ssme.compilation.utils.CompilationUtilsPlugin;

public class CreateJavaProject {
	private String	bundleName;
	private String	zipLocation;
	private String	projectName;
	
	/**
	 * Construct a descriptor that points to a zip file located in a particular bundle at the given location within that
	 * bundle. Also provided is the path where to unzip the zip file.
	 * 
	 * @param bundleName
	 *        The bundle in the runtime that contains the zipped up project contents.
	 * @param zipLocation
	 *        The location within the bundle where the zip file is located.
	 * @param projectName
	 *        The name of the project to create.
	 */
	public CreateJavaProject(String bundleName, String zipLocation, String projectName) {
		this.bundleName = bundleName;
		this.zipLocation = zipLocation;
		this.projectName = projectName;
	}
	
	/**
	 * @return
	 */
	public IProject create() {
		URL interpreterZipUrl = FileLocator.find(Platform.getBundle(bundleName), new Path(zipLocation), null);
		
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		
		if (project.exists()) {
			CompilationUtilsPlugin.getDefault().getLog().log(
				new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, "The project '" + projectName
						+ "' already exists. The class will be generated in this project."));
			return project;
		}
		
		try {
			final IProjectDescription description = ResourcesPlugin.getWorkspace().newProjectDescription(projectName);
			description.setNatureIds(new String[] { "org.eclipse.jdt.core.javanature" });
			ICommand command = description.newCommand();
			command.setBuilderName("org.eclipse.jdt.core.javabuilder");
			description.setBuildSpec(new ICommand[] { command });
			
			// We make sure that the project is created from this point forward.
			project.create(description, null);
			
			ZipInputStream zipFileStream = new ZipInputStream(interpreterZipUrl.openStream());
			ZipEntry zipEntry = zipFileStream.getNextEntry();
			
			// We derive a regexedProjectName so that the dots don't end up being
			// interpreted as the dot operator in the regular expression language.
			String regexedProjectName = projectName.replaceAll("\\.", "\\."); //$NON-NLS-1$ //$NON-NLS-2$
			
			while (zipEntry != null) {
				// We will construct the new file but we will strip off the project
				// directory from the beginning of the path because we have already
				// created the destination project for this zip.
				File file = new File(project.getLocation().toString(), zipEntry.getName().replaceFirst(
					"^" + regexedProjectName + "/", "")); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				
				if (false == zipEntry.isDirectory()) {
					
					/*
					 * Copy files (and make sure parent directory exist)
					 */
					File parentFile = file.getParentFile();
					if (null != parentFile && false == parentFile.exists()) {
						parentFile.mkdirs();
					}
					OutputStream os = null;
					
					try {
						os = new FileOutputStream(file);
						
						byte[] buffer = new byte[102400];
						while (true) {
							int len = zipFileStream.read(buffer);
							if (zipFileStream.available() == 0)
								break;
							os.write(buffer, 0, len);
						}
					}
					finally {
						if (null != os) {
							os.close();
						}
					}
				}
				
				zipFileStream.closeEntry();
				zipEntry = zipFileStream.getNextEntry();
			}
			project.open(null);
			project.refreshLocal(IResource.DEPTH_INFINITE, null);
		}
		catch (IOException e) {
			CompilationUtilsPlugin.getDefault().getLog().log(
				new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, IStatus.ERROR, e.getMessage(), e));
		}
		catch (CoreException e) {
			CompilationUtilsPlugin.getDefault().getLog().log(e.getStatus());
		}
		return project;
	}
}
