/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.simulation;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.pop.ssme.compilation.utils.CompilationUtilsPlugin;

/**
 * This class generate the build<language>.xml file to compile and/or execute the code generated by Polychrony. This
 * generated files needs the presence of the PolychronyJavaLibs.jar file inside the source location (for Java).
 * 
 */
public class GenerateBuildFile {
	/**
	 * The directory where the source and library files (such as PolychronyJavaLibs.jar) are
	 */
	private IPath	sourceLocation;
	
	/**
	 * The name of the process to simulate
	 */
	private String	name;
	
	/**
	 * Constructor
	 * 
	 * @param source
	 *        The directory where the source and library files (such as PolychronyJavaLibs.jar) are
	 * @param name
	 *        The name of the process to simulate
	 */
	public GenerateBuildFile(IPath source, String name) {
		this.sourceLocation = source;
		this.name = name;
	}
	
	public void generateJavaFile() {
		if (sourceLocation == null) {
			CompilationUtilsPlugin.getDefault().getLog().log(
				new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID,
					"GenerateBuildFile: The sourceLocation is null"));
		}
		
		// Check that the directory exists and that all directories exists
		File dir = sourceLocation.toFile();
		if (!dir.exists())
			dir.mkdirs();
		
		// The Ant file will be generated only if it does not already exist
		File file = new File(dir, "buildJava.xml");
		if (file.exists())
			return;
		
		PrintWriter pw = null;
		try {
			pw = new PrintWriter(new BufferedWriter(new FileWriter(file)));
			
			pw.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
			pw.println("<project name=\"" + name + "\" default=\"exec\" basedir=\".\">");
			pw.println();
			pw.println("\t<property name=\"bin\" value=\"${basedir}/bin\"/>");
			pw.println();
			pw.println("\t<!-- =================================");
			pw.println("\t      target: clean");
			pw.println("\t     ================================= -->");
			pw.println("\t<target name=\"clean\" description=\"Delete the binary directory and all its content\">");
			pw.println("\t    <delete dir=\"${bin}\"/>");
			pw.println("\t</target>");
			pw.println();
			pw.println("\t<!-- =================================");
			pw.println("\t      target: build");
			pw.println("\t     ================================= -->");
			pw.println("\t<target name=\"build\" depends=\"clean\" description=\"Compile the " + name
					+ " simulation\">");
			pw.println("\t    <mkdir dir=\"${bin}\" description=\"Create the binary directory\" />");
			pw.println("\t    <javac srcdir=\"${basedir}\" destdir=\"${bin}\" includes=\"*.java\" ");
			pw.println("\t    \tclasspath=\"PolychronyJavaLibs.jar\" debug=\"on\" />");
			pw.println("\t</target>");
			pw.println();
			pw.println("\t<!-- =================================");
			pw.println("\t      target: exec");
			pw.println("\t     ================================= -->");
			pw
					.println("\t<target name=\"exec\" depends=\"build\" description=\"Execute the " + name
							+ " simulation\">");
			pw.println("\t    <java classname=\"" + name + "_main\">");
			pw.println("\t    \t<classpath>");
			pw.println("\t    \t\t<pathelement location=\"${basedir}/PolychronyJavaLibs.jar\"/>");
			pw.println("\t    \t\t<pathelement path=\"${bin}${path.separator}${java.class.path}\"/>");
			pw.println("\t    \t</classpath>");
			pw.println("\t    </java>");
			pw.println("\t</target>");
			pw.println();
			pw.println("</project>");
			
			pw.close();
		}
		catch (IOException ioe) {
			CompilationUtilsPlugin.getDefault().getLog().log(
				new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, IStatus.ERROR, ioe.getMessage(), ioe));
		}
	}
}
