/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.compilation.utils.views;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.pop.ssme.compilation.utils.CompilationUtilsPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.ViewPart;

import org.eclipse.pop.ssme.compilation.Functionality;
import org.eclipse.pop.ssme.compilation.FunctionalityKind;
import org.eclipse.pop.ssme.compilation.Generator;
import org.eclipse.pop.ssme.compilation.GeneratorKind;
import org.eclipse.pop.ssme.compilation.Scenario;
import org.eclipse.pop.ssme.compilation.ScenarioItem;
import org.eclipse.pop.ssme.compilation.SsmeCompFactory;
import org.eclipse.pop.ssme.compilation.impl.FunctionalityImpl;

public class CompilationView extends ViewPart implements ISelectionListener {
	private final static String			FILE_EXTENSION		= "ssc";
	
	/** Functionality buttons */
	private Group						functionalities;
	private Button[]					functionality_btn	= new Button[FunctionalityKind.VALUES.size()];
	
	/** Generator buttons */
	private Group						generators;
	private Button[]					generator_btn		= new Button[GeneratorKind.VALUES.size()];
	
	/**
	 * The list which contains the compilation scenario and button to load and save
	 */
	private Group						scenario;
	private Text						descriptionTxt;
	private List						scenario_lst;
	private Button						displayWarnings_btn;
	private Button						load_btn;
	private Button						save_btn;
	private Button						save_as_btn;
	private Button						reset_btn;
	
	private CompilerSelectionListener	listener			= new CompilerSelectionListener();
	
	/** file from/in which the scenario is loaded/saved */
	private String						filePath			= null;
	
	private boolean						isScenarioSaved		= true;
	
	/**
	 * The constructor.
	 */
	public CompilationView() {}
	
	/**
	 * This is a callback that will allow us to create the viewer and initialize it.
	 */
	public void createPartControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new RowLayout(SWT.HORIZONTAL));
		
		/* ================ FUNCTIONALITIES =================== */
		functionalities = new Group(composite, SWT.SHADOW_NONE);
		functionalities.setText("Functionalities");
		functionalities.setLayout(new GridLayout(2, true));
		
		for (int i = 0; i < FunctionalityKind.VALUES.size(); i++) {
			functionality_btn[i] = new Button(functionalities, SWT.PUSH);
			FunctionalityKind val = FunctionalityKind.get(i);
			functionality_btn[i].setText(val.getLiteral());
			functionality_btn[i].setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
			functionality_btn[i].addSelectionListener(listener);
		}
		
		/* ================ GENERATORS =================== */
		generators = new Group(composite, SWT.SHADOW_NONE);
		generators.setText("Generators");
		generators.setLayout(new GridLayout(2, true));
		
		for (int i = 0; i < GeneratorKind.VALUES.size(); i++) {
			generator_btn[i] = new Button(generators, SWT.PUSH);
			GeneratorKind val = GeneratorKind.get(i);
			generator_btn[i].setText(val.getLiteral());
			generator_btn[i].setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
			generator_btn[i].addSelectionListener(listener);
		}
		
		/* ================ SCENARIO =================== */
		scenario = new Group(composite, SWT.SHADOW_NONE);
		scenario.setText("Scenario: none");
		
		Label lbl = new Label(scenario, SWT.NONE);
		lbl.setText("Description:");
		lbl.setBounds(10, 15, 200, 15);
		descriptionTxt = new Text(scenario, SWT.NONE);
		descriptionTxt.setBounds(10, 30, 200, 15);
		
		scenario_lst = new List(scenario, SWT.BORDER | SWT.SINGLE | SWT.V_SCROLL);
		scenario_lst.setBounds(10, 50, 200, 140);
		
		// Add a check box to decide to display or not warnings
		displayWarnings_btn = new Button(scenario, SWT.CHECK);
		displayWarnings_btn.setText("Display all warnings");
		displayWarnings_btn.setBounds(10, 195, 200, 12);
		displayWarnings_btn.addSelectionListener(listener);
		
		/* ================ BUTTONS =================== */
		Composite buttons = new Composite(composite, SWT.NONE);
		buttons.setBounds(890, 10, 100, 150);
		buttons.setLayout(new FillLayout(SWT.VERTICAL));
		
		load_btn = new Button(buttons, SWT.PUSH);
		load_btn.setText("Load");
		load_btn.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent event) {
				loadScenario();
			}
			
			public void widgetDefaultSelected(SelectionEvent event) {
				loadScenario();
			}
		});
		
		save_btn = new Button(buttons, SWT.PUSH);
		save_btn.setText("Save");
		save_btn.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent event) {
				saveScenario(false);
			}
			
			public void widgetDefaultSelected(SelectionEvent event) {
				saveScenario(false);
			}
		});
		
		save_as_btn = new Button(buttons, SWT.PUSH);
		save_as_btn.setText("Save As");
		save_as_btn.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent event) {
				saveScenario(true);
			}
			
			public void widgetDefaultSelected(SelectionEvent event) {
				saveScenario(true);
			}
		});
		
		reset_btn = new Button(buttons, SWT.PUSH);
		reset_btn.setText("Reset");
		reset_btn.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent event) {
				reset();
			}
			
			public void widgetDefaultSelected(SelectionEvent event) {
				reset();
			}
		});
		
		reset();
	}
	
	/**
	 * Initialize the status of all functionality and generator buttons, and empty the list.
	 */
	private void reset() {
		// Reset the path to the file used to load/save a scenario
		filePath = null;
		isScenarioSaved = true;
		scenario.setText("Scenario: none");
		descriptionTxt.setText("");
		
		// Uncheck the display warning checkbox
		displayWarnings_btn.setSelection(false);
		// Empty the list and disable the execute and save button
		scenario_lst.removeAll();
		save_btn.setEnabled(false);
		save_as_btn.setEnabled(false);
		
		// Enables funtionalities and generators that can be initially called
		functionality_btn[FunctionalityKind.RETIMING].setEnabled(true);
		functionality_btn[FunctionalityKind.BOOLEAN2_EVENTS].setEnabled(true);
		functionality_btn[FunctionalityKind.SIGNAL_UNIFICATIONS].setEnabled(true);
		functionality_btn[FunctionalityKind.CLOCK_CALCULUS].setEnabled(true);
		functionality_btn[FunctionalityKind.EVENTS2_BOOLEAN].setEnabled(true);
		functionality_btn[FunctionalityKind.ABSTRACTION].setEnabled(true);
		generator_btn[GeneratorKind.SIGNAL].setEnabled(true);
		generator_btn[GeneratorKind.SIGALI].setEnabled(true);
		generator_btn[GeneratorKind.SSME].setEnabled(true);
		
		// Disables funtionalities and generators that can't be initially called
		functionality_btn[FunctionalityKind.CLUSTERING].setEnabled(false);
		functionality_btn[FunctionalityKind.SEQUENTIALIZING].setEnabled(false);
		functionality_btn[FunctionalityKind.FLATTENING].setEnabled(false);
		generator_btn[GeneratorKind.SIGNAL_ABSTRACTION].setEnabled(false);
		generator_btn[GeneratorKind.CANSI].setEnabled(false);
		generator_btn[GeneratorKind.CPP].setEnabled(false);
		generator_btn[GeneratorKind.JAVA].setEnabled(false);
		generator_btn[GeneratorKind.PROFILING].setEnabled(false);
		generator_btn[GeneratorKind.LUSTRE].setEnabled(false);
		generator_btn[GeneratorKind.SYNDEX].setEnabled(false);
	}
	
	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		functionality_btn[0].setFocus();
	}
	
	/**
	 * 
	 */
	public void loadScenario() {
		// If the scenario is not saved yet, we ask the user if we had to save
		// it before reseting the list.
		if (!isScenarioSaved) {
			MessageBox messageBox = new MessageBox(getSite().getShell(), SWT.ICON_QUESTION | SWT.YES | SWT.NO);
			messageBox.setMessage("Do you want to save your scenario before loading the new one ?");
			messageBox.setText("Saving scenario");
			int response = messageBox.open();
			if (response == SWT.YES)
				saveScenario(false);
		}
		
		FileDialog loadDialog = new FileDialog(getSite().getShell(), SWT.OPEN);
		loadDialog.setFilterExtensions(new String[] { "*." + FILE_EXTENSION });
		loadDialog.setText("Load Scenario...");
		IWorkspaceRoot ws = ResourcesPlugin.getWorkspace().getRoot();
		loadDialog.setFilterPath(ws.getLocation().toString());
		String path = loadDialog.open();
		
		if (path != null) {
			ResourceSet resourceSet = new ResourceSetImpl();
			
			URI fileURI = URI.createFileURI(path);
			Resource resource = resourceSet.getResource(fileURI, true);
			
			Scenario scenario_model = null;
			for (EObject obj : resource.getContents()) {
				if (obj instanceof Scenario)
					scenario_model = (Scenario) obj;
			}
			if (scenario_model == null) {
				MessageBox messageBox = new MessageBox(getSite().getShell(), SWT.ICON_ERROR | SWT.OK);
				messageBox.setMessage("The file \"" + path + "\" does not contain a compilation scenario.");
				messageBox.setText("Saving scenario");
				messageBox.open();
			}
			else {
				reset();
				filePath = path;
				displayWarnings_btn.setSelection(scenario_model.isDisplayWarnings());
				for (ScenarioItem item : scenario_model.getElements()) {
					if (item instanceof FunctionalityImpl) {
						org.eclipse.pop.ssme.compilation.FunctionalityKind kind = ((FunctionalityImpl) item).getKind();
						activateButton(functionality_btn[kind.getValue()]);
					}
					else if (item instanceof Generator) {
						GeneratorKind kind = ((Generator) item).getKind();
						activateButton(generator_btn[kind.getValue()]);
					}
				}
				isScenarioSaved = true;
				scenario.setText("Scenario: " + fileURI.trimFileExtension().lastSegment());
				if (scenario_model.getDescription() != null)
					descriptionTxt.setText(scenario_model.getDescription());
			}
		}
	}
	
	/**
	 * @param force
	 *        true when the saveScenario must save in another file, false otherwise.
	 */
	public void saveScenario(boolean force) {
		if (force || filePath == null) {
			FileDialog saveDialog = new FileDialog(getSite().getShell(), SWT.SAVE);
			saveDialog.setFilterExtensions(new String[] { "*." + FILE_EXTENSION });
			saveDialog.setText("Save Scenario as...");
			IWorkspaceRoot ws = ResourcesPlugin.getWorkspace().getRoot();
			saveDialog.setFilterPath(ws.getLocation().toString());
			filePath = saveDialog.open();
			if (filePath == null)
				return;
			if (!filePath.endsWith("." + FILE_EXTENSION))
				filePath += "." + FILE_EXTENSION;
		}
		
		Scenario scenario_model = SsmeCompFactory.eINSTANCE.createScenario();
		scenario_model.setDisplayWarnings(displayWarnings_btn.getSelection());
		String[] items = scenario_lst.getItems();
		for (int i = 0; i < items.length; i++) {
			FunctionalityKind func = FunctionalityKind.get(items[i]);
			if (func != null) {
				Functionality f = SsmeCompFactory.eINSTANCE.createFunctionality();
				f.setKind(func);
				scenario_model.getElements().add(f);
			}
			else {
				GeneratorKind gen = GeneratorKind.get(items[i]);
				if (gen != null) {
					Generator g = SsmeCompFactory.eINSTANCE.createGenerator();
					g.setKind(gen);
					scenario_model.getElements().add(g);
				}
				else {
					CompilationUtilsPlugin.getDefault().getLog().log(
						new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID,
							"Unrecognized features in the scenario : " + items[i]));
				}
			}
		}
		
		try {
			// Create a resource set
			ResourceSet resourceSet = new ResourceSetImpl();
			
			// Get the URI of the model file.
			URI fileURI = URI.createFileURI(filePath);
			String filename = fileURI.lastSegment();
			filename = filename.substring(0, filename.length() - 1 - FILE_EXTENSION.length());
			
			if (descriptionTxt.getText() != null)
				scenario_model.setDescription(descriptionTxt.getText());
			else
				scenario_model.setDescription(filename);
			
			// Create a resource for this file.
			Resource resource = resourceSet.createResource(fileURI);
			
			// Add the initial model object to the contents.
			resource.getContents().add(scenario_model);
			
			// Save the contents of the resource to the file system.
			Map<Object, Object> options = new HashMap<Object, Object>();
			options.put(XMLResource.OPTION_ENCODING, "UTF-8");
			resource.save(options);
			isScenarioSaved = true;
			scenario.setText("Scenario: " + filename);
			
		}
		catch (Exception exception) {
			CompilationUtilsPlugin.getDefault().getLog().log(
				new Status(IStatus.ERROR, CompilationUtilsPlugin.PLUGIN_ID, 0, "Save Scenario error.", exception));
		}
		
	}
	
	private void activateButton(Button btn) {
		isScenarioSaved = false;
		save_btn.setEnabled(true);
		save_as_btn.setEnabled(true);
		
		if (btn.equals(displayWarnings_btn))
			return;
		
		String literal = btn.getText();
		scenario_lst.add(btn.getText());
		
		if (btn.getParent().equals(functionalities)) {
			btn.setEnabled(false);
			FunctionalityKind func = FunctionalityKind.get(literal);
			if (FunctionalityKind.EVENTS2_BOOLEAN_LITERAL.equals(func)) {
				functionality_btn[FunctionalityKind.RETIMING].setEnabled(false);
				functionality_btn[FunctionalityKind.BOOLEAN2_EVENTS].setEnabled(false);
				functionality_btn[FunctionalityKind.SIGNAL_UNIFICATIONS].setEnabled(false);
				functionality_btn[FunctionalityKind.CLOCK_CALCULUS].setEnabled(false);
				functionality_btn[FunctionalityKind.ABSTRACTION].setEnabled(false);
				functionality_btn[FunctionalityKind.CLUSTERING].setEnabled(true);
				functionality_btn[FunctionalityKind.SEQUENTIALIZING].setEnabled(true);
				functionality_btn[FunctionalityKind.FLATTENING].setEnabled(true);
				generator_btn[GeneratorKind.PROFILING].setEnabled(true);
				generator_btn[GeneratorKind.SYNDEX].setEnabled(false);
			}
			else if (FunctionalityKind.ABSTRACTION_LITERAL.equals(func)) {
				functionality_btn[FunctionalityKind.RETIMING].setEnabled(false);
				functionality_btn[FunctionalityKind.BOOLEAN2_EVENTS].setEnabled(false);
				functionality_btn[FunctionalityKind.SIGNAL_UNIFICATIONS].setEnabled(false);
				functionality_btn[FunctionalityKind.CLOCK_CALCULUS].setEnabled(false);
				functionality_btn[FunctionalityKind.EVENTS2_BOOLEAN].setEnabled(false);
				functionality_btn[FunctionalityKind.CLUSTERING].setEnabled(true);
				functionality_btn[FunctionalityKind.SEQUENTIALIZING].setEnabled(true);
				functionality_btn[FunctionalityKind.FLATTENING].setEnabled(true);
				generator_btn[GeneratorKind.SIGNAL_ABSTRACTION].setEnabled(true);
				generator_btn[GeneratorKind.PROFILING].setEnabled(true);
			}
			else if (FunctionalityKind.SEQUENTIALIZING_LITERAL.equals(func)) {
				functionality_btn[FunctionalityKind.CLUSTERING].setEnabled(false);
				functionality_btn[FunctionalityKind.FLATTENING].setEnabled(false);
				generator_btn[GeneratorKind.CANSI].setEnabled(true);
				generator_btn[GeneratorKind.CPP].setEnabled(true);
				generator_btn[GeneratorKind.JAVA].setEnabled(true);
				generator_btn[GeneratorKind.LUSTRE].setEnabled(true);
				generator_btn[GeneratorKind.SYNDEX].setEnabled(true);
			}
			else if (FunctionalityKind.FLATTENING_LITERAL.equals(func)) {
				functionality_btn[FunctionalityKind.SEQUENTIALIZING].setEnabled(false);
				generator_btn[GeneratorKind.SYNDEX].setEnabled(false);
			}
		}
	}
	
	class CompilerSelectionListener implements SelectionListener {
		public void widgetSelected(SelectionEvent event) {
			Button btn = (Button) event.getSource();
			activateButton(btn);
		}
		
		public void widgetDefaultSelected(SelectionEvent event) {
			Button btn = (Button) event.getSource();
			activateButton(btn);
		}
	}
	
	/**
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart,
	 *      org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {

	}
}
