/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.pop.ssme.polychrony.preferences.SsmePreferenceConstants;
import org.eclipse.pop.ssme.polychrony.traceability.TreesInterface;

/**
 * This file contains all SIGNAL services functions concerning the
 * transformations of a SIGNAL graph.
 */
public class PKServices {
	private final static String NATIVE_LIB_NOT_FOUND = "The native library for POP is not available.";
	private final static String NO_NATIVE_RESOURCES = "Cannot find native resources to initialize the SignalToolBox";
	private final static String BAD_SIGNALTOOLBOX_VERSION = "Bad SignalToolBox version, expected version: ";
	private final static String UPDATE_SIGNALTOOBOX_PREFERENCES = " Update your POP preferences (Select: Windows->Preferences->Pop)";
	private final static String BAD_SIGNALTOOLBOX_PATH = "Bad SignalToolBox path (or old version)";
	private final static String USING_SIGNALTOOLBOX_USER_ENVIRONMENT = "Using the SignalToolBox defined in your environment. To modify it," + UPDATE_SIGNALTOOBOX_PREFERENCES;
	private static boolean SignaltoolBoxLoaded = false;

	// Access to the environment for SignalToolBox.
	private static String pK_Signal_bin;
	public static String pK_Version = "xxx";
	private static String pK_Signal_obj = null;
	private static String pK_SignalLib_Std = null;
	private static String pK_ROOT = null;

	/*
	 * Loads the Signal Java-C interface library to access to compilation and
	 * verification services.
	 */
	static {
		// System.out.println(" Loading native library");
		// Looking for the SignalToolBox root, in the Polychrony preferences.
		ILog alog = PkPlugin.getDefault().getLog();
		IPreferenceStore pref = PkPlugin.getDefault().getPreferenceStore();
		pK_ROOT = pref.getString(SsmePreferenceConstants.SIGNALTOOLBOX_ROOT);
		int i = 1;
		String pK_Signal = "";
		while (pK_Version.equals("xxx") && (i <= 2)) {
			pK_Signal = pK_ROOT + File.separatorChar + "Signal";
			pK_Signal_bin = pK_Signal + File.separatorChar + "bin";
			// System.out.println(" Trying the adress pK_Signal_bin " + pK_Signal_bin);
			try {
				// First line = version, then ...
				String file = pK_Signal_bin + File.separatorChar + EnvironmentConstants.POP_SIGNALTOOLBOX_PREF;
				FileReader fr = new FileReader(file);
				BufferedReader br = new BufferedReader(fr);
				pK_Version = br.readLine();

				fr.close();
			} catch (IOException ule) {
				i++;
				pK_ROOT = System.getenv("pK_ROOT"); // Perhaps, the SignaltoolBox is already available in the user environment.
			}
		}

		if (pK_Version.equals("xxx")) {
			alog.log(new Status(IStatus.ERROR, PkPlugin.PLUGIN_ID, BAD_SIGNALTOOLBOX_PATH, null));
			alog.log(new Status(IStatus.ERROR, PkPlugin.PLUGIN_ID, UPDATE_SIGNALTOOBOX_PREFERENCES, null));
			alog.log(new Status(IStatus.ERROR, PkPlugin.PLUGIN_ID, NATIVE_LIB_NOT_FOUND, null));
		} 
		else if (pK_Version.equals(EnvironmentConstants.EXPECTED_SIGNALTOOLBOX)) {
			// System.out.println(" Loading native library, version = " + pK_Version);
			pK_Signal_obj = pK_Signal + File.separatorChar + "obj";
			pK_SignalLib_Std = pK_ROOT + File.separatorChar + "SignalLibraries" + File.separatorChar + "Std";
			try {
				String OslibName = System.mapLibraryName("SignalToolBoxEclipse" + "_" + EnvironmentConstants.EXPECTED_SIGNALTOOLBOX);
				String str = pK_Signal_obj + File.separatorChar + OslibName;
				System.out.println(" Loading native library,  " + str);				
				System.load(str);
				SignaltoolBoxLoaded = true;
				if ( i == 2 )
						{
						pref.setValue(SsmePreferenceConstants.SIGNALTOOLBOX_ROOT, pK_ROOT);
						alog.log(new Status(IStatus.INFO, PkPlugin.PLUGIN_ID, USING_SIGNALTOOLBOX_USER_ENVIRONMENT, null));
						}
			} catch (UnsatisfiedLinkError ule) {
				alog.log(new Status(IStatus.ERROR, PkPlugin.PLUGIN_ID, NATIVE_LIB_NOT_FOUND, ule));
			}
		}

	}

	public final static PKServices INSTANCE = new PKServices();
	private SignalParser parser;
	private SignalTreePrinter printer;
	private SignalTreeBuilting builder;
	private PKTrees treeAPI;
	private SignalTreeComment comment;
	private TreesInterface treesInterface;
	private SignalTreeExploring explorer;

	/**
	 * Constructor
	 */
	private PKServices() {

		if (SignaltoolBoxLoaded) {

			setEnv("Signal_bin", pK_Signal_bin);
			setEnv("Signal_obj", pK_Signal_obj);
			setEnv("pK_VERSION", pK_Version);
			try {

				newMachine();
				IPreferenceStore pref = PkPlugin.getDefault().getPreferenceStore();

				// Initialization of the necessary Signal environment variables
				String value = pref.getString(SsmePreferenceConstants.SIGNAL_LIBRARY_PATH);
				setSignalLibraryPath(value);

				if (!("".equals(value))) value = value + File.pathSeparator;
				value = value + pK_SignalLib_Std; // @ of the Signal Std library.
				setEnv(EnvironmentConstants.LIBRARY_PATH, value);

				// No effect if they are modified during the session (internal
				// size of the SignalToolBox virtual machine)
				int val = pref.getInt(SsmePreferenceConstants.TYPE_TABLE_SIZE);
				if (val <= 0)
					val = EnvironmentConstants.TYPE_TABLE_SIZE_DEFAULT_VALUE;
				setEnv(EnvironmentConstants.TYPE_TABLE_SIZE, "" + val);

				val = pref.getInt(SsmePreferenceConstants.T_MEM_ARB);
				if (val <= 0)
					val = EnvironmentConstants.TREE_MEMORY_SIZE_DEFAULT_VALUE;
				setEnv(EnvironmentConstants.TREE_MEMORY_SIZE, "" + val);

				val = pref.getInt(SsmePreferenceConstants.CHAIN_MEM_ARB);
				if (val <= 0)
					val = EnvironmentConstants.STRING_MEMORY_SIZE_DEFAULT_VALUE;
				setEnv(EnvironmentConstants.STRING_MEMORY_SIZE, "" + val);

				initMachine();
			} catch (Exception e) {
				e.printStackTrace();
			}

			parser = new SignalParser();
			printer = new SignalTreePrinter();
			builder = new SignalTreeBuilting();
			treeAPI = new PKTrees();
			comment = new SignalTreeComment();
			treesInterface = new TreesInterface();
			explorer = new SignalTreeExploring();
		}

	}

	public boolean isSignalToolBoxLoaded() {
		return SignaltoolBoxLoaded;
	}

	public String getWaitedVersion() {
		return EnvironmentConstants.EXPECTED_SIGNALTOOLBOX;
	}

	/**
	 * @return
	 */
	public SignalParser getParser() {
		return parser;
	}

	/**
	 * @return the printer
	 */
	public SignalTreePrinter getPrinter() {
		return printer;
	}

	/**
	 * @return the tree builder
	 */
	public SignalTreeBuilting getTreeBuilder() {
		return builder;
	}

	/**
	 * @return the tree builder
	 */
	public SignalTreeExploring getTreeExplorer() {
		return explorer;
	}

	/**
	 * @return the treeApi
	 */
	public PKTrees getTreeAPI() {
		return treeAPI;
	}

	/**
	 * @return the tree comment
	 */
	public SignalTreeComment getCommentAPI() {
		return comment;
	}

	public TreesInterface getTraceabilityAPI() {
		return treesInterface;
	}

	static public void setSignalLibraryPath(String value) {
		if (SignaltoolBoxLoaded) {
			if (!("".equals(value)))
				value = value + File.pathSeparator;
			value = value + pK_SignalLib_Std; // @ of the Signal Std library.
			setEnv(EnvironmentConstants.LIBRARY_PATH, value);
		}
	}

	/*******************************************************************************************************************
	 * How to manage an interface
	 * 
	 * C function file : pKServicesAPI_g.h
	 ******************************************************************************************************************/

	/**
	 * Instantiate a new PKService Machine This method must be called before
	 * calling initMachine()
	 */
	static private synchronized native void newMachine();

	/**
	 * Initialize the use of the API.
	 * 
	 * @return the pkService API instance
	 */
	static private synchronized native void initMachine();

	static public synchronized native void resetMachine();

	/**
	 * This method enables/disables the warning flag. This method has to be
	 * called before any tree or graph transformation to be used.
	 * 
	 * @param enable
	 *            true to display the warnings, false otherwise.
	 */
	static public synchronized native void displayWarnings(boolean enable);

	/**
	 * Finalyzing the use of the PKServices
	 */
	public synchronized native void finalize();

	/*******************************************************************************************************************
	 * Applying transformations on the tree
	 * 
	 * C function file : pKServicesAPImkGraph_g.h
	 ******************************************************************************************************************/

	/**
	 * Abstract tree representation to graph translation (entry point for the
	 * GUI integrated compiler). The following steps are applied :
	 * <ul>
	 * <li>translating to kernel-tree representation.</li>
	 * <li>production the graph.</li>
	 * 
	 * @param tree
	 *            tree node built by the getDeclarationAST of a Model
	 *            Declaration or a Module.
	 * @param retTreeError
	 *            true to return error-code for the tree.
	 * @param paramTree
	 *            abstract tree of the parameters file.
	 * @param retParamTreeError
	 *            true to return error-code of the parser for paramTree.
	 * @param dirName
	 *            the complete name of the directory where to generate the
	 *            compiling result.
	 * @param generateLISFile
	 *            true to produce the annotated original Signal program
	 *            (X_LIS.SIG file) when errors or not.
	 * @param vgraph
	 *            (out) : the graph of an instance of the model defined in
	 *            srcfile.
	 * @return a non null value when errors are detected, 0 otherwise.
	 */
	static public synchronized native GraphValue tree2DCG(long tree, boolean retTreeError, long paramTree,
			boolean retParamTreeError, String outputDirectory, boolean generateLISFile, boolean bannounce);

	/**
	 * It produces in a file the text representation of a Signal model (tree)
	 * represented as abstract tree, with its associated parameters (params).
	 * The name of the file is created from the name of the model (suffixed by
	 * _LIS).
	 * 
	 * @param tree
	 *            a Signal Tree model
	 * @param params
	 *            the effective associated parameters of the tree model.
	 * @return "true" if the operation successes, "false" otherwise.
	 */
	static public synchronized native boolean printSignalLIS(long tree, long params);

	/**
	 * This method translates an abstract syntax tree corresponding to a Signal
	 * program into its SSME representation, according to the SSME meta-model.
	 * 
	 * @param tree
	 *            the abstract syntax tree to translate.
	 * @param outputDirectory
	 *            the directory in which the SSME file is generated. the name of
	 *            the file is given by the name of the Signal model, suffixed by
	 *            ".ssme".
	 * @return the value 0 if the operation succeeds, greater than 0 otherwise.
	 * 
	 */
	static public synchronized native boolean signal2SSME(long tree, String outputDirectory);

	static public synchronized native boolean signal2SSME(String srcFile, String outputDirectory);

	/**
	 * Deprecated. SME will disappear soon. This method translates the
	 * representation of a textual Signal program into its SME representation,
	 * according to the SSME meta-model.
	 * 
	 * @param srcFile
	 *            the file that contains the Signal program to translate.
	 * @param outputDirectory
	 *            the directory in which the SME file is generated. the name of
	 *            the file is given by the name of the Signal model, suffixed by
	 *            ".sme".
	 * @return the value 0 if the operation succeeds, greater than 0 otherwise.
	 * 
	 */
	static public synchronized native boolean signal2SME(String srcFile, String outputDirectory);

	static public synchronized native boolean signal2SME(long tree, String outputDirectory);
	/*******************************************************************************************************************
	 * Applying transformations on the graph
	 * 
	 * C function file : pKServicesAPIGraph_g.h
	 ******************************************************************************************************************/

	/**
	 * @param graphNode
	 * @param mainFile
	 * @param bodyFile
	 * @param ioFile
	 * @return
	 */
	static public synchronized native boolean genCodeC(long graphNode, boolean mainFile, boolean bodyFile,
			boolean ioFile);

	/**
	 * @param graphNode
	 * @param mainFile
	 * @param bodyFile
	 * @param ioFile
	 * @return
	 */
	static public synchronized native boolean genCodeCPP(long graphNode, boolean mainFile, boolean bodyFile,
			boolean ioFile);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean genCodeCPPSeparate(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean genCodeCSeparate(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean DotUnparse(long graphNode);

	/**
	 * @param graphNode
	 * @param mainFile
	 * @param bodyFile
	 * @param ioFile
	 * @return
	 */
	static public synchronized native boolean genCodeJava(long graphNode, boolean mainFile, boolean bodyFile,
			boolean ioFile);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean genCodeJavaSeparate(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean expressionExpand(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean unparse(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native long produce(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native long produceFlat(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean SynDExUnparse(long graphNode);

	/**
	 * @param graphNode
	 * @param mainFile
	 * @param bodyFile
	 * @param ioFile
	 * @return
	 */
	static public synchronized native boolean genCodeJavaThreads(long graphNode, boolean mainFile, boolean bodyFile,
			boolean ioFile);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean IGraph2Vhdl(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean SigaliPrintExtraction(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native long produceInterfaceAbstraction(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native long buildAbstraction(long graphNode);

	/**
	 * @param graphNode
	 * @param delayMultiplexing
	 */
	static public synchronized native void memoryOptimizations(long graphNode, boolean delayMultiplexing);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native GraphValue DCGPoly2DCGBool(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean transformBooleanClocks2Events(long graphNode);

	/**
	 * @param graphNode
	 */
	static public synchronized native void separateBooleansOthers(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native GraphValue DCGBasic2DCGPoly(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean setIntrinsicClocks(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean dependenceReinforcing(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean addExceptionsForConstraints(long graphNode);

	/**
	 * @param graphNode
	 * @param debug
	 * @return
	 */
	static public synchronized native boolean hasStaticCycle(long graphNode, boolean debug);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean splitDefaultDefinitions(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean setDelayMemoryOverloading(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean moveIn2OutDelays(long graphNode);

	/**
	 * @param graphNode
	 * @param vlevel
	 */
	static public synchronized native void computeClockExclusions(long graphNode, int vlevel);

	/**
	 * @param graphNode
	 * @param level
	 */
	static public synchronized void computeClockExclusions(long graphNode, ExclusiveClockLevelKind level) {
		computeClockExclusions(graphNode, level.getValue());
	}

	/**
	 * @param graphNode
	 * @param asSubGraph
	 * @return
	 */
	static public synchronized native boolean expandModelReference(long graphNode, boolean asSubGraph);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean breakCPG(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean checkExplicitTyping(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean printCHStatistics(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean phylumClustering(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean buildClockProcessInterface(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean hasUnsolvedClockConstraints(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean delayTreeBuild(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean groupPartialDefinitions(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean applyTabularMorphism(long graphNode);

	/**
	 * @param graphNode
	 * @param clockOfDependences
	 * @return
	 */
	static public synchronized native boolean buildIOClockSubHierarchy(long graphNode, boolean clockOfDependences);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean nonProductiveSignalElimination(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean removeSignalEqualities(long graphNode);

	/**
	 * @param graphNode
	 * @param genCode
	 * @return
	 */
	static public synchronized native boolean separatedCompilingPoly(long graphNode, boolean genCode);

	/**
	 * @param graphNode
	 * @param alwaysCode
	 * @param check
	 * @param delayMultiplexing
	 * @return
	 */
	static public synchronized native GraphValue Serializing(long graphNode, boolean btotalOrder, boolean alwaysCode,
			boolean check, boolean delayMultiplexing);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean setClockDependences(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean unparseAbstraction(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean splitDataStructure(long graphNode);

	/**
	 * @param graphNode
	 */
	static public synchronized native void separateStatesOthers(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native void computeProductiveClockStateVar(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native GraphValue DCGBool2DCGFlat(long graphNode);

	/**
	 * @param graphNode
	 * @param subGraph
	 * @return
	 */
	static public synchronized native long produceSubgraph(long graphNode, int subGraph);

	/**
	 * @param graphNode
	 * @param maxRef
	 * @return
	 */
	static public synchronized native boolean substituteExpr2Var(long graphNode, int maxRef);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean computeIOTransitiveClosure(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean removeSignalClones(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native void computeProductiveClock(long graphNode);

	/**
	 * @param graphNode
	 * @return
	 */
	static public synchronized native boolean LustreUnparse(long graphNode);

	/*******************************************************************************************************************
	 * Parse SIGNAL file
	 * 
	 ******************************************************************************************************************/

	/**
	 * This method parses a SIG source file and a parameter file and returns the
	 * corresponding graph. Take care, here, it is not an AST, but the
	 * tree2Graph method has already been called.
	 * 
	 * @param srcfile
	 *            the SIG file
	 * @param paramfile
	 *            the parameter file, or the empty string is none.
	 * @return the graph representing the files
	 */
	static public synchronized native long parse(String srcFile, String paramFile);

	/*******************************************************************************************************************
	 * Set environment variables needed by the compiler THOSE NEEDED BY THE
	 * COMPILER HAS TO BE SET AFTER THE INIT MACHINE !!!
	 * 
	 ******************************************************************************************************************/

	/**
	 * @param key
	 *            a string to identify the environment variable
	 * @param value
	 *            the value associated to the key
	 */
	static public synchronized native void setEnv(String key, String value);

	/**
	 * @param key
	 *            a string to identify the environment variable
	 * @return the value of the environment variable identified by
	 *         <code>key</code> if it exists, null otherwise.
	 */
	static public synchronized native String getEnv(String key);

}
