/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony;

import java.util.StringTokenizer;

/**
 * This file contains Java API of the POP.ree (ntrees type) representation. It is a low level representation.
 * Conversions of type between C and Java
 */
public class PKTrees {
	/** Value returned when there is an error */
	public final static long	ERROR_NODE	= -1;
	
	public PKTrees() {}
	
	/*******************************************************************************************************************
	 * Memory initialization and finalization
	 * 
	 * C function file : intetrees_g.c
	 ******************************************************************************************************************/
	
	/**
	 * Tree Memory finalizing. This method must be called at the end of Tree session.
	 */
	public synchronized native void memoryFinalize();
	
	/**
	 * Global Tree Memory initializing. This method MUST BE CALLED BEFORE USING OTHER FUNCTIONS.
	 */
	public synchronized native void memoryInitialize();
	
	/*******************************************************************************************************************
	 * builting, exploring, modifying primitives methods.
	 * 
	 * C function file : intetreesGetSet_g.c
	 ******************************************************************************************************************/
	
	/**
	 * Builting char atomic tree
	 * 
	 * @param opCode
	 *        an operator
	 * @param c
	 *        a character
	 * @return a new atomic tree with <code>opCode</code> as operator and <code>c</code> as char value.
	 */
	public synchronized native long atomChar(int opCode, char c);
	
	/**
	 * Builting int atomic tree
	 * 
	 * @param opCode
	 *        an operator
	 * @param val
	 *        an integer value
	 * @return a new atomic tree with <code>opCode</code> as operator and <code>val</code> as integer value.
	 */
	public synchronized native long atomInt(int opCode, long val);
	
	/**
	 * Builting string atomic tree
	 * 
	 * @param opCode
	 *        the code representing an operator
	 * @param st
	 *        a string
	 * @return a new atomic tree with <code>opCode</code> as operator and <code>st</code> as string value.
	 */
	public synchronized native long atomString(int opCode, String st);
	
	/**
	 * Replaces the <code>i</code>-th son of <code>listNode</code> by <code>node</code>.
	 * 
	 * @param listNode
	 *        a node representing a list
	 * @param node
	 *        a tree node
	 * @param i
	 *        index of the element to modify
	 */
	public synchronized native void changChild(long listNode, long node, int i);
	
	/**
	 * Getting atomic tree value.
	 * 
	 * @param node
	 *        an atomic tree node (assumed)
	 * @return the char denoted by the atomic tree <code>node</code> (assumed)
	 */
	public synchronized native char charOf(long node);
	
	/**
	 * Getting a son.
	 * 
	 * @param i
	 *        index of the child to get
	 * @param node
	 *        the address of the node
	 * @return the <code>i</code>-th child of <code>node</code>
	 */
	public synchronized native long child(int i, long node);
	
	/**
	 * @param node
	 *        a tree node
	 * @return a copy of <code>node</code>.
	 */
	public synchronized native long copyList(long node);
	
	/**
	 * Deletes the <code>i</code>-th son of the tree <code>listNode</code>.
	 * 
	 * @param listNode
	 *        a node representing a list
	 * @param i
	 *        index of the son to delete
	 * @return the deleted son
	 */
	public synchronized native long destroy(long listNode, int i);
	
	/**
	 * Getting father
	 * 
	 * @param node
	 *        the address of the node
	 * @return the father of <code>node</code>
	 */
	public synchronized native long getFather(long node);
	
	/**
	 * Getting nil tree.
	 * 
	 * @return the nil tree representation
	 */
	public synchronized native long getNilTree();
	
	/**
	 * Inserts <code>node</code> at the left of the <code>i</code>-th son of <code>listNode</code>.
	 * 
	 * @param i
	 *        index of the element
	 * @param node
	 *        the new node to insert
	 * @param listNode
	 *        a node representing a list
	 */
	public synchronized native void insertLeft(int i, long node, long listNode);
	
	/**
	 * Inserts <code>node</code> at the right of the <code>i</code>-th son of <code>listNode</code>.
	 * 
	 * @param i
	 *        index of the element
	 * @param node
	 *        the new node to insert
	 * @param listNode
	 *        a node representing a list
	 */
	public synchronized native void insertRight(int i, long node, long listNode);
	
	/**
	 * Get the atomic tree integer value
	 * 
	 * @param node
	 *        an atomic tree node (assumed)
	 * @return the integer value denoted by <code>node</code>
	 */
	public synchronized native long intOf(long node);
	
	/**
	 * Test if the list is empty
	 * 
	 * @param node
	 *        an operand tree
	 * @return true if <code>node</code> is an empty tree.
	 */
	public synchronized native boolean isEmpty(long node);
	
	/**
	 * Testing nil tree.
	 * 
	 * @param node
	 *        the address of the node
	 * @return true if the operand tree <code>node</code> is the undefined empty tree.
	 */
	public synchronized native boolean isNilTree(long node);
	
	/**
	 * Getting a brother node
	 * 
	 * @param i
	 *        index of the brother to get
	 * @param node
	 *        a node
	 * @return the <code>i</code>-th left brother of <code>node</code>, niltree if it does not exist.
	 */
	public synchronized native long left(int i, long node);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @param node1
	 *        node used as first son
	 * @param node2
	 *        node used as second son
	 * @return a new (binary) tree with </code>opCode</code> as operator, and <code>node1</code> and <code>node2</code>
	 *         as sons.
	 */
	public synchronized native long makeBinary(int opCode, long node1, long node2);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @return a new (list) empty tree with </code>opCode</code> as operator.
	 */
	public synchronized native long makeEmptyList(int opCode);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @param elem
	 *        a node
	 * @return a new (list) tree with </code>opCode</code> as operator and </code>elem</code> as first element.
	 */
	public synchronized native long makeList(int opCode, long elem);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @return a new (nullary) tree with </code>opCode</code> as operator, and no sons.
	 */
	public synchronized native long makeNullary(int opCode);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @param node1
	 *        node used as first son
	 * @param node2
	 *        node used as second son
	 * @param node3
	 *        node used as third son
	 * @param node4
	 *        node used as fourth son
	 * @return a new (quaternary) tree with </code>opCode</code> as operator, and <code>node1</code>, <code>node2</code>
	 *         , <code>node3</code>, and <code>node4</code> as sons.
	 */
	public synchronized native long makeQuaternary(int opCode, long node1, long node2, long node3, long node4);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @param node1
	 *        node used as first son
	 * @param node2
	 *        node used as second son
	 * @param node3
	 *        node used as third son
	 * @return a new (ternary) tree with </code>opCode</code> as operator, and <code>node1</code>, <code>node2</code>,
	 *         and <code>node3</code> as sons.
	 */
	public synchronized native long makeTernary(int opCode, long node1, long node2, long node3);
	
	/**
	 * @param opCode
	 *        the code representing the operator used as root node
	 * @param node
	 *        node used as son
	 * @return a new (unary) tree with </code>opCode</code> as operator, and <code>node1</code> as son.
	 */
	public synchronized native long makeUnary(int opCode, long node);
	
	/**
	 * Modifies the value assigned to an atomic char tree node.
	 * 
	 * @param node
	 *        an atomic char tree node (assumed)
	 * @param c
	 *        the new character value
	 */
	public synchronized native void modifCharVal(long node, char c);
	
	/**
	 * Modifies the value assigned to an atomic integer tree node.
	 * 
	 * @param node
	 *        an atomic integer tree node (assumed)
	 * @param val
	 *        the new integer value
	 */
	public synchronized native void modifIntVal(long node, int val);
	
	/**
	 * Get the tree operator
	 * 
	 * @param node
	 *        a node
	 * @return the operator code of <code>node</code>
	 */
	public synchronized native int oper(long node);
	
	/**
	 * @param node
	 *        a node
	 * @return the rank of the operand tree <code>node</code>.
	 */
	public synchronized native int posit(long node);
	
	/**
	 * Adds an element node at the end of a list node
	 * 
	 * @param listNode
	 *        a node representing a list
	 * @param node
	 *        the node to add
	 * @return <code>listNode</code> in which <code>node</code> is added at the end of the list.
	 */
	public synchronized native long post(long listNode, long node);
	
	/**
	 * Adds an element node at the beginning of a list node
	 * 
	 * @param listNode
	 *        a node representing a list
	 * @param node
	 *        the node to add
	 * @return <code>listNode</code> in which <code>node</code> is added at the beginning of the list.
	 */
	public synchronized native long pre(long listNode, long node);
	
	/**
	 * Getting a brother node
	 * 
	 * @param i
	 *        index of the brother to get
	 * @param node
	 *        a node
	 * @return the <code>i</code>-th right brother of <code>node</code>, niltree if it does not exist.
	 */
	public synchronized native long right(int i, long node);
	
	/**
	 * Set the tree operator. <code>opCode</code> becomes the operator of <code>node</code>. No controls are done on the
	 * arity of <code>node</code>.
	 * 
	 * @param node
	 *        a node
	 * @param opCode
	 *        the new code of the operator
	 * @return the modified tree node
	 */
	public synchronized native long setOp(long node, int opCode);
	
	/**
	 * Get the atomic tree string value.
	 * 
	 * @param node
	 *        an atomic tree string node (assumed)
	 * @return the string denoted by <code>node</code>
	 */
	public synchronized native String stringOf(long node);
	
	/**
	 * Get the tree arity
	 * 
	 * @param node
	 *        a node
	 * @return the arity of <code>node</code>.
	 */
	public synchronized native int treeArity(long node);
	
	/**
	 * Removes <code>node</code> from the list containing it.
	 * 
	 * @param node
	 *        a node
	 */
	public synchronized native void treeDestroy(long node);
	
	/*******************************************************************************************************************
	 * persistent storage primitives
	 * 
	 * C function file : intetreesSavingLoading_g.c
	 ******************************************************************************************************************/
	
	/**
	 * Loading Tree from a file.
	 * 
	 * @param filename
	 *        the name of the file to load
	 * @return the loaded tree or niltree when an error occurred
	 */
	public synchronized native long treesLoad(String filename);
	
	/**
	 * Saving a Tree into a file.
	 * 
	 * @param filename
	 *        the name of the file in which the tree will be saved
	 * @param node
	 *        the tree node to save
	 * @return true if the operation is performed successfully, false otherwise.
	 */
	public synchronized native boolean treesSave(String filename, long node);
	
	/*******************************************************************************************************************
	 * attributes managing primitives
	 * 
	 * C function file : intetreesAttr_g.c
	 ******************************************************************************************************************/
	
	/**
	 * Get the link from a tree-attribute to tree.
	 * 
	 * @param node
	 *        a node
	 * @return a tree node at which the tree-attribute <code>node</code> is assigned.
	 */
	public synchronized native long attrCarrier(long e);
	
	/**
	 * Test if an attribute is assigned to a tree.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the attribute rank number
	 * @return true if an attribute with num as attribute number is assigned to the operand tree <code>node</code>,
	 *         false otherwise.
	 */
	public synchronized native boolean attrExist(long node, int num);
	
	/**
	 * Get the attribute value. The attribute to get is assumed to be of int type.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @return the int value of the <code>num</code>-th attribute assigned to <code>node</code>.
	 */
	public synchronized native int attrInt(long node, int num);
	
	/**
	 * Get the attribute value. The attribute to get is assumed to be of ntrees type.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @return the ntrees value of the <code>num</code>-th attribute assigned to <code>node</code>.
	 */
	public synchronized native long attrNtrees(long node, int num);
	
	/**
	 * Get the number of attributes.
	 * 
	 * @param node
	 *        a tree node
	 * @return the number of attributes assigned to <code>node</code>.
	 */
	public synchronized native int attrNumber(long node);
	
	/**
	 * Get the attribute value. The attribute to get is assumed to be a pointer.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @return the pointer value of the <code>num</code>-th attribute assigned to <code>node</code>.
	 */
	public synchronized native long attrPtr(long node, int num);
	
	/**
	 * Get the attribute value. The attribute to get is assumed to be a string.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @return the string value of the <code>num</code>-th attribute assigned to <code>node</code>.
	 */
	public synchronized native String attrStr(long node, int num);
	
	/**
	 * Removes, from the set of attributes assigned to <code>node</code>, the <code>num</code>-th attribute.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @return true if the <code>num</code>-th attribute was assigned to <code>node</code>, false otherwise.
	 */
	public synchronized native boolean destroyAttr(long node, int num);
	
	/**
	 * Assignes an integer attribute to a tree node.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @param val
	 *        integer value of the attribute
	 */
	public synchronized native void makeAttrInt(long node, int num, int val);
	
	/**
	 * Assignes a ntrees attribute to a tree node.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @param val
	 *        ntrees value of the attribute
	 */
	public synchronized native void makeAttrNtrees(long node, int num, long val);
	
	/**
	 * Assignes a pointer attribute to a tree node.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @param ptrVal
	 *        pointer value of the attribute
	 */
	public synchronized native void makeAttrPtr(long node, int num, long ptrVal);
	
	/**
	 * Assignes a string attribute to a tree node.
	 * 
	 * @param node
	 *        a tree node
	 * @param num
	 *        the rank of the attribute
	 * @param str
	 *        pointer value of the attribute
	 */
	public synchronized native void makeAttrStr(long node, int num, String str);
	
	/*******************************************************************************************************************
	 * garbaging functionallities
	 * 
	 * C function file : intetreesGarbage_g.c
	 ******************************************************************************************************************/
	
	/**
	 * Reseting tree memory.
	 */
	public synchronized native void globalMemoryReset();
	
	/**
	 * Reseting tree memory (user tree area and registers).
	 */
	public synchronized native void memoryReset();
	
	/**
	 * Sets the memory tree index (free area) to the index stored by the previous call to ntreesMemoryStoreIndex method.
	 */
	public synchronized native void memoryRestoreIndex();
	
	/**
	 * Store current index of the memory tree (free area). This index should be used to reset the memory using the
	 * ntrees_memoryRestoreIndex method. It allows to build trees from this point and then to delete them after printing
	 * for example.
	 */
	public synchronized native void memoryStoreIndex();
	
	/**
	 * Test if the size of available area in tree memory seems sufficient.
	 * 
	 * @return true if the garbage seems necessary, false otherwise.
	 */
	public synchronized native boolean needGarbage();
	
	/**
	 * This value represents the maximum length of a string that can be passed to the native part minus The 2
	 * double-quotes added at the beginning and at the end
	 */
	private final static int	STRING_MAX_LENGTH	= 254;
	
	/**
	 * @param enonce
	 * @return
	 */
	public long makePragmaEnonce(String enonce) {
		long enonceNode;
		
		if (enonce != null && "".equals(enonce) == false) {
			enonceNode = makeEmptyList(SignalOperators.opchai);
			StringTokenizer token = new StringTokenizer(enonce, "\n\r\f");
			while (token.hasMoreElements()) {
				String st = token.nextToken();
				int length = st.length();
				if (length > STRING_MAX_LENGTH) {
					while (length > 0) {
						if (length > STRING_MAX_LENGTH) {
							int i = st.lastIndexOf(" ", STRING_MAX_LENGTH);
							if (i == -1)
								i = STRING_MAX_LENGTH;
							enonceNode = post(enonceNode, atomString(SignalOperators.opcstcha, "\""
									+ st.substring(0, i) + "\""));
							st = st.substring(i);
							length = st.length();
						}
						else {
							enonceNode = post(enonceNode, atomString(SignalOperators.opcstcha, "\"" + st + "\""));
							length = 0;
						}
					}
				}
				else
					enonceNode = post(enonceNode, atomString(SignalOperators.opcstcha, "\"" + st + "\""));
			}
		}
		else
			enonceNode = getNilTree();
		return enonceNode;
	}
	
	public long string2ident(String s) {
		return atomString(SignalOperators.opnom, s);
	}
}
