/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony;

/**
 * This file contains all SignalTreeManager functions concerning the exploration of a Signal Tree. Conversions of type
 * between C and Java
 */
public class SignalTreeExploring {
	/**
	 * Get the expression of process (body) of a model.
	 * 
	 * @param modelNode
	 *        the address of a node that represents a model (Assumed).
	 * @return the address of the node representing the expression of process of modelNode.
	 */
	public synchronized native long getBody(long node);
	
	/**
	 * Getting a call in a body of a process.
	 * 
	 * @param name
	 *        an identifier representing the name of the called model (assumed).
	 * @param modelNode
	 *        a node representing a model (assumed) in which the search is done.
	 * @return the reference to the (first?) calling tree if it is any, PKTrees.ERROR_NODE otherwise.
	 */
	public synchronized native long getCallToIn(long name, long modelNode);
	
	public synchronized native DeclarationValue n_getDeclaration(long node, int kind);
	
	/**
	 * Get the father of the declaration tree of signals according to the kind parameter
	 * 
	 * @param modelNode
	 *        a Signal model (assumed)
	 * @param kind
	 *        the kind of signal/constant (local, parameter, input, or output)
	 * @return a DeclarationValue which contains the address of father of the declaration tree of signals according to
	 *         the kind parameter and the index of the son of the returned tree to select to have the kind-declaration
	 *         area.
	 */
	public synchronized DeclarationValue getDeclaration(long modelNode, IdentifierKind kind) {
		return n_getDeclaration(modelNode, kind.getValue());
	}
	
	/**
	 * Returns the declaration of the signal/constant without its type from a complete declaration. For a list of
	 * declarations, it takes the first element of the list.
	 * 
	 * @param declNode
	 *        the node representing the declaration
	 * @return a node which contains the declaration without the type
	 */
	public synchronized native long getDeclarationWithoutType(long node);
	
	/**
	 * Returns the description of a model: process, function, action, node
	 * 
	 * @param modelNode
	 *        the model from which we want the description
	 * @return the description of a model.
	 */
	public synchronized native long getDescription(long node);
	
	/**
	 * Returns the (first) string of "external" tree.
	 * 
	 * @param externalNode
	 *        the external node
	 * @return the string of "external" tree.
	 */
	public synchronized native String getExternalValue(long node);
	
	/**
	 * Returns the string associated with a signal (p) when it is an identifier, or an access to an element of array or
	 * an access to a field of a structure, or null otherwise.
	 * <ul>
	 * <li>For X, it returns X</li>
	 * <li>For X.Y, it returns X</li>
	 * <li>For X[ ], it returns X</li>
	 * </ul>
	 * 
	 * @param idNode
	 *        the node representing the identifier
	 * @return a string, or null otherwise.
	 */
	public synchronized native String getIdentStr(long node);
	
	/**
	 * Returns the expression of the initial value associated with a signal declaration.
	 * 
	 * @param declNode
	 *        the node representing the signal declaration.
	 * @return the expression of the initial value associated with declNode or niltree otherwise.
	 */
	public synchronized native long getInitExpr(long node);
	
	/**
	 * Returns the list of the inputs of a model.
	 * 
	 * @param modelNode
	 *        the node representing a model (Assumed)
	 * @return the list of the inputs of modelNode.
	 */
	public synchronized native long getInputs(long modelNode);
	
	/**
	 * Returns the tree composed of the inputs/outputs of model.
	 * 
	 * @param modelNode
	 *        the node representing a model (Assumed)
	 * @return the tree composed of the inputs/outputs of modelNode.
	 */
	public synchronized native long getInputsOutputs(long modelNode);
	
	/**
	 * Returns the interface node of a model.
	 * 
	 * @param modelNode
	 *        the node representing a model (Assumed)
	 * @return the interface of modelNode
	 */
	public synchronized native long getInterface(long modelNode);
	
	/**
	 * Returns the i-th local model of a model
	 * 
	 * @param modelNode
	 *        the node representing a model (Assumed)
	 * @param i
	 *        the index of the model
	 * @return the i-th local model of modelNode if it exists, null otherwise.
	 */
	public synchronized native long getIthModel(long modelNode, int index);
	
	/**
	 * Returns the local declarations of a model.
	 * 
	 * @param modelNode
	 *        the node representing a model (Assumed)
	 * @return the local declarations of modelNode
	 */
	public synchronized native long getLocalDeclarations(long modelNode);
	
	/**
	 * Returns the father trees of a process. If the process is directly under a module, it returns the module. If such
	 * object does not exist, it return niltree.
	 * 
	 * @param modelNode
	 *        the node representing a model (Assumed)
	 * @return the father trees of modelNode, if it exists, null otherwise.
	 */
	public synchronized native long getModelFather(long modelNode);
	
	/**
	 * Get the model in which the operand tree is.
	 * 
	 * @param node
	 *        the operand tree node
	 * @return the model in which node is. If such object does not exist, it returns niltree.
	 */
	public synchronized native long getModelInWhich(long node);
	
	/**
	 * Returns the name of a model as a node.
	 * 
	 * @param node
	 *        the operand tree node
	 * @return the name of node if node is a model, niltree otherwise.
	 */
	public synchronized native long getModelName(long node);
	
	/**
	 * Returns the name of a model as a string.
	 * 
	 * @param node
	 *        the operand tree node
	 * @return the string representing the name of the model.
	 */
	public synchronized native String getModelNameStr(long node);
	
	/**
	 * Getting name of a model for a call.
	 * 
	 * @param calledNode
	 *        a call object (assumed).
	 * @return the name of callNode
	 */
	public synchronized native long getNameCalledModel(long callNode);
	
	/**
	 * Returns the name of the signal/constant from a declaration. For a list of declarations, it takes the first
	 * element of the list.
	 * 
	 * @param declNode
	 *        a declaration node (assumed).
	 * @return the name of the signal/constant from <code>declNode</code>.
	 */
	public synchronized native long getNameOfDeclaration(long declNode);
	
	/**
	 * Returns the number of inputs of a model.
	 * 
	 * @param modelNode
	 *        a node representing a process (assumed).
	 * @return the number of inputs of a <code>modelNode</code>.
	 */
	public synchronized native int getNbInputs(long modelNode);
	
	/**
	 * Returns the number of outputs of a model.
	 * 
	 * @param modelNode
	 *        a node representing a process (assumed).
	 * @return the number of outputs of a <code>modelNode</code>.
	 */
	public synchronized native int getNbOutputs(long modelNode);
	
	/**
	 * Returns the number of parameters of a model.
	 * 
	 * @param modelNode
	 *        a node representing a process (assumed).
	 * @return the number of parameters of a <code>modelNode</code>.
	 */
	public synchronized native int getNbParameters(long modelNode);
	
	/**
	 * Returns the NIL tree representation.
	 * 
	 * @return the NIL tree representation.
	 */
	public synchronized native long getNilTree();
	
	/**
	 * Returns the arity of an operator.
	 * 
	 * @param op
	 *        SIGNAL operator
	 * @return the arity of the SIGNAL operator <code>op</code>.
	 */
	public synchronized native int getOperatorArity(int opValue);
	
	/**
	 * Returns the list of the outputs of a model.
	 * 
	 * @param modelNode
	 *        a model (assumed).
	 * @return the list of the outputs of <code>modelNode</code>.
	 */
	public synchronized native long getOutputs(long modelNode);
	
	/**
	 * Returns the list of the parameters of a model.
	 * 
	 * @param modelNode
	 *        a model (assumed).
	 * @return the list of the parameters of <code>modelNode</code>.
	 */
	public synchronized native long getParameters(long modelNode);
	
	/**
	 * Returns the list of the pragmas of a model.
	 * 
	 * @param modelNode
	 *        a model (assumed).
	 * @return the list of the pragmas of <code>modelNode</code>.
	 */
	public synchronized native long getPragmas(long modelNode);
	
	/**
	 * Getting the definition of a pragma.
	 * 
	 * @param pragmaNode
	 *        the node representing a pragma (Assumed).
	 * @return the definition (name, list of affected objects, and value) of pragmaNode.
	 */
	public synchronized native PragmaValues getPragmaValues(long pragmaNode);
	
	/**
	 * Getting root of a tree.
	 * 
	 * @param node
	 *        a node in the AST.
	 * @return the root of the tree if node is not a nil tree, PKTrees.ERROR_NODE otherwise.
	 */
	public synchronized native long getRootTree(long node);
	
	/**
	 * Returns the external graph of a model.
	 * 
	 * @param modelNode
	 *        a node representing a model (assumed).
	 * @return the external graph of <code>modelNode</code>.
	 */
	public synchronized native long getSpec(long modelNode);
	
	/**
	 * Returns the value of type for a declaration of a signal/constant or a list declarations (in this case, it takes
	 * the first element of the list). The value of a type contains the operator and the node representing the type.
	 * 
	 * @param declNode
	 *        a node assumed to be a declaration or a list of declarations of signals/constants.
	 * @return a type value
	 */
	public synchronized native TypeValue getType(long declNode);
}
