/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony;

/**
 * This file contains all SignalTreeManager functions for testing any node of the AST. Conversions of type between C and
 * Java
 */
public class SignalTreeTesting {
	/**
	 * Test if the node of the AST represents an action model object.
	 * 
	 * @param node
	 *        the address of the node
	 * @return true if the node represents an action, false otherwise.
	 */
	public synchronized native boolean isAction(long node);
	
	/**
	 * Test if the node of the AST is the body of a model object.
	 * 
	 * @param node
	 *        the address of the node
	 * @return true if the node is the body of a model object, false otherwise.
	 */
	public synchronized native boolean isBodyTree(long node);
	
	/**
	 * Test if the node of the AST is a list of declaration of constants.
	 * 
	 * @param node
	 *        the address of the node
	 * @return true if the node is a list of declaration of constants, false otherwise.
	 */
	public synchronized native boolean isConstantDecl(long node);
	
	/**
	 * Test if the node of the AST is a list of effective parameters.
	 * 
	 * @param node
	 *        the address of the node
	 * @return true if the node is a list of effective parameters, false otherwise.
	 */
	public synchronized native boolean isEffectiveParameter(long node);
	
	/**
	 * Test if the <code>opValue</code> parameter is a Signal process expression operator
	 * 
	 * @param opValue
	 *        the value of the operator
	 * @return true if opValue represents a Signal process expression operator, false otherwise
	 */
	public synchronized native boolean isExprProc(int opValue);
	
	/**
	 * Test if the <code>node</code> is an extern model. An extern model has no expression and no local declarations.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true if the node is an extern model, false otherwise.
	 */
	public synchronized native boolean isExternModel(long node);
	
	/**
	 * Test if the <code>node</code> represents a function model object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true if the node is a function mode object, false otherwise.
	 */
	public synchronized native boolean isFunction(long node);
	
	/**
	 * Test if <code>node1</code> and <code>node2</code> are based on the same identifier
	 * 
	 * @param node1
	 *        the address of a node
	 * @param node2
	 *        the address of another node
	 * @return true if the two nodes are based on the same identifier, false otherwise.
	 */
	public synchronized native boolean isIdentEqual(long node1, long node2);
	
	/**
	 * Test if the <code>node</code> represents an implicit type object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true if the node represents an implicit type object, false otherwise.
	 */
	public synchronized native boolean isImplicitTypeTree(long node);
	
	/**
	 * Test if an identifier <code>idNode</code> is in the list of declarations <code>decListNode</code>.
	 * 
	 * @param decListNode
	 *        the address of the declaration list node (ntree structure)
	 * @param idNode
	 *        the address of the identifier node (ntree structure)
	 * @return true if an identifier (idNode) is in a list of declarations(decListNode), false otherwise.
	 */
	public synchronized native boolean isInDeclList(long declListNode, long node);
	
	/**
	 * Test if the <code>node</code> represents an input/output specification object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true if the node represents an input/output specification object.
	 */
	public synchronized native boolean isInputOutput(long node);
	
	/**
	 * Test if the node of the AST represents an interface
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true if the node is an interface object, false otherwise.
	 */
	public synchronized native boolean isInterface(long node);
	
	/**
	 * Test if the node of the AST represents a list of objects
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true, if it <code>node</code> is a list of objects, false otherwise.
	 */
	public synchronized native boolean isList(long node);
	
	/**
	 * Test if the <code>node</code> denotes a local declaration list.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) in the AST
	 * @return true if <code>node</code> denotes a list of local declarations, false otherwise.
	 */
	public synchronized native boolean isLocalDeclList(long node);
	
	/**
	 * Test if the <code>node</code> represents a model(process, node, function, action) object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) in the AST
	 * @return true if the node represents a model object, false otherwise.
	 */
	public synchronized native boolean isModel(long node);
	
	/**
	 * Retrieve the model declaration in a list of declarations. The search is based on the name of the model.
	 * 
	 * @param name
	 *        the address of a node representing an identifier (assumed).
	 * @param modelList
	 *        the address of a list of declarations (models, signals,...) (assumed).
	 * @return the address of the model node when it exists, PKTrees.ERROR_NODE otherwise.
	 */
	public synchronized native long isModelInList(long name, long declList);
	
	/**
	 * Test if the <code>node</code> represents a model object node without parameters.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) representing a model or a module. In this last case, it returns
	 *        always true.
	 * @return true if there is no parameters associated to the node, false otherwise.
	 */
	public synchronized native boolean isModelWithoutParameters(long node);
	
	/**
	 * Test if the node is a module.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is a module, false otherwise.
	 */
	public synchronized native boolean isModule(long node);
	
	/**
	 * Test if the node is a list of modules.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is a list of modules, false otherwise.
	 */
	public synchronized native boolean isModulesList(long node);
	
	/**
	 * Test if the node is the name of a model.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is the name of a model, false otherwise.
	 */
	public synchronized native boolean isNameOfModel(long node);
	
	/**
	 * Test if the <code>node</code> represents a node model object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) of the AST
	 * @return true if the node is a node mode object, false otherwise.
	 */
	public synchronized native boolean isNode(long node);
	
	/**
	 * Test if the node is an operator model.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is an operator model, false otherwise.
	 */
	public synchronized native boolean isOperatorModel(long node);
	
	/**
	 * Test if the <code>no</code>-th element of the list of parameters is a type.
	 * 
	 * @param paramListNode
	 *        the address of the node (ntree structure) representing the list of parameters (assumed).
	 * @param no
	 *        the rank at which the test is done
	 * @return true if the no-th element of paramListNode is a type, false otherwise.
	 */
	public synchronized native boolean isParamAtype(long paramDeclList, int no);
	
	/**
	 * Test if <code>node</code> is the parameter list declaration of a process.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is the parameter list declaration of a process, false otherwise.
	 */
	public synchronized native boolean isParametersOfModel(long node);
	
	/**
	 * Test if <code>node</code> is a list of pragmas or a nil tree.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if node is a list of pragmas or nil tree, false otherwise.
	 */
	public synchronized native boolean isPragmas(long node);
	
	/**
	 * Test if the <code>node</code> represents a process model object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure) in the AST
	 * @return true if the node represents a process model object, false otherwise.
	 */
	public synchronized native boolean isProcess(long node);
	
	/**
	 * Test if the <code>node</code> represents a private model(process, node, function, action) object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node represents a private model object, false otherwise.
	 */
	public synchronized native boolean isPrivateDeclaration(long node);
	
	/**
	 * Test if <code>node</code> is a list of declaration of signals.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is a list of declaration of signals, false otherwise.
	 */
	public synchronized native boolean isSignalDecl(long node);
	
	/**
	 * Test if the <code>node</code> is an external graph specification object.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if node is an external graph specification object, false otherwise.
	 */
	public synchronized native boolean isSpec(long node);
	
	/**
	 * Test if <code>node</code> is a list of declaration of types.
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is a list of declaration of types, false otherwise.
	 */
	public synchronized native boolean isTypeDecl(long node);
	
	/**
	 * Test if the <code>opNum</code> parameter is a unary expression
	 * 
	 * @param opValue
	 *        the value of the operator constant
	 * @return true if the parameter is an unary operator, false otherwise.
	 */
	public synchronized native boolean isUnaryExprProc(int opValue);
	
	/**
	 * Test if the node is an undefined object. An undefined object is either:
	 * <ul>
	 * <li>a nil tree,</li>
	 * <li>or a tree with opnil operator,</li>
	 * <li>or a tree with opnullproc operator</li>
	 * </ul>
	 * 
	 * @param node
	 *        the address of the node (ntree structure)
	 * @return true if the node is an undefined object, false otherwise.
	 */
	public synchronized native boolean isUndefinedTree(long node);
	
	/**
	 * Test if the two nodes are equals. This function is not complete (it does not test all the Signal operators).
	 * 
	 * @param node1
	 *        a node
	 * @param node2
	 *        another node
	 * @return true if the both nodes are equals. This function is not complete (does not test all the Signal
	 *         operators).
	 */
	public synchronized native boolean isTreeEq(long node1, long node2);
}
