/*******************************************************************************
 * Copyright (c) 2010, 2014 INRIA-CNRS (Espresso/TEA team).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Loic Besnard, Francois Fabre, Thierry Gautier: Initial API and implementation and/or initial documentation
 */

package org.eclipse.pop.ssme.polychrony.traceability;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.pop.ssme.polychrony.PKTrees;
import org.eclipse.pop.ssme.polychrony.PkPlugin;
import org.eclipse.pop.ssme.polychrony.SignalOperators;
import org.eclipse.pop.ssme.polychrony.utils.PolychronyDiagnostic;

/**
 * In this class is described the interface map between the native Polychrony AST and the SSME AST.
 */
public class TreesInterface {
	
	/** Interface HashMap between the Polychrony native AST and the SSME AST */
	private Map<EObject, Long>	interfaceMapSSMESignal;
	
	/** List containing all the SME Nodes with their eventual error message */
	private List<SSMENode>			SSMENodesList;
	
	public TreesInterface() {
		interfaceMapSSMESignal = new HashMap<EObject, Long>();
	}
	
	public List<SSMENode> getSMENodesList() {
		return SSMENodesList;
	}
	
	public Map<EObject, Long> getInterfaceMapSMESignal() {
		return interfaceMapSSMESignal;
	}
	
	public void clearMap() {
		interfaceMapSSMESignal.clear();
	}
	
	/**
	 * Creation of the SME nodes list with all the nodes of the interface hashmap
	 */
	public void initializeSSMENodesList() {
		SSMENodesList = new ArrayList<SSMENode>();
		
		for (Entry<EObject, Long> entryMap : interfaceMapSSMESignal.entrySet()) {
			SSMENode smeNode = new SSMENode();
			smeNode.setNode(entryMap.getKey());
			SSMENodesList.add(smeNode);
		}
	}
	
	/**
	 * This method prints the informations key => value and value => key(s) of the interface hashmap between SME and
	 * Signal.
	 * 
	 * @param debug
	 *        set it to true if you want to display in the console the values (representing the Signal nodes in the AST)
	 *        corresponding to the EObjectImpl keys in the interface hashmap between SME and Signal.
	 * @param twosides
	 *        set it to true if you want to display in the console the EObjectImpl SME nodes corresponding to the Signal
	 *        nodes of the hashmap.
	 */
	public void printInformations(final boolean debug) {
		if (debug == true) {
			StringBuffer buffer = new StringBuffer();
			buffer.append("\n-------------------------------------------------------------------\n\n");
			buffer.append("Interface HashMap between the Polychrony native AST and the SSME AST\n\n");
			buffer.append("Hashmap size : " + interfaceMapSSMESignal.size() + "\n\n");
			
			for (Entry<EObject, Long> map : interfaceMapSSMESignal.entrySet()) {
				EObject key = map.getKey();
				long value = map.getValue();
				buffer.append(key.toString() + "\n" + value + "\n\n");
			}
			buffer.append("\n\n");
			
			System.out.println(buffer.toString());
		}
	}
	
	/**
	 * This methods displays a tree from its root, with the level of each node in the hierarchy.
	 * 
	 * @param node
	 *        the current node to display
	 * @param level
	 *        the level of the current node
	 * @param debug
	 *        if false, nothing is printed on the screen
	 */
	public void printTree(final long node, final int level, final boolean debug) {
		if (debug) {
			if (level == 0) {
				System.out.println("\nAbstract Syntax Tree :\n");
			}
			PKTrees API = PkPlugin.getTreeAPI();
			System.out.print("LEVEL " + level + " : ");
			for (int i = 0; i < level; i++) {
				System.out.print("......");
			}
			if (node == PKTrees.ERROR_NODE) {
				System.out.println("(error)");
			}
			if (node == API.getNilTree()) {
				System.out.println("(nil)");
			}
			else {
				System.out.print("(node : " + node + ", arity: " + API.treeArity(node) + ")");
				if (getKey(node) != null) // if there is a SSME object corresponding to the node
				{
					System.out.println(" ==> " + getKey(node));
				}
				else {
					System.out.println();
				}
				if (API.oper(node) >= SignalOperators.opunary) {
					for (int i = 1; i <= API.treeArity(node); i++) {
						long child = API.child(i, node);
						printTree(child, level + 1, debug);
					}
				}
				else {
					System.out.print("LEVEL " + (level + 1) + " : ");
					for (int i = 0; i < level + 1; i++) {
						System.out.print("......");
					}
					System.out.println("(node : " + API.intOf(node) + ")");
				}
			}
		}
	}
	
	/**
	 * This methods prints the errors on the SSME model.
	 * 
	 * @param tree
	 *        the AST representing the diagram
	 * @param debug
	 *        prints the debug information if true
	 */
	public void printErrorsOnModel(final long tree, final boolean debug) {
		PKTrees API = PkPlugin.getTreeAPI();
		
		/*
		 * The error tree is given by the native getAnnotations method
		 */
		long errorTree = PkPlugin.getCommentAPI().getAnnotations(tree, API.getNilTree());
		if (errorTree != API.getNilTree()) {
			/*
			 * For each child of the error tree, the intOf method gets the integer value corresponding to the node
			 */
			for (int i = 1; i <= API.treeArity(errorTree); i++) {
				long errorTreeChild = API.child(i, errorTree);
				long intOfErrorTreeChild = API.intOf(errorTreeChild);
				
				/*
				 * The annotations tree is given by the getAnnotation method, from the integer value of each child of
				 * the error tree. The annotations tree is a tree of strings representing error or warning messages.
				 */
				long annotationsTree = PkPlugin.getCommentAPI().getAnnotation(intOfErrorTreeChild);
				if (debug) {
					System.out.println("\n-----------------");
					System.out.println("CORRESPONDENCE " + i + ":");
					System.out.println("-----------------\n");
					System.out.println("Node address : " + intOfErrorTreeChild + "\n");
				}
				
				/*
				 * The getKey method gets the nodes of the Signal tree in the SME graphical world.
				 */
				EObject smeObject = getKey(intOfErrorTreeChild);
				
				if (debug) {
					if (smeObject == null) {
						System.out.println("No correspondence in the SSME nodes table\n");
					}
					else {
						System.out.println("Associated SSME node :\n\t" + smeObject + "\n");
					}
				}
				
				/*
				 * For each child of the annotation tree, the message is given by the stringOf method, and the add
				 * method of the PolychronyDiagnostic class allows us to represent this error on the SME diagram.
				 */
				for (int j = 1; j <= API.treeArity(annotationsTree); j++) {
					long annotationsTreeChild = API.child(j, annotationsTree);
					String annotation = API.stringOf(annotationsTreeChild);
					if (debug) {
						System.out.println("Error / Warning message : " + annotation + "\n");
					}
					if (smeObject != null && !annotation.matches("(.)*cf no-((\\d)+)")) {
						annotation = annotation.replaceAll(" \\(ref-(.)*\\)", "");
						
						for (SSMENode smeNode : SSMENodesList) {
							if (smeNode.getNode().equals(smeObject)) {
								smeNode.setError(true);
								smeNode.setErrorMessage(annotation);
							}
						}
						
						PolychronyDiagnostic.add(Diagnostic.ERROR, annotation, smeObject);
					}
				}
			}
		}
	}
	
	/**
	 * This method returns the key corresponding to a value in the map (In the case where only one key is corresponding
	 * to a value)
	 * 
	 * @param value
	 *        the value whose corresponding key we want
	 * @return the corresponding key
	 */
	public EObject getKey(final long value) {
		EObject key = null;
		for (Entry<EObject, Long> entryMap : interfaceMapSSMESignal.entrySet()) {
			EObject currentKey = entryMap.getKey();
			long currentValue = entryMap.getValue();
			if (currentValue == value) {
				key = currentKey;
			}
		}
		return key;
	}
	
	 
}
