/**
 * Copyright (c) 2013 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick Gottschaemmer - initial API and implementation.
 */
package org.eclipse.recommenders.livedoc.cli;

import static org.eclipse.recommenders.utils.Constants.EXT_JAR;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.util.artifact.SubArtifact;
import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.eclipse.recommenders.livedoc.ILivedoc;
import org.eclipse.recommenders.livedoc.Livedoc;
import org.eclipse.recommenders.livedoc.cli.aether.RepositoryClient;
import org.eclipse.recommenders.livedoc.cli.args4j.CliOptions;
import org.eclipse.recommenders.utils.Urls;
import org.eclipse.recommenders.livedoc.utils.LivedocChecks;
import org.eclipse.recommenders.livedoc.utils.LivedocPreconditionException;
import org.eclipse.recommenders.utils.Zips;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.ConsoleAppender;

/**
 * This class controls all aspects of the application's execution
 */
public class Application implements IApplication {

    private static final File TEMP_DIR = new File(FileUtils.getTempDirectory(), "livedoc");
    private static final File SOURCES_TEMP_DIR = new File(TEMP_DIR, "sources");
    private static final File JAVADOC_TEMP_DIR = new File(TEMP_DIR, "javadoc");
    private static final File SOURCEREPO_CACHE_DIR = new File(TEMP_DIR, "sourceRepo/cache");
    private static final org.slf4j.Logger LOG = LoggerFactory.getLogger(Application.class);

    private CliOptions settings;
    private RepositoryClient repoClient;

    // TODO Using a "real", non-zero error code causes a popup window.
    private static final Integer EXIT_ERROR = IApplication.EXIT_OK;

    @Override
    public Object start(IApplicationContext context) throws Exception {
        try {
            LivedocChecks.preconditions(true);

            initializeSettings(getArguments(context));

            configureLogback();

            prepareTempDirectory();

            initializeRepositoryClient();

            Artifact sourceArtifact = downloadSourcesArtifact(settings.getCoordinate(), settings.getSourcesRepository());

            List<Artifact> dependencies = resolveDependencies(sourceArtifact);

            String outPutFileName = outputName(sourceArtifact);

            File sourcesDir = unpackSources(sourceArtifact.getFile());

            File tmpOutput = new File(JAVADOC_TEMP_DIR, outPutFileName);

            Artifact modelCoordinate = settings.getModelCoordinate() != null ? settings.getModelCoordinate() : settings
                    .getCoordinate();

            generateJavaDoc(modelCoordinate, sourcesDir, tmpOutput, dependencies);

            File output = new File(settings.getOutputDir(), createDirectoryHierarchy(sourceArtifact));
            output.mkdirs();
            FileUtils.copyDirectory(tmpOutput, output);

            if (settings.isJarOutput() || settings.getJavadocRepository() != null) {
                File jarFile = jarOutput(tmpOutput);

                if (settings.isJarOutput()) {
                    FileUtils.copyFileToDirectory(jarFile, settings.getOutputDir());
                }

                if (settings.getJavadocRepository() != null) {
                    uploadJavadocArtifact(sourceArtifact, jarFile, settings.getJavadocRepository());
                }
            }

            System.out.println("Done.");
            return IApplication.EXIT_OK;
        } catch (IllegalArgumentException | IOException | LivedocPreconditionException e) {
            System.out.println("Failed.");
            return EXIT_ERROR;
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            System.out.println("Failed.");
            return EXIT_ERROR;
        }
    }

    private List<Artifact> resolveDependencies(Artifact sourceArtifact) {

        if (settings.noDependencies()) {
            return Collections.emptyList();
        }
        try {
            if (settings.isTransitive()) {
                return repoClient.resolveTransitiveDependencies(sourceArtifact, settings.getSourcesRepository());
            } else {
                return repoClient.resolveDirectDependencies(sourceArtifact, settings.getSourcesRepository());
            }
        } catch (IOException e) {
            LOG.error("Could not resolve any dependencies of source artifact {}", sourceArtifact, e);
            return Collections.emptyList();
        }
    }

    private void initializeSettings(String... args) throws IllegalArgumentException {
        CliOptions settings = new CliOptions();
        CmdLineParser parser = new CmdLineParser(settings);
        parser.setUsageWidth(100);
        try {
            parser.parseArgument(args);
            this.settings = settings;
        } catch (CmdLineException e) {
            System.err.println(e.getMessage());
            System.err.println();

            System.err.println("SYNPOSIS:");
            System.err.print(" ./livedoc");
            parser.printSingleLineUsage(System.err);
            System.err.println();
            System.err.println();

            System.err.println("PARAMETERS:");
            parser.printUsage(System.err);

            throw new IllegalArgumentException(e);
        }
    }

    private void configureLogback() {

        LoggerContext lc = (LoggerContext) LoggerFactory.getILoggerFactory();
        Logger logger = lc.getLogger(Logger.ROOT_LOGGER_NAME);

        ConsoleAppender<ILoggingEvent> appender = new ConsoleAppender<ILoggingEvent>();
        appender.setContext(lc);
        appender.setName("STDOUT");
        PatternLayoutEncoder pl = new PatternLayoutEncoder();
        pl.setContext(lc);
        StringBuilder pattern = new StringBuilder("%d{HH:mm:ss.SSS} [%thread] %-5level %logger{36} - %msg%n");

        if (!settings.isDebug()) {
            pattern.append("%exceptionMessage");
            pattern.append("%nopex");
        }

        pl.setPattern(pattern.toString());
        pl.start();

        appender.setEncoder(pl);
        appender.start();
        logger.addAppender(appender);
    }

    private void initializeRepositoryClient() throws IOException {
        File cacheDir = new File(SOURCEREPO_CACHE_DIR, Urls.mangle(settings.getSourcesRepository()));
        cacheDir.mkdirs();
        repoClient = new RepositoryClient(cacheDir);
    }

    private String[] getArguments(IApplicationContext context) {
        String[] args = (String[]) context.getArguments().get(IApplicationContext.APPLICATION_ARGS);

        // Drop bogus "-showlocation" argument
        if (args.length > 0 && args[0].equalsIgnoreCase("-showlocation")) {
            args = Arrays.copyOfRange(args, 1, args.length);
        }

        return args;
    }

    private Artifact downloadSourcesArtifact(Artifact coordinate, URL remoteRepository) throws IOException {
        if (settings.isVerbose()) {
            System.out.printf("Downloading sources for artifact \"%s\".\n", coordinate);
        }

        try {
            Artifact sourcesCoordinate = new SubArtifact(coordinate, "sources", EXT_JAR);
            return repoClient.download(sourcesCoordinate, remoteRepository);
        } catch (IOException e) {
            System.err.printf("Failed to download sources artifact for \"%s\" (%s).\n", coordinate, e);
            throw e;
        }
    }

    private void uploadJavadocArtifact(Artifact sourceArtifact, File artifactFile, URL remoteRepository)
            throws IOException {
        if (settings.isVerbose()) {
            System.out.printf("Uploading Javadoc for artifact \"%s\".\n", sourceArtifact);
        }

        try {
            Artifact uploadArtifact = new SubArtifact(sourceArtifact, "javadoc", EXT_JAR);
            uploadArtifact = uploadArtifact.setFile(artifactFile);
            repoClient.upload(uploadArtifact, remoteRepository);
        } catch (IOException e) {
            System.err.printf("Failed to upload Javadoc artifact for \"%s\" (%s).\n", sourceArtifact, e);
            throw e;
        }
    }

    private File jarOutput(File directory) throws IOException {

        String jarFileName = new StringBuffer(directory.getName()).append(".jar").toString();

        File output = new File(directory.getParentFile() + File.separator + jarFileName);
        Zips.zip(directory, output);
        return output;
    }

    private String createDirectoryHierarchy(Artifact sourceArtifact) {

        StringBuffer sb = new StringBuffer();
        sb.append(sourceArtifact.getGroupId().replace(".", File.separator)).append(File.separator)
                .append(sourceArtifact.getArtifactId()).append(File.separator).append(sourceArtifact.getBaseVersion());
        return sb.toString();
    }

    private void generateJavaDoc(Artifact modelCoordinate, File sourcesDir, File outputDir, List<Artifact> dependencies)
            throws IOException, LivedocPreconditionException {

        List<String> subpackages;
        if (settings.getSubpackages() != null) {
            subpackages = settings.getSubpackages();
        } else {
            subpackages = filterSourceFiles(sourcesDir);
        }

        ILivedoc livedoc = new Livedoc(sourcesDir, outputDir, subpackages, modelCoordinate.getGroupId(),
                modelCoordinate.getArtifactId(), modelCoordinate.getVersion());

        livedoc.setVerbose(settings.isVerbose());
        livedoc.setHighlight(settings.isHighlight());
        livedoc.setProviderArguments(settings.getProviderArguments());
        livedoc.setSplitIndex(settings.isSplitIndex());
        livedoc.setModelRepositories(settings.getModelsRepositories());
        livedoc.setEncoding(settings.getCharset());
        livedoc.setDoclintOptions(settings.getDoclintOptions());

        for (Artifact artifact : dependencies) {
            livedoc.addAdditonalClasspathEntry(artifact.getFile());
        }

        livedoc.generate();
    }

    private String outputName(Artifact artifact) {
        StringBuffer sb = new StringBuffer(artifact.getArtifactId());
        sb.append("-");
        sb.append(artifact.getVersion());
        sb.append("-javadoc");
        return sb.toString();
    }

    private List<String> filterSourceFiles(File sourceFiles) {
        List<File> subpackages = Arrays.asList(sourceFiles.listFiles(new FileFilter() {

            @Override
            public boolean accept(File file) {

                // TODO: Performance tweak: don't use FileUtils.listFiles
                return !file.getName().equals("META-INF") && file.isDirectory()
                        && !FileUtils.listFiles(file, new String[] { "java" }, true).isEmpty();
            }
        }));

        List<String> result = new ArrayList<String>(subpackages.size());

        for (File file : subpackages) {
            result.add(file.getName());
        }
        return result;
    }

    private File unpackSources(File sourcesJar) throws IOException {
        String fileName = StringUtils.removeEnd(sourcesJar.getName(), ".jar");
        File destFolder = new File(SOURCES_TEMP_DIR, fileName);
        if (!settings.isResume()) {
            Zips.unzip(sourcesJar, destFolder);
        }
        return destFolder;
    }

    private void prepareTempDirectory() throws Exception {

        if (settings.isResume()) {
            return;
        }

        SOURCEREPO_CACHE_DIR.mkdirs();

        try {
            if (JAVADOC_TEMP_DIR.exists()) {
                FileUtils.cleanDirectory(JAVADOC_TEMP_DIR);
            } else {
                JAVADOC_TEMP_DIR.mkdirs();
            }
            if (SOURCES_TEMP_DIR.exists()) {
                FileUtils.cleanDirectory(SOURCES_TEMP_DIR);
            } else {
                SOURCES_TEMP_DIR.mkdirs();
            }
        } catch (IOException e) {
            LOG.error("Couldn't clear livedoc tmp folders. \n", e);
            throw e;
        }
    }

    @Override
    public void stop() {
    }
}
