/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.providers.overrides;

import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.extractTypeName;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.htmlLink;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.listParameterTypes;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.strong;
import static org.eclipse.recommenders.utils.Checks.ensureIsInRange;
import static org.eclipse.recommenders.utils.Constants.CLASS_OVRP_MODEL;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.eclipse.recommenders.apidocs.ClassOverridePatterns;
import org.eclipse.recommenders.apidocs.MethodPattern;
import org.eclipse.recommenders.apidocs.SingleZipOverridePatternsModelProvider;
import org.eclipse.recommenders.livedoc.providers.AbstractLiveDocProvider;
import org.eclipse.recommenders.livedoc.providers.LiveDocProviderException;
import org.eclipse.recommenders.livedoc.providers.ProviderOutput;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelProvider;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.IUniqueName;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.models.UniqueTypeName;
import org.eclipse.recommenders.utils.names.IMethodName;
import org.eclipse.recommenders.utils.names.ITypeName;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.sun.javadoc.ClassDoc;

public class MethodOverridePatternsProvider extends AbstractLiveDocProvider<MethodOverridePatternsConfiguration> {

    private IModelProvider<IUniqueName<ITypeName>, ClassOverridePatterns> modelProvider;
    private Optional<ClassOverridePatterns> model;

    @Override
    public String getId() {
        return "ovrp";
    }

    @Override
    public void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException {
        super.setUp(pc, repo, index);

        Optional<File> modelArchive = fetchModelArchive(CLASS_OVRP_MODEL);

        if (!modelArchive.isPresent()) {
            throw new LiveDocProviderException(String.format("No %s model available for given coordinate.", getId()));
        }

        modelProvider = new SingleZipOverridePatternsModelProvider(modelArchive.get());

        try {
            modelProvider.open();
        } catch (IOException e) {
            throw new LiveDocProviderException(String.format("Exception while opening model provider for %s models.",
                    getId()), e);
        }
    }

    @Override
    public void beginClass(ClassDoc newClassDoc) {
        ITypeName typeName = extractTypeName(newClassDoc);
        UniqueTypeName uniqueTypeName = new UniqueTypeName(getProjectCoordinate(), typeName);
        model = modelProvider.acquireModel(uniqueTypeName);
    }

    @Override
    public void endClass(ClassDoc oldClassDoc) {
        if (model.isPresent()) {
            modelProvider.releaseModel(model.get());
        }
    }

    @Override
    public ProviderOutput documentClass(ClassDoc holder) {
        if (!model.isPresent()) {
            return null;
        }
        // filter out one Element patterns, non-relevant patterns, then sort and limit to number threshold
        final MethodPattern[] methodPatterns = model.get().getPatterns();
        final List<MethodPattern> patterns = Arrays.stream(methodPatterns)
                .filter(oneElementPatterns())
                .filter(relevantPatterns())
                .sorted(methodPatternComparator())
                .limit(getConfiguration().getNumberThreshold())
                .collect(Collectors.toList());

        if (patterns.isEmpty()) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        sb.append("<h5>Method override patterns:</h5>")
                .append("The following groups of methods are frequently overridden together:").append("<br>")
                .append("<br>").append("<ul>");

        for (int i = 0; i < patterns.size(); i++) {
            MethodPattern pattern = patterns.get(i);

            sb.append("<li>");
            sb.append("<b>");
            sb.append("Pattern #" + (i + 1));
            sb.append(", ");
            sb.append("</b>");
            sb.append(pattern.getNumberOfObservations() + " times observed");

            sb.append("<br>").append("<br>");

            List<Entry<IMethodName, Double>> methods = new ArrayList<Entry<IMethodName, Double>>(pattern.getMethods()
                    .entrySet());
            sortMethods(methods);

            for (Iterator<Entry<IMethodName, Double>> iterator = methods.iterator(); iterator.hasNext();) {
                Entry<IMethodName, Double> entry = iterator.next();
                if (entry.getValue() >= 0.7 || methods.indexOf(entry) < 3) {

                    sb.append("<code>");

                    sb.append(strong(htmlLink(entry.getKey()))).append(listParameterTypes(entry.getKey()));

                    sb.append("</code>");

                    sb.append(" - ").append("<font color=\"#0000FF\">").append(asPercentage(entry) + "%")
                            .append("</font>");

                    if (iterator.hasNext()) {
                        sb.append("<br>");
                    }
                }
            }
            if (i < methodPatterns.length - 1) {
                sb.append("<br>").append("<br>");
            }
            sb.append("</li>");
        }
        sb.append("</ul>");
        highlight(sb);

        return new ProviderOutput(sb.toString(), patterns.size());
    }

    private java.util.function.Predicate<? super MethodPattern> relevantPatterns() {
        return pattern -> pattern.getNumberOfObservations() >= getConfiguration().getTimesObservedThreshold();
    }

    private java.util.function.Predicate<? super MethodPattern> oneElementPatterns() {
        return pattern -> pattern.getMethods().keySet().size() > 1;
    }

    private Comparator<MethodPattern> methodPatternComparator() {
        return (p1, p2) -> p2.getNumberOfObservations() - p1.getNumberOfObservations();
    }

    private void sortMethods(List<Entry<IMethodName, Double>> methods) {
        Collections.sort(methods, new Comparator<Entry<IMethodName, Double>>() {

            @Override
            public int compare(Entry<IMethodName, Double> o1, Entry<IMethodName, Double> o2) {
                if (!o1.getValue().equals(o2.getValue())) {
                    return o2.getValue().compareTo(o1.getValue());
                } else {
                    return o1.getKey().getName().compareTo(o2.getKey().getName());
                }
            }

        });

    }

    private int asPercentage(Entry<IMethodName, Double> entry) {
        Double rel = entry.getValue();
        ensureIsInRange(rel, 0, 1, "relevance '%f' not in interval [0, 1]", rel);
        return (int) Math.round(rel * 100);
    }

    @Override
    public void tearDown() throws LiveDocProviderException {
        if (modelProvider != null) {
            try {
                modelProvider.close();
            } catch (IOException e) {
                throw new LiveDocProviderException(String.format(
                        "Exception while closing model provider for %s models.", getId()), e);
            }
        }
    }

    @Override
    public MethodOverridePatternsConfiguration newProviderConfiguration() {
        return new MethodOverridePatternsConfiguration();
    }

}
