/**
 * Copyright (c) 2010, 2014 Darmstadt University of Technology.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Andreas Sewe - initial API and implementation.
 */
package org.eclipse.recommenders.internal.livedoc.javadoc;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.ModelCoordinate;
import org.eclipse.recommenders.models.ProjectCoordinate;

import com.google.common.base.Optional;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSet.Builder;
import com.google.common.collect.Lists;

public class CompositeModelIndex implements IModelIndex {

    private final List<IModelIndex> componentIndexes;

    public CompositeModelIndex(Iterable<IModelIndex> componentIndexes) {
        this.componentIndexes = Lists.newArrayList(componentIndexes);
    }

    @Override
    public void open() throws IOException {
        for (IModelIndex componentIndex : componentIndexes) {
            componentIndex.open();
        }
    }

    @Override
    public void close() throws IOException {
        for (IModelIndex componentIndex : componentIndexes) {
            componentIndex.close();
        }
    }

    @Override
    public ImmutableSet<ModelCoordinate> getKnownModels(String modelType) {
        Builder<ModelCoordinate> builder = ImmutableSet.builder();

        for (IModelIndex componentIndex : componentIndexes) {
            builder.addAll(componentIndex.getKnownModels(modelType));
        }

        return builder.build();
    }

    @Override
    public Optional<ModelCoordinate> suggest(ProjectCoordinate pc, String modelType) {
        for (IModelIndex componentIndex : componentIndexes) {
            Optional<ModelCoordinate> suggestion = componentIndex.suggest(pc, modelType);
            if (suggestion.isPresent()) {
                return suggestion;
            }
        }
        return Optional.absent();
    }

    @Override
    public ImmutableSet<ModelCoordinate> suggestCandidates(ProjectCoordinate pc, String modelType) {
        Builder<ModelCoordinate> builder = ImmutableSet.builder();

        for (IModelIndex componentIndex : componentIndexes) {
            builder.addAll(componentIndex.suggestCandidates(pc, modelType));
        }

        return builder.build();
    }

    @Override
    public Optional<ProjectCoordinate> suggestProjectCoordinateByArtifactId(String artifactId) {
        for (IModelIndex componentIndex : componentIndexes) {
            Optional<ProjectCoordinate> suggestion = componentIndex.suggestProjectCoordinateByArtifactId(artifactId);
            if (suggestion.isPresent()) {
                return suggestion;
            }
        }
        return Optional.absent();
    }

    @Override
    public Optional<ProjectCoordinate> suggestProjectCoordinateByFingerprint(String fingerprint) {
        for (IModelIndex componentIndex : componentIndexes) {
            Optional<ProjectCoordinate> suggestion = componentIndex.suggestProjectCoordinateByFingerprint(fingerprint);
            if (suggestion.isPresent()) {
                return suggestion;
            }
        }
        return Optional.absent();
    }

	@Override
	public void updateIndex(File index) throws IOException {
		 for (IModelIndex componentIndex : componentIndexes) {
			 componentIndex.updateIndex(index);
		 }
	}
}
