/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.providers;

import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.utils.Nullable;

import com.sun.javadoc.AnnotationTypeDoc;
import com.sun.javadoc.AnnotationTypeElementDoc;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.ConstructorDoc;
import com.sun.javadoc.FieldDoc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.PackageDoc;
import com.sun.javadoc.RootDoc;

/**
 * Lifecycle:
 * <p>
 * <code>
 * 
 * {@link #setUp()}  {@link #documentOverview(RootDoc)}
 * <br> ( {@link #beginPackage(PackageDoc)} 
 * <br>
 * <br> {@link #documentPackage(PackageDoc)} ( 
 * <br>
 * <br>    ( {@link #beginClass(ClassDoc)} 
 * <br>            {@link #documentClass(ClassDoc)} ( {@link #documentField(FieldDoc)} | {@link #documentConstructor(ConstructorDoc)} | {@link #documentMethod(MethodDoc)} )* 
 * <br>        {@link #endClass(ClassDoc)} )
 * <br>    |
 * <br>    ( {@link #beginAnnotationType(AnnotationTypeDoc)}
 * <br>            {@link #documentAnnotationType(AnnotationTypeDoc)} ({@link #documentAnnotationTypeElement(AnnotationTypeElementDoc)} )* 
 * <br>        {@link #endAnnotationType(AnnotationTypeDoc)} )
 * <br>
 * <br> )*
 * <br><br> {@link #endPackage(PackageDoc)} )* 
 * <br>{@link #tearDown()}
 * 
 * </code>
 */
public interface ILivedocProvider<C extends ProviderConfiguration> extends Comparable<ILivedocProvider<?>> {

    /**
     * Return the id of this Livedoc Provider, must be equal to extension id and cmd line parameter
     * 
     * @return the id of this Provider, e.g. ovrm
     */
    String getId();

    /**
     * A ranking for ordering Providers on the same level (e.g. class output)
     */
    int getRanking();

    /**
     * Set the configuration arguments for this provider.
     *
     */
    void setArguments(@Nullable String[] args);

    /**
     * Get the configuration for this provider.
     * 
     * @see ProviderConfiguration
     */
    C getConfiguration();

    /**
     * Create a (new) ProviderConfiguration which serves as a Java bean for args4j
     */
    C newProviderConfiguration();

    /**
     * Do everything necessary to get ready for documentation output, e.g. download model archive for coordinate etc.
     * 
     * @throws LiveDocProviderException
     *             thrown if this Provider wasn't successfully initiated.
     */
    void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException;

    /**
     * Given a RootDoc, return a correspondending ProviderOutput representing this Provider's output for this RootDoc.
     * Return <code>null</code> if this Provider has no proper output for this RootDoc.
     * 
     * @return <code>null</code> if there's no output for this RootDoc on this Provider
     */
    ProviderOutput documentOverview(RootDoc holder);

    /**
     * This method will be called when Livedoc processes a new PackageDoc. At the end of the Package,
     * {@link #endPackage(PackageDoc)} will be called. Usage for model resolving etc.
     * 
     * @param newPackageDoc
     *            the new PackageDoc
     */
    void beginPackage(PackageDoc newPackageDoc);

    /**
     * Given a PackageDoc, return a correspondending ProviderOutput representing this Provider's output for this
     * PackageDoc. Return <code>null</code> if this Provider has no proper output for this PackageDoc.
     * <p>
     * Called directly after {@link #beginPackage(PackageDoc)}
     * 
     * @return <code>null</code> if there's no output for this PackageDoc on this Provider
     */
    ProviderOutput documentPackage(PackageDoc holder);

    /**
     * This method will be called when Livedoc processes a new ClassDoc. At the end of the ClassDoc,
     * {@link #endClass(ClassDoc)} will be called. Usage for model resolving etc.
     * 
     * @see #endClass(ClassDoc)
     * 
     * @param newClassDoc
     *            the new ClassDoc
     */
    void beginClass(ClassDoc newClassDoc);

    /**
     * Given a ClassDoc, return a correspondending ProviderOutput representing this Provider's output for this ClassDoc.
     * Return <code>null</code> if this Provider has no proper output for this ClassDoc.
     * <p>
     * If you want to generate ProviderOutput for ClassDocs, you'll maybe want to also check / overwrite
     * {@link #beginPackage(PackageDoc)} and {@link #endPackage(PackageDoc)}
     * <p>
     * Called directly after {@link #beginClass(ClassDoc)}
     * 
     * @return <code>null</code> if there's no output for this PackageDoc on this Provider
     */
    ProviderOutput documentClass(ClassDoc holder);

    /**
     * Given a FieldDoc, return a correspondending ProviderOutput representing this Provider's output for this FieldDoc.
     * Return <code>null</code> if this Provider has no proper output for this FieldDoc.
     * <p>
     * If you want to generate ProviderOutput for FieldDocs, you'll maybe want to also check / overwrite
     * {@link #beginClass(ClassDoc)} and {@link #endClass(ClassDoc)}
     * 
     * @return <code>null</code> if there's no output for this FieldDoc on this Provider
     */
    ProviderOutput documentField(FieldDoc holder);

    /**
     * Given a ConstructorDoc, return a correspondending ProviderOutput representing this Provider's output for this
     * ConstructorDoc. Return <code>null</code> if this Provider has no proper output for this ConstructorDoc.
     * <p>
     * If you want to generate ProviderOutput for ConstructorDocs, you'll maybe want to also check / overwrite
     * {@link #beginClass(ClassDoc)} and {@link #endClass(ClassDoc)}
     * 
     * @return <code>null</code> if there's no output for this ConstructorDoc on this Provider
     */
    ProviderOutput documentConstructor(ConstructorDoc holder);

    /**
     * Given a MethodDoc, return a correspondending ProviderOutput representing this Provider's output for this
     * MethodDoc. Return <code>null</code> if this Provider has no proper output for this MethodDoc.
     * <p>
     * If you want to generate ProviderOutput for MethodDocs, you'll maybe want to also check / overwrite
     * {@link #beginClass(ClassDoc)} and {@link #endClass(ClassDoc)}
     * 
     * @return <code>null</code> if there's no output for this MethodDoc on this Provider
     */
    ProviderOutput documentMethod(MethodDoc holder);

    /**
     * Indicates that the current ClassDoc is finished. Usually, a call to a <code>begin*(*)</code> method for a new Doc
     * object (or eventually {@link #tearDown()}) will follow. Usage for model releasing etc.
     * 
     * @see #beginClass(ClassDoc)
     * 
     * @param oldClassDoc
     *            the old ClassDoc
     */
    void endClass(ClassDoc oldClassDoc);

    /**
     * This method will be called when Livedoc processes a new AnnotationTypeDoc. At the end of the AnnotationType,
     * {@link #endAnnotationType(AnnotationTypeDoc)} will be called. Usage for model resolving etc.
     * 
     * @see #endAnnotationType(AnnotationTypeDoc)
     * 
     * @param newAnnotationType
     *            the new AnnotationTypeDoc
     */
    void beginAnnotationType(AnnotationTypeDoc newAnnotationType);

    /**
     * Given a AnnotationTypeDoc, return a correspondending ProviderOutput representing this Provider's output for this
     * AnnotationTypeDoc. Return <code>null</code> if this Provider has no proper output for this AnnotationTypeDoc.
     * <p>
     * If you want to generate ProviderOutput for AnnotationTypeDocs, you'll maybe want to also check / overwrite
     * {@link #beginPackage(PackageDoc)} and {@link #endPackage(PackageDoc)}
     * <p>
     * Called directly after {@link #beginAnnotationType(AnnotationTypeDoc)}
     * 
     * @see #documentAnnotationTypeElement(AnnotationTypeElementDoc)
     * 
     * @return <code>null</code> if there's no Provider for this AnnotationTypeDoc on this Provider
     */
    ProviderOutput documentAnnotationType(AnnotationTypeDoc holder);

    /**
     * Given a AnnotationTypeElementDoc, return a correspondending ProviderOutput representing this Provider's output
     * for this AnnotationTypeElementDoc. Return <code>null</code> if this Provider has no proper output for this
     * AnnotationTypeElementDoc.
     * <p>
     * If you want to generate ProviderOutput for AnnotationTypeElementDocs, you'll maybe want to also check / overwrite
     * {@link #beginAnnotationType(AnnotationTypeDoc)} and {@link #endAnnotationType(AnnotationTypeDoc)}
     * 
     * @return <code>null</code> if there's no output for this AnnotationTypeElementDoc on this Provider
     */
    ProviderOutput documentAnnotationTypeElement(AnnotationTypeElementDoc holder);

    /**
     * Indicates that the current AnnotationType is finished. Usually, a call to a <code>begin*(*)</code> method for a
     * new Doc object (or eventually {@link #tearDown()}) will follow. Usage for model releasing etc.
     * 
     * @see #beginAnnotationType(AnnotationTypeDoc)
     * 
     * @param oldAnnotationTypeDoc
     *            the old AnnotationTypeDoc
     */
    void endAnnotationType(AnnotationTypeDoc oldAnnotationTypeDoc);

    /**
     * Indicates that the current PackaDoc is finished. Usually, a call to {@link #beginPackage(PackageDoc)} for a new
     * PackageDoc (or eventually {@link #tearDown()}) will follow. Usage for model releasing etc.
     * 
     * @see #beginPackage(PackageDoc)
     * 
     * @param oldPackageDoc
     *            the old PackageDoc
     */
    void endPackage(PackageDoc oldPackageDoc);

    /**
     * Indicates that the Livedoc run for this Provider is finished (no more PackagesDocs, ClassDocs, etc.) <br>
     * Do everything necessary to tear down this Provider(e.g. close streams etc.)
     * 
     * @return <code>true</code>, if this Provider was successfully closed.
     */
    void tearDown() throws LiveDocProviderException;
}
