/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.cli.args4j;

import java.util.List;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.OptionDef;
import org.kohsuke.args4j.spi.OptionHandler;
import org.kohsuke.args4j.spi.Parameters;
import org.kohsuke.args4j.spi.Setter;

import com.google.common.collect.Lists;

public class ProviderArgumentsHandler extends OptionHandler<ProviderCmdLineArguments> {

    List<String> usedIds = Lists.newLinkedList();

    public ProviderArgumentsHandler(CmdLineParser parser, OptionDef option,
            Setter<? super ProviderCmdLineArguments> setter) {
        super(parser, option, setter);
    }

    @Override
    public int parseArguments(Parameters params) throws CmdLineException {

        int i = 0;
        String id = params.getParameter(i);

        if (usedIds.contains(id)) {
            throw new CmdLineException(owner, String.format("Found duplicate Provider in command line arguments: %s \n"
                    + "Please make sure you use each Provider only once!", id));
        }
        usedIds.add(id);

        List<String> args = Lists.newLinkedList();

        boolean enclosingbracket = false;
        if (checkForAdditionalArguments(params)) {
            for (i = 1; i < params.size(); i++) {
                args.add(params.getParameter(i).replaceAll("[\\[\\]]", ""));

                if (params.getParameter(i).endsWith("]")) {
                    enclosingbracket = true;
                    break;
                }
            }

            if (!enclosingbracket) {
                throw new CmdLineException(owner, "Did not find expected token \']\' after \'[\'");
            }
        }

        setter.addValue(new ProviderCmdLineArguments(id, args.toArray(new String[args.size()])));
        return ++i;
    }

    private boolean checkForAdditionalArguments(Parameters params) throws CmdLineException {
        // params.size() > 1, as we need at least 2 additional parameters
        // (params.getParameter(0) is the provider id itself)
        return params.size() > 1 && params.getParameter(1).startsWith("[");
    }

    @Override
    public String getDefaultMetaVariable() {
        return "PROVIDER_ID [PROVIDER_ARGUMENTS]";
    }

}
