/**
 * Copyright (c) 2010, 2013 Darmstadt University of Technology.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Andreas Sewe - initial API and implementation.
 *    Olav Lenz - Transfer to provider
 */
package org.eclipse.recommenders.livedoc.providers.calls;

import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.asIMethodName;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.code;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.extractTypeName;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.topMethods;
import static org.eclipse.recommenders.livedoc.utils.MethodDocumentationBuilder.ListType.SEQUENTIAL;
import static org.eclipse.recommenders.utils.Constants.CLASS_CALL_MODELS;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.recommenders.calls.ICallModel;
import org.eclipse.recommenders.calls.ICallModelProvider;
import org.eclipse.recommenders.calls.SingleZipCallModelProvider;
import org.eclipse.recommenders.livedoc.providers.AbstractLiveDocProvider;
import org.eclipse.recommenders.livedoc.providers.LiveDocProviderException;
import org.eclipse.recommenders.livedoc.providers.ProviderOutput;
import org.eclipse.recommenders.livedoc.utils.ItemBuilder;
import org.eclipse.recommenders.livedoc.utils.MethodDocumentationBuilder;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.models.UniqueTypeName;
import org.eclipse.recommenders.utils.Recommendation;
import org.eclipse.recommenders.utils.names.IMethodName;
import org.eclipse.recommenders.utils.names.ITypeName;

import com.google.common.base.Optional;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.MethodDoc;

public class CallAlsoProvider extends AbstractLiveDocProvider<CallAlsoConfiguration> {

    private ICallModelProvider modelProvider;
    private Optional<ICallModel> model;

    @Override
    public String getId() {
        return "call-also";
    }

    @Override
    public void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException {
        super.setUp(pc, repo, index);
        Optional<File> modelArchive = fetchModelArchive(CLASS_CALL_MODELS);

        if (!modelArchive.isPresent()) {
            throw new LiveDocProviderException(String.format("No %s model available for given coordinate.", getId()));
        }

        modelProvider = new SingleZipCallModelProvider(modelArchive.get());

        try {
            modelProvider.open();
        } catch (IOException e) {
            throw new LiveDocProviderException(String.format("Exception while opening model provider for %s models.",
                    getId()));
        }
    }

    @Override
    public void beginClass(ClassDoc newClassDoc) {
        ITypeName methodDeclaringType = extractTypeName(newClassDoc);
        UniqueTypeName uniqueTypeName = new UniqueTypeName(getProjectCoordinate(), methodDeclaringType);
        model = modelProvider.acquireModel(uniqueTypeName);
    }

    @Override
    public void endClass(ClassDoc oldClassDoc) {
        if (model.isPresent()) {
            modelProvider.releaseModel(model.get());
        }
    }

    @Override
    public ProviderOutput documentMethod(MethodDoc holder) {
        if (!holder.isStatic()) {
            return generateMethodDoc(holder);
        }
        return null;
    }

    private ProviderOutput generateMethodDoc(MethodDoc holder) {
        if (model.isPresent()) {
            // TODO: Fix for arrays!
            IMethodName currentMethod = asIMethodName(holder);
            model.get().reset();
            if (methodsInObservedState(model.get(), Collections.singleton(currentMethod))) {

                List<Recommendation<IMethodName>> recommendations = model.get().recommendCalls();
                recommendations = topMethods(recommendations, getConfiguration().getNumberThreshold(),
                        getConfiguration().getPercentageThreshold() / 100.0);

                if (!recommendations.isEmpty()) {
                    MethodDocumentationBuilder mdb = MethodDocumentationBuilder.create().title("Call Also:")
                            .description("People who called " + code(holder.name()) + " also called:")
                            .beginList(SEQUENTIAL);

                    Iterator<Recommendation<IMethodName>> it = recommendations.iterator();
                    while (it.hasNext()) {
                        Recommendation<IMethodName> recommendation = it.next();
                        mdb.addItem(ItemBuilder.create(holder).method(recommendation.getProposal())
                                .withPercentage(recommendation.getRelevance() * 100).build());

                    }

                    mdb.endList();
                    return new ProviderOutput(highlight(mdb.build()), recommendations.size());
                }
            }
        }
        return null;
    }

    private boolean methodsInObservedState(ICallModel model, Set<IMethodName> methods) {
        return model.setObservedCalls(methods);
    }

    @Override
    public void tearDown() throws LiveDocProviderException {
        if (modelProvider != null) {
            try {
                modelProvider.close();
            } catch (IOException e) {
                throw new LiveDocProviderException(String.format(
                        "Exception while closing model provider for %s models.", getId()));
            }
        }
    }

    @Override
    public CallAlsoConfiguration newProviderConfiguration() {
        return new CallAlsoConfiguration();
    }
}
