/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.providers.calls;

import static com.google.common.collect.ComparisonChain.start;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.code;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.extractTypeName;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.htmlLink;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.listParameterTypes;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.strong;
import static org.eclipse.recommenders.utils.Constants.CLASS_SELFC_MODEL;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.recommenders.apidocs.ClassSelfcallDirectives;
import org.eclipse.recommenders.livedoc.providers.AbstractLiveDocProvider;
import org.eclipse.recommenders.livedoc.providers.LiveDocProviderException;
import org.eclipse.recommenders.livedoc.providers.ProviderOutput;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelProvider;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.models.UniqueTypeName;
import org.eclipse.recommenders.utils.names.IMethodName;
import org.eclipse.recommenders.utils.names.ITypeName;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.collect.Lists;
import com.sun.javadoc.ClassDoc;

public class ClassSelfCallProvider extends AbstractLiveDocProvider<ClassSelfCallConfiguration> {

    private IModelProvider<UniqueTypeName, ClassSelfcallDirectives> modelProvider;
    private Optional<ClassSelfcallDirectives> model;

    @Override
    public String getId() {
        return "selfc";
    }

    @Override
    public void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException {
        super.setUp(pc, repo, index);

        Optional<File> modelArchive = fetchModelArchive(CLASS_SELFC_MODEL);

        if (!modelArchive.isPresent()) {
            throw new LiveDocProviderException(String.format("No %s model available for given coordinate.", getId()));
        }

        modelProvider = new SingleZipClassSelfCallsModelProvider(modelArchive.get());

        try {
            modelProvider.open();
        } catch (IOException e) {
            throw new LiveDocProviderException(String.format("Exception while opening model provider for %s models.",
                    getId()));
        }
    }

    @VisibleForTesting
    void setUp(ProjectCoordinate pc, IModelProvider<UniqueTypeName, ClassSelfcallDirectives> modelProvider)
            throws LiveDocProviderException {
        super.setUp(pc, null, null);
        this.modelProvider = modelProvider;
    }

    @Override
    public void beginClass(ClassDoc newClassDoc) {
        ITypeName methodDeclaringType = extractTypeName(newClassDoc);
        UniqueTypeName uniqueTypeName = new UniqueTypeName(getProjectCoordinate(), methodDeclaringType);
        model = modelProvider.acquireModel(uniqueTypeName);
    }

    @Override
    public ProviderOutput documentClass(ClassDoc holder) {
        if (!model.isPresent()) {
            return null;
        }

        List<Entry<IMethodName, Integer>> recommendations = preprocessEntries(model.get());
        if (recommendations.isEmpty()) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        sb.append("<h5>").append("Self Calls:").append("</h5>");
        sb.append("<p>").append("The following methods are frequently called by implementors of ")
                .append(code(holder.name())).append(":").append("</p>");

        sb.append("<ul>");
        Iterator<Entry<IMethodName, Integer>> it = recommendations.iterator();
        while (it.hasNext()) {
            Entry<IMethodName, Integer> entry = it.next();
            IMethodName method = entry.getKey();
            float percentage = calculatePercentage(model.get().getNumberOfSubclasses(), entry);

            sb.append("<li>");
            sb.append("<code>");
            sb.append(strong(htmlLink(method))).append(listParameterTypes(method));
            sb.append("</code>");

            sb.append(String.format(" (<font color=\"#0000FF\">%1$.0f%% - %2$d times</font>)", percentage,
                    entry.getValue()));

            if (it.hasNext()) {
                sb.append(", ");
            }
            sb.append("</li>");
        }
        sb.append("</ul>");

        return new ProviderOutput(highlight(sb).toString(), recommendations.size());
    }

    @Override
    public void endClass(ClassDoc oldClassDoc) {
        if (model.isPresent()) {
            modelProvider.releaseModel(model.get());
        }
    }

    @Override
    public void tearDown() throws LiveDocProviderException {
        if (modelProvider != null) {
            try {
                modelProvider.close();
            } catch (IOException e) {
                throw new LiveDocProviderException(String.format(
                        "Exception while closing model provider for %s models.", getId()));
            }
        }
    }

    private List<Entry<IMethodName, Integer>> preprocessEntries(ClassSelfcallDirectives directive) {
        List<Entry<IMethodName, Integer>> entries = sortEntries(directive.getCalls().entrySet());
        entries = filterByNumberThreshold(entries);
        entries = filterByPercentageAndTimesObservedThreshold(entries, directive.getNumberOfSubclasses());
        return entries;
    }

    private List<Entry<IMethodName, Integer>> sortEntries(Set<Entry<IMethodName, Integer>> entrySet) {
        List<Entry<IMethodName, Integer>> entries = Lists.newArrayList(entrySet);
        Collections.sort(entries, new Comparator<Entry<IMethodName, Integer>>() {

            @Override
            public int compare(Entry<IMethodName, Integer> o1, Entry<IMethodName, Integer> o2) {
                return start().compare(o2.getValue(), o1.getValue())
                        .compare(o1.getKey().getName(), o2.getKey().getName()).result();
            }
        });
        return entries;
    }

    private List<Entry<IMethodName, Integer>> filterByNumberThreshold(List<Entry<IMethodName, Integer>> entries) {
        int numberThreshold = getConfiguration().getNumberThreshold();
        if (entries.size() > numberThreshold) {
            return entries.subList(0, numberThreshold);
        }
        return entries;
    }

    private List<Entry<IMethodName, Integer>> filterByPercentageAndTimesObservedThreshold(
            List<Entry<IMethodName, Integer>> entries, int numberOfDefinitions) {
        List<Entry<IMethodName, Integer>> filterd = Lists.newArrayList();
        for (Entry<IMethodName, Integer> entry : entries) {
            Integer times = entry.getValue();
            if (times >= getConfiguration().getTimesObservedThreshold()) {
                if (calculatePercentage(numberOfDefinitions, entry) >= getConfiguration().getPercentageThreshold()) {
                    filterd.add(entry);
                }
            }
        }
        return filterd;
    }

    private int calculatePercentage(int numberOfDefinitions, Entry<IMethodName, Integer> entry) {
        return Math.round(entry.getValue() * 100.0f / numberOfDefinitions);
    }

    @Override
    public ClassSelfCallConfiguration newProviderConfiguration() {
        return new ClassSelfCallConfiguration();
    }
}
