/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.providers.calls;

import static com.google.common.collect.ComparisonChain.start;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.*;
import static org.eclipse.recommenders.utils.Constants.CLASS_SELFM_MODEL;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.recommenders.apidocs.MethodSelfcallDirectives;
import org.eclipse.recommenders.apidocs.SingleZipMethodSelfCallsModelProvider;
import org.eclipse.recommenders.livedoc.providers.AbstractLiveDocProvider;
import org.eclipse.recommenders.livedoc.providers.LiveDocProviderException;
import org.eclipse.recommenders.livedoc.providers.ProviderOutput;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelProvider;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.IUniqueName;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.models.UniqueMethodName;
import org.eclipse.recommenders.utils.names.IMethodName;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.collect.Lists;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.MethodDoc;

public class MethodSelfCallProvider extends AbstractLiveDocProvider<MethodSelfCallConfiguration> {

    private IModelProvider<IUniqueName<IMethodName>, MethodSelfcallDirectives> modelProvider;

    @Override
    public String getId() {
        return "selfm";
    }

    @Override
    public void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException {
        super.setUp(pc, repo, index);

        Optional<File> modelArchive = fetchModelArchive(CLASS_SELFM_MODEL);

        if (!modelArchive.isPresent()) {
            throw new LiveDocProviderException(String.format("No %s model available for given coordinate.", getId()));
        }

        modelProvider = new SingleZipMethodSelfCallsModelProvider(modelArchive.get());

        try {
            modelProvider.open();
        } catch (IOException e) {
            throw new LiveDocProviderException(String.format("Exception while opening model provider for %s models.",
                    getId()));
        }
    }

    @VisibleForTesting
    void setUp(ProjectCoordinate pc, IModelProvider<IUniqueName<IMethodName>, MethodSelfcallDirectives> modelProvider)
            throws LiveDocProviderException {
        super.setUp(pc, null, null);
        this.modelProvider = modelProvider;
    }

    @Override
    public ProviderOutput documentMethod(MethodDoc holder) {
        // TODO: Fix for arrays!
        IMethodName asIMethodName = asIMethodName(holder);
        UniqueMethodName uniqueMethodName = new UniqueMethodName(getProjectCoordinate(), asIMethodName);
        Optional<MethodSelfcallDirectives> directives = modelProvider.acquireModel(uniqueMethodName);

        if (!directives.isPresent()) {
            return null;
        }

        List<Entry<IMethodName, Integer>> entries = preprocessEntries(directives.get());
        if (entries.isEmpty()) {
            modelProvider.releaseModel(directives.get());
            return null;
        }

        modelProvider.releaseModel(directives.get());
        return createDoc(holder, directives.get().getNumberOfDefinitions(), entries);
    }

    private List<Entry<IMethodName, Integer>> preprocessEntries(MethodSelfcallDirectives directive) {
        List<Entry<IMethodName, Integer>> entries = sortEntries(directive.getCalls().entrySet());
        entries = filterByNumberThreshold(entries);
        entries = filterByPercentageAndTimesObservedThreshold(entries, directive.getNumberOfDefinitions());
        return entries;
    }

    private List<Entry<IMethodName, Integer>> sortEntries(Set<Entry<IMethodName, Integer>> entrySet) {
        List<Entry<IMethodName, Integer>> entries = Lists.newArrayList(entrySet);
        Collections.sort(entries, new Comparator<Entry<IMethodName, Integer>>() {

            @Override
            public int compare(Entry<IMethodName, Integer> o1, Entry<IMethodName, Integer> o2) {
                return start().compare(o2.getValue(), o1.getValue())
                        .compare(o1.getKey().getName(), o2.getKey().getName()).result();
            }
        });
        return entries;

    }

    private List<Entry<IMethodName, Integer>> filterByNumberThreshold(List<Entry<IMethodName, Integer>> entries) {
        int numberThreshold = getConfiguration().getNumberThreshold();
        if (entries.size() > numberThreshold) {
            return entries.subList(0, numberThreshold);
        }
        return entries;
    }

    private List<Entry<IMethodName, Integer>> filterByPercentageAndTimesObservedThreshold(
            List<Entry<IMethodName, Integer>> entries, int numberOfDefinitions) {
        List<Entry<IMethodName, Integer>> filterd = Lists.newArrayList();
        for (Entry<IMethodName, Integer> entry : entries) {
            Integer times = entry.getValue();
            if (times >= getConfiguration().getTimesObservedThreshold()) {
                if (calculatePercentage(numberOfDefinitions, entry) >= getConfiguration().getPercentageThreshold()) {
                    filterd.add(entry);
                }
            }
        }
        return filterd;
    }

    private int calculatePercentage(int numberOfDefinitions, Entry<IMethodName, Integer> entry) {
        return Math.round(entry.getValue() * 100.0f / numberOfDefinitions);
    }

    private ProviderOutput createDoc(MethodDoc holder, int numberOfDefinitions,
            List<Entry<IMethodName, Integer>> entries) {
        StringBuilder sb = new StringBuilder();

        IMethodName holderMethod = asIMethodName(holder);

        sb.append("<dl>").append("<dt><span class=\"strong\">Self Calls:</span></dt>").append("<dd>");
        sb.append("The following methods are frequently called by implementors of ")
                .append(code(holder.name().concat(listParameterTypes(holderMethod)))).append(", ");

        ClassDoc containingClass = holder.containingClass();

        if (containingClass.isInterface()) {
            sb.append("implementing");
        } else {
            sb.append("subclassing");
        }
        sb.append(" ").append(code(containingClass.name())).append(":");
        sb.append("<br>").append("(Based on ").append(numberOfDefinitions).append(" direct implementations)")
                .append("<br>").append("<br>");

        Iterator<Entry<IMethodName, Integer>> it = entries.iterator();
        while (it.hasNext()) {
            Entry<IMethodName, Integer> entry = it.next();
            sb.append("<code>");
            IMethodName method = entry.getKey();

            if (method.equals(holderMethod)) {
                sb.append(strong(color("#7f0055", "super").concat(".")));
            }

            sb.append(strong(htmlLink(holder.containingClass(), method))).append(listParameterTypes(method));
            sb.append("</code>");

            float percentage = calculatePercentage(numberOfDefinitions, entry);
            sb.append(String.format(" (<font color=\"#0000FF\">%1$.0f%% - %2$d times</font>)", percentage,
                    entry.getValue()));

            if (it.hasNext()) {
                sb.append(", ");
            }
        }

        sb.append("</dd>").append("</dl>");
        highlight(sb);

        return new ProviderOutput(sb.toString(), entries.size());
    }

    @Override
    public void tearDown() throws LiveDocProviderException {
        if (modelProvider != null) {
            try {
                modelProvider.close();
            } catch (IOException e) {
                throw new LiveDocProviderException(String.format(
                        "Exception while closing model provider for %s models.", getId()));
            }
        }
    }

    @Override
    public MethodSelfCallConfiguration newProviderConfiguration() {
        return new MethodSelfCallConfiguration();
    }

}
