/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.providers.overrides;

import static java.lang.String.format;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.extractTypeName;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.htmlLink;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.listParameterTypes;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.methodSignatureEquals;
import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.strong;
import static org.eclipse.recommenders.utils.Constants.CLASS_OVRM_MODEL;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.recommenders.livedoc.providers.AbstractLiveDocProvider;
import org.eclipse.recommenders.livedoc.providers.LiveDocProviderException;
import org.eclipse.recommenders.livedoc.providers.ProviderConfiguration;
import org.eclipse.recommenders.livedoc.providers.ProviderOutput;
import org.eclipse.recommenders.livedoc.utils.LiveDocUtils;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.models.UniqueTypeName;
import org.eclipse.recommenders.overrides.IOverrideModel;
import org.eclipse.recommenders.overrides.SingleZipOverrideModelProvider;
import org.eclipse.recommenders.utils.Recommendation;
import org.eclipse.recommenders.utils.Recommendations;
import org.eclipse.recommenders.utils.names.IMethodName;
import org.eclipse.recommenders.utils.names.ITypeName;

import com.google.common.base.Optional;
import com.google.common.collect.Maps;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.MethodDoc;

public class OverrideMethodsProvider extends AbstractLiveDocProvider<OverrideMethodsConfiguration> {

    private SingleZipOverrideModelProvider modelProvider;
    private Optional<IOverrideModel> model;
    private HashMap<IOverrideModel, List<Recommendation<IMethodName>>> filteredMap;

    @Override
    public String getId() {
        return "ovrm";
    }

    @Override
    public void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException {
        super.setUp(pc, repo, index);
        Optional<File> modelArchive = fetchModelArchive(CLASS_OVRM_MODEL);

        if (!modelArchive.isPresent()) {
            throw new LiveDocProviderException(format("No %s model available for given coordinate.", getId()));
        }

        modelProvider = new SingleZipOverrideModelProvider(modelArchive.get());

        try {
            modelProvider.open();
        } catch (IOException e) {
            throw new LiveDocProviderException(format("Exception while opening model provider for %s models.", getId()));
        }
        filteredMap = Maps.newHashMapWithExpectedSize(modelProvider.acquireableTypes().size());
    }

    @Override
    public void beginClass(ClassDoc newClassDoc) {
        ITypeName typeName = extractTypeName(newClassDoc);
        model = ovrmModel(typeName);
    }

    @Override
    public void endClass(ClassDoc oldClassDoc) {
        if (model.isPresent()) {
            modelProvider.releaseModel(model.get());
        }
    }

    @Override
    public ProviderOutput documentClass(ClassDoc holder) {
        if (model.isPresent() && filteredNotNull(model.get())) {
            ITypeName typeName = extractTypeName(holder);
            StringBuilder sb = new StringBuilder();

            List<Recommendation<IMethodName>> recommendOverrides = sortAndFilter(model.get());

            sb.append("<h5>Method override recommendations</h5>")
                    .append("The following methods are frequently overridden by subclasses of ").append("<code>")
                    .append(typeName.getClassName()).append("</code>").append(":").append("<br>").append("<br>")
                    .append("<ul>");

            for (Iterator<Recommendation<IMethodName>> iterator = recommendOverrides.iterator(); iterator.hasNext();) {

                Recommendation<IMethodName> recommendation = iterator.next();

                int relevance = Recommendations.asPercentage(recommendation);

                IMethodName method = recommendation.getProposal();

                sb.append("<li>");

                sb.append("<code>");
                sb.append(strong(htmlLink(method))).append(listParameterTypes(method));
                sb.append("</code>");

                sb.append(" - ").append("<font color=\"#0000FF\">").append(relevance + "%").append("</font>");

                if (iterator.hasNext()) {
                    sb.append(", ");
                }

                sb.append("</li>");
            }
            sb.append("</ul>");
            highlight(sb);
            return new ProviderOutput(sb.toString(), recommendOverrides.size());
        } else {
            return null;
        }
    }

    @Override
    public ProviderOutput documentMethod(MethodDoc holder) {
        if (!holder.isAbstract()) {
            return generateMethodDoc(holder);
        }
        return null;
    }

    private ProviderOutput generateMethodDoc(MethodDoc methodDoc) {
        if (model.isPresent() && filteredNotNull(model.get())) {
            ITypeName methodDeclaringType = extractTypeName(methodDoc);
            StringBuilder sb = new StringBuilder();

            List<Recommendation<IMethodName>> recommendations = sortAndFilter(model.get());

            int relevance = 0;

            for (Recommendation<IMethodName> recommendation : recommendations) {

                if (methodSignatureEquals(recommendation.getProposal(), methodDoc)) {
                    relevance = Recommendations.asPercentage(recommendation);

                    sb.append("<dl>").append("<dt>Method override recommendation:</dt>").append("<dd>")
                            .append("Subclasses of ").append(methodDeclaringType.getClassName()).append(" ");

                    sb.append(percentageToRecommendationPhrase(relevance));

                    sb.append(" override this method ").append("(").append("<font color=\"#0000FF\">")
                            .append(relevance + "%").append("</font>").append(")").append("</dd>").append("</dl>");

                    highlight(sb);
                    break;
                }
            }
            return new ProviderOutput(sb.toString(), recommendations.size());
        }
        return null;
    }

    private String percentageToRecommendationPhrase(final int percentage) {
        if (percentage >= 95) {
            return "always";
        } else if (percentage >= 65) {
            return "usually";
        } else if (percentage >= 25) {
            return "sometimes";
        } else if (percentage >= 10) {
            return "occasionally";
        } else {
            return "rarely";
        }
    }

    private boolean filteredNotNull(IOverrideModel model) {
        List<Recommendation<IMethodName>> filteres = sortAndFilter(model);
        return !filteres.isEmpty();
    }

    private List<Recommendation<IMethodName>> sortAndFilter(IOverrideModel model) {

        if (filteredMap.containsKey(model)) {
            return filteredMap.get(model);
        }

        List<Recommendation<IMethodName>> recommendations = model.recommendOverrides();

        List<Recommendation<IMethodName>> filteredList = LiveDocUtils.topMethods(recommendations, getConfiguration()
                .getNumberThreshold(), getConfiguration().getPercentageThreshold() / 100.0);

        // fill map IType/Model --> List<Recommendation>
        filteredMap.put(model, filteredList);
        return filteredList;
    }

    private Optional<IOverrideModel> ovrmModel(ITypeName typeName) {
        UniqueTypeName key = new UniqueTypeName(getProjectCoordinate(), typeName);
        Optional<IOverrideModel> model = modelProvider.acquireModel(key);
        return model;
    }

    @Override
    public void tearDown() throws LiveDocProviderException {
        if (modelProvider != null) {
            try {
                modelProvider.close();
            } catch (IOException e) {
                throw new LiveDocProviderException(String.format(
                        "Exception while closing model provider for %s models.", getId()), e);
            }
        }
    }

    @Override
    public OverrideMethodsConfiguration newProviderConfiguration() {
        return new OverrideMethodsConfiguration();
    }
}
