/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.internal.livedoc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.recommenders.livedoc.providers.ILivedocProvider;
import org.eclipse.recommenders.utils.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

public class PluginProviderManager implements IProviderManager {

    private static final String EXTENSION_POINT_ID = "org.eclipse.recommenders.livedoc.providers";
    private static final Logger LOG = LoggerFactory.getLogger(PluginProviderManager.class);

    private List<ILivedocProvider<?>> providers;

    @Override
    public List<ILivedocProvider<?>> loadProviders(@Nullable Map<String, String[]> providerArguments) {

        IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(
                EXTENSION_POINT_ID);

        providers = createProviders(Lists.newArrayList(elements), optionalProviderConfigurations(providerArguments));

        return providers;
    }

    private Optional<Map<String, String[]>> optionalProviderConfigurations(Map<String, String[]> providerArguments) {
        if (providerArguments == null || providerArguments.isEmpty()) {
            return Optional.absent();
        } else {
            return Optional.of(providerArguments);
        }
    }

    private List<ILivedocProvider<?>> createProviders(List<IConfigurationElement> elements,
            Optional<Map<String, String[]>> providerArguments) {

        List<ILivedocProvider<?>> providers = new ArrayList<ILivedocProvider<?>>(elements.size());

        Map<String, IConfigurationElement> elementsMap = Maps.uniqueIndex(elements,
                new Function<IConfigurationElement, String>() {

                    @Override
                    public String apply(IConfigurationElement input) {
                        return input.getAttribute("id");
                    }
                });

        // filter registered Providers with providerConfigurations or loadByDefault flag
        elements = filterIConfigurationElements(elementsMap, providerArguments);

        // configure executable Providers from extensions
        for (IConfigurationElement element : elements) {
            try {
                ILivedocProvider<?> provider = (ILivedocProvider<?>) element.createExecutableExtension("class");
                provider.getConfiguration().setName(element.getAttribute("name"));
                providers.add(provider);
            } catch (CoreException e) {
                LOG.error("Couldn't load Provider \"{}\"", element.getAttribute("id"), e);
            }
        }

        // set arguments to providers
        if (providerArguments.isPresent()) {
            Map<String, String[]> providerArgsMap = providerArguments.get();

            for (ILivedocProvider<?> provider : providers) {
                provider.setArguments(providerArgsMap.get(provider.getId()));
            }
        }

        Collections.sort(providers);
        return providers;
    }

    private List<IConfigurationElement> filterIConfigurationElements(Map<String, IConfigurationElement> elementsMap,
            Optional<Map<String, String[]>> providerArguments) {

        List<IConfigurationElement> result = new ArrayList<IConfigurationElement>(elementsMap.size());

        // Provider arguments are present?
        // Then filter with them, loadByDefault isn't necessary
        // All required arguments should be present
        if (providerArguments.isPresent()) {
            for (String id : providerArguments.get().keySet()) {

                if (elementsMap.containsKey(id)) {
                    result.add(elementsMap.get(id));
                } else {
                    LOG.error("Unknown provider \"{}\", skipping it." + "\n \t Known Providers are:" + "\n \t \t {}",
                            id, Joiner.on(", ").join(elementsMap.keySet()));
                }
            }
            return result;
        }

        // Otherwise, check if they are non-loadByDefault Providers and skip these.
        for (IConfigurationElement provider : elementsMap.values()) {
            if (loadByDefault(provider)) {
                result.add(provider);
            }
        }
        return result;
    }

    private boolean loadByDefault(IConfigurationElement provider) {
        // assume "true" as default case for providers with no loadByDefault value
        // see also org.eclipse.recommenders.livedoc.providers extension point
        String loadByDefault = provider.getAttribute("loadByDefault");
        return (loadByDefault == null || loadByDefault.equals("true"));
    }

    @Override
    public List<String> getProviderIds() {

        return Lists.transform(providers, new Function<ILivedocProvider<?>, String>() {

            @Override
            public String apply(ILivedocProvider<?> input) {
                return input.getId();
            }
        });
    }

    @Override
    public List<ILivedocProvider<?>> getProviders() {
        return providers;
    }
}
